/**			read_rc.c			**/

/** (C) Copyright 1985, Dave Taylor			**/

/** This file contains programs to allow the user to have a .msgrc file
    in their home directory containing any of the following: 

	fullname= <username string>
	maildir = <directory>
	mailbox = <file>
	editor  = <editor>
	savemail= <savefile>
	shell   = <shell>
	print   = <print command>
	weedout = <list of headers to weed out>
	prefix  = <copied message prefix string>

	alternatives = <list of addresses that forward to us>

    and/or the logical arguments:
	
	autocopy [on|off]
	copy     [on|off]	
	resolve  [on|off]
	weed     [on|off]
	noheader [on|off]
	titles   [on|off]
	editout  [on|off]
	savebyname [on|off]
	movepage [on|off]

    Lines starting with '#' are considered comments and are not checked
    any further!

    Modified 10/85 to know about "Environment" variables..
    Modified 12/85 for the 'prefix' option
    Modified  2/86 for the "movepage" flag
**/

#include <stdio.h>
#include <ctype.h>

#ifdef BSD
#undef tolower
#endif

#include "headers.h"

char *shift_lower(), *strtok(), *getenv();

#define NOTWEEDOUT	0
#define WEEDOUT		1
#define ALTERNATIVES	2

read_rc_file()
{
	/** this routine does all the actual work of reading in the
	    .rc file... **/

	FILE *file;
	char buffer[SLEN], filename[SLEN];
	char word1[SLEN], word2[SLEN];
	int  errors = 0, last = NOTWEEDOUT;

	dprint0("read_rc_file()\n");
	
	sprintf(filename,"%s/%s", home, msgrcfile);

	default_weedlist();

	alternative_addresses = NULL; 	/* none yet! */

	if ((file = fopen(filename, "r")) == NULL)
	  return;	/* we're done! */

	while (fgets(buffer, SLEN, file) != NULL) {
	  no_ret(buffer);	 	/* remove return */
	  if (buffer[0] == '#') { 	/* comment       */
	    last = NOTWEEDOUT;
	    continue;
	  }
	  if (strlen(buffer) < 2) {	/* empty line    */
	    last = NOTWEEDOUT;
	    continue;
	  }

	  breakup(buffer, word1, word2);

	  strcpy(word1, shift_lower(word1));	/* to lower case */

	  if (word2[0] == 0 && (last != WEEDOUT || last != ALTERNATIVES)) {
	    fprintf(stderr, "Malformed line from .msgrc file: %s\n", buffer);
	    errors++;
	    continue;
	  }
	
	  if (equal(word1,"maildir") || equal(word1,"folders")) {
	    expand_env(folders, word2);
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "fullname") || equal(word1,"username") ||
		   equal(word1, "name")) {
	    strcpy(full_username, word2);
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "prefix")) {
	    strcpy(prefixchars, word2);
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "shell")) {
	    expand_env(shell, word2);
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "mailbox")) {
	    expand_env(mailbox, word2);
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "editor") || equal(word1,"mailedit")) {
	    expand_env(editor, word2);
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "savemail") || equal(word1, "saveto")) {
	    expand_env(savefile, word2);
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "print") || equal(word1, "printmail")) {
	    expand_env(printout, word2);
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "autocopy")) {
	    auto_copy = equal(shift_lower(word2), "on");
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "copy") || equal(word1, "auto_cc")) {
	    auto_cc = equal(shift_lower(word2), "on");
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "resolve")) {
	    resolve_mode = equal(shift_lower(word2), "on");
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "weed")) {
	    filter = equal(shift_lower(word2), "on");
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "noheader")) {
	    noheader = equal(shift_lower(word2), "on");
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "titles")) {
	    title_messages = equal(shift_lower(word2), "on");
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "editout")) {
	    edit_outbound = equal(shift_lower(word2), "on");
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "savebyname") || equal(word1, "savename")) {
	    save_by_name = equal(shift_lower(word2), "on");
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "movepage") || equal(word1, "page") ||
		   equal(word1, "movewhenpaged")) {
	    move_when_paged = equal(shift_lower(word2), "on");
	    last = NOTWEEDOUT;
	  }
	  else if (equal(word1, "weedout")) {
	    weedout(word2);
	    last = WEEDOUT;
	  }
	  else if (equal(word1, "alternatives")) {
	    alternatives(word2);
	    last = ALTERNATIVES;
	  }
	  else if (last == WEEDOUT)	/* could be multiple line weedout */
	    weedout(buffer);
	  else if (last == ALTERNATIVES)	/* multi-line addresses   */
	    alternatives(buffer);
	  else {
	    fprintf(stderr, "Unknown line from .rc file: %s\n", buffer);
	    errors++;
	  }
	}

	if (errors) {
	  fprintf(stderr,"Msg quit from errors in .msgrc file\n");
	  exit(errors);
	}
}
	
weedout(string)
char *string;
{
	/** This routine is called with a list of headers to weed out.   **/

	register int i;
	char *strptr, *header;

	dprint1("weedout(string='%s')\n", string);

	strptr = (char *) string;

	while ((header = strtok(strptr, "\t ,\"'")) != NULL) {
	  if (weedcount == MAX_IN_WEEDLIST)
	    exit(fprintf(stderr, "Read in too many weed headers.  Max is %d\n", 
		   MAX_IN_WEEDLIST-1));
	  if (strlen(header) > 0)
	    if (strlen(header) > NLEN)
	      fprintf(stderr, "Bad weed header: %s.  Too Long - Len is < %d!\n",
	             header, NLEN);
	    else {
	      for (i=0; i < strlen(header); i++)
	        if (header[i] == '_') header[i] = ' ';
	      strncpy(weedlist[weedcount++], header, NLEN);
	    }
	  strptr = (char *) NULL;
	}
}

alternatives(string)
char *string;
{
	/** This routine is called with a list of alternative addresses
	    that you may receive mail from (forwarded) **/

	char *strptr, *address;
	struct addr_rec *current_record, *previous_record;

	dprint1("alternatives(string='%s')\n", string);

	previous_record = NULL;

	strptr = (char *) string;

	while ((address = strtok(strptr, "\t ,\"'")) != NULL) {
	  if (previous_record == NULL) {
	    previous_record = (struct addr_rec *) 
			       malloc(sizeof *alternative_addresses);

	    strcpy(previous_record->address, address);
	    previous_record->next = NULL;
	    alternative_addresses = previous_record;
	  }
	  else {
	    current_record = (struct addr_rec *) 
			       malloc(sizeof *alternative_addresses);
	  
	    strcpy(current_record->address, address);
	    current_record->next = NULL;
	    previous_record->next = current_record;
	    previous_record = current_record;
	  }
	  strptr = (char *) NULL;
	}
}

default_weedlist()
{
	/** install the default headers to weed out! **/

	dprint0("default_weedlist()\n");

	weedcount = 0;

	strncpy(weedlist[weedcount++], ">From", NLEN);
	strncpy(weedlist[weedcount++], "In-Reply-To", NLEN);
	strncpy(weedlist[weedcount++], "References", NLEN);
	strncpy(weedlist[weedcount++], "Newsgroups", NLEN);
	strncpy(weedlist[weedcount++], "Received", NLEN);
	strncpy(weedlist[weedcount++], "Apparently-To:", NLEN);
	strncpy(weedlist[weedcount++], "Message-Id:", NLEN);
}

int
matches_weedlist(buffer)
char *buffer;
{
	/** returns true iff the first 'n' characters of 'buffer' 
	    match an entry of the weedlist **/
	
	register int i;

	dprint1("matches_weedlist(buffer='%s')\n", buffer);

	for (i=0;i < weedcount; i++)
	  if (strncmp(buffer, weedlist[i], strlen(weedlist[i])) == 0) 
	    return(1);

	return(0);
}

breakup(buffer, word1, word2)
char *buffer, *word1, *word2;
{
	/** This routine breaks buffer down into word1, word2 where 
	    word1 is alpha characters only, and there is an equal
	    sign delimiting the two...
		alpha = beta
	    For lines with more than one 'rhs', word2 is set to the
	    entire string...
	**/

	register int i;
	
	for (i=0;i<strlen(buffer) && isalpha(buffer[i]); i++)
	  word1[i] = buffer[i];

	word1[i++] = '\0';	/* that's the first word! */

	/** spaces before equal sign? **/

	while (buffer[i] == ' ' || buffer[i] == '\t') i++;

	if (buffer[i] == '=') i++;

	/** spaces after equal sign? **/

	while (buffer[i] == ' ' || buffer[i] == '\t') i++;

	if (i < strlen(buffer))
	  strcpy(word2, (char *) (buffer + i));
	else
	  word2[0] = '\0';
}

expand_env(dest, buffer)
char *dest, *buffer;
{
	/** expand possible metacharacters in buffer and then copy
	    to dest... 
	    This routine knows about "~" being the home directory,
	    and "$xxx" being an environment variable.
	**/

	char  *word, *string, next_word[SLEN];
	
	if (buffer[0] == '/') {
	  dest[0] = '/';
	  dest[1] = '\0';
	}
	else
	  dest[0] = '\0';
	string = (char *) buffer;

	while ((word = strtok(string, "/")) != NULL) {
	  if (word[0] == '$') {
	    strcpy(next_word, getenv((char *) (word + 1)));
	    if (strlen(next_word) == 0)
	      leave(printf("\n\rCan't expand environment variable '%s'\n\r",
		    word));
	  }
	  else if (word[0] == '~' && word[1] == '\0')
	    strcpy(next_word, home);
	  else
	    strcpy(next_word, word);

	  sprintf(dest, "%s%s%s", dest, (strlen(dest) > 0? "/":""),
		          next_word);

	  string = (char *) NULL;
	}
}
