/**		date.c		**/

/** return the current date and time in a readable format! **/
/** also returns an ARPA RFC-822 format date...            **/

/** (C) Copyright 1985, Dave Taylor **/

#include "headers.h"
#ifdef BSD
#  include <sys/time.h>
#else
#  include <time.h>
#endif

#include <ctype.h>

#ifdef BSD
#undef toupper
#endif

#define MONTHS_IN_YEAR	11	/* 0-11 equals 12 months! */
#define FEB		 1	/* 0 = January 		  */
#define DAYS_IN_LEAP_FEB 29	/* leap year only 	  */

#define ampm(n)		(n > 12? n - 12 : n)
#define am_or_pm(n)	(n > 11? (n > 23? "am" : "pm") : "am")
#define leapyear(year)	((year % 4 == 0) && (year % 100 != 0))

char *dayname[] = { "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday",
		  "Friday", "Saturday", "" };

char *monname[] = { "January", "February", "March", "April", "May", "June",
		  "July", "August", "September", "October", "November",
		  "December", ""};

char *arpa_dayname[] = { "Sun", "Mon", "Tue", "Wed", "Thu",
		  "Fri", "Sat", "" };

char *arpa_monname[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
		  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec", ""};

int  days_in_month[] = { 31,    28,    31,    30,    31,     30, 
		  31,     31,    30,   31,    30,     31,  -1};

#ifdef BSD
  char *tzname();
#else
  extern char *tzname[];
#endif

char *get_date()
{
	/** return the date in the format exemplified by;
		Thursday, April 18th 1985 at 8:35 pm
	**/

	static char buffer[SLEN];	/* static character buffer       */
	struct tm *the_time,		/* Time structure, see CTIME(3C) */
	          *localtime();
	char      *suffix();	 	/* digit suffix for date	 */
	long	  junk;			/* time in seconds....		 */

	dprint0("get_date()\n");

	junk = time(0);	/* this must be here for it to work! */
	the_time = localtime(&junk);
		
	sprintf(buffer, "%s, %s %d%s %d at %d:%02d %s",
		 dayname[the_time->tm_wday],	/* weekday */
		 monname[the_time->tm_mon],	/* month   */
		 the_time->tm_mday,		/* day     */
		 suffix(the_time->tm_mday),	/* suffix  */
		 the_time->tm_year + 1900,	/* year    */
		 ampm(the_time->tm_hour),	/* hour    */
		 the_time->tm_min,		/* minute  */
                 ((the_time->tm_hour == 12 || the_time->tm_hour == 24)
		  && the_time->tm_min == 0) ? (the_time->tm_hour == 12? "noon" :
		  "midnight") : am_or_pm(the_time->tm_hour));	/* am | pm */

	return( (char *) buffer);
}

char *suffix(day)
int day;
{
	/** this routine returns the suffix appropriate for the
	    specified number to make it an ordinal number.  ie,
	    if given '1' it would return 'st', and '2' => 'nd'
	**/

	static char buffer[10];
	register int digit;

	digit = day % 10;

	if (digit == 0 || digit > 3)
	  strcpy(buffer,"th");
	else if (digit == 1)
	  strcpy(buffer,"st");
	else if (digit == 2)
	  strcpy(buffer, "nd");
	else
	  strcpy(buffer, "rd");

	return( (char *) buffer);
}

char *get_arpa_date()
{
	/** returns an ARPA standard date.  The format for the date
	    according to DARPA document RFC-822 is exemplified by;

	       	      Mon, 12 Aug 85 6:29:08 MST

	**/

	static char buffer[SLEN];	/* static character buffer       */
	struct tm *the_time,		/* Time structure, see CTIME(3C) */
		  *localtime();
	long	   junk;		/* time in seconds....		 */

	dprint0("get_arpa_date()\n");

	junk = time(0);	/* this must be here for it to work! */
	the_time = localtime(&junk);

	sprintf(buffer, "%s, %d %s %d %d:%02d:%02d %s",
	  arpa_dayname[the_time->tm_wday],
	  the_time->tm_mday % 32,
	  arpa_monname[the_time->tm_mon],
	  the_time->tm_year % 100,
	  the_time->tm_hour % 24,
	  the_time->tm_min  % 61,
	  the_time->tm_sec  % 61,
#ifdef BSD
	  tzname());
#else
	  tzname[the_time->tm_isdst]);
#endif
	
	return( (char *) buffer);
}

char *full_month(month)
char *month;
{
	/** Given a three letter month abbreviation, return the 
	    full name of the month.   If can't figure it out, just
	    return the given argument. **/

	char   name[4];
	register int i;

	/** ensure name in correct case... **/

	strncpy(name, shift_lower(month), 3);
	name[0] = toupper(name[0]);

	/** now simply step through arpa_monname table to find a match **/

	for (i=0; i < 12; i++)
	  if (strncmp(name, arpa_monname[i], 3) == 0)
	    return((char *) monname[i]);
	
	return( (char *) month);
}

days_ahead(days, buffer)
int days;
char *buffer;
{
	/** return in buffer the date (Day, Mon Day, Year) of the date
	    'days' days after today. **/

	struct tm *the_time,		/* Time structure, see CTIME(3C) */
		  *localtime();
	long	   junk;		/* time in seconds....		 */

	dprint1("days_ahead(days=%d)\n", days);

	junk = time(0);	/* this must be here for it to work! */
	the_time = localtime(&junk);

	/* increment the day of the week */

	the_time->tm_wday = (the_time->tm_wday + days) % 7;

	/* the day of the month... */
	the_time->tm_mday += days;
	
	if (the_time->tm_mday > days_in_month[the_time->tm_mon]) {
	  if (the_time->tm_mon == FEB && leapyear(the_time->tm_year)) {
	    if (the_time->tm_mday > DAYS_IN_LEAP_FEB) {
	      the_time->tm_mday -= days_in_month[the_time->tm_mon];
	      the_time->tm_mon += 1;
	    }
	  }
	  else {
	    the_time->tm_mday -= days_in_month[the_time->tm_mon];
	    the_time->tm_mon += 1;
	  }
	}

	/* check the month of the year */
	if (the_time->tm_mon > MONTHS_IN_YEAR) {
	  the_time->tm_mon -= MONTHS_IN_YEAR;
	  the_time->tm_year += 1;
	}

	/* now, finally, build the actual date string */

	sprintf(buffer, "%s, %d %s %d",
	  arpa_dayname[the_time->tm_wday],
	  the_time->tm_mday % 32,
	  arpa_monname[the_time->tm_mon],
	  the_time->tm_year % 100);
}

int
valid_date(day, mon, year)
char *day, *mon, *year;
{
	/** validate the given date - returns TRUE iff the date
	    handed is reasonable and valid.  **/

	register int daynum, yearnum;

	dprint3("valid_date(day='%s', month='%s', year='%s')\n", day, mon, year);

	daynum = atoi(day);
	yearnum = atoi(year);
	
	if (daynum < 1 || daynum > 31)
	  return(0);
	
	if (yearnum < 1 || (yearnum > 100 && yearnum < 1900) ||
	    yearnum > 2000)
	  return(0);
	
	return(1);
}

fix_date(entry)
struct header_rec *entry;
{
	/** This routine will 'fix' the date entry for the specified
	    message.  This consists of 1) adjusting the year to 0-99
	    and 2) altering time from HH:MM:SS to HH:MM am|pm **/ 

	dprint3("fix_date(month='%s', day='%s', year='%s')\n",
		 entry->month, entry->day, entry->year);

	if (atoi(entry->year) > 99) 	
	  sprintf(entry->year,"%d", atoi(entry->year) - 1900);

	fix_time(entry->time);
}

fix_time(timestring)
char *timestring;
{
	/** Timestring in format HH:MM:SS (24 hour time).  This routine
	    will fix it to display as: HH:MM [am|pm] **/

	int hour, minute;

	dprint1("fix_time(string='%s')\n", timestring);

	sscanf(timestring, "%d:%d", &hour, &minute);

	if (hour < 1 || hour == 24) 
	  sprintf(timestring, "12:%2d (midnight)", minute);
	else if (hour < 12)
	  sprintf(timestring, "%d:%2.2d am", hour, minute);
	else if (hour == 12)
	  sprintf(timestring, "%d:%2.2d (noon)", hour, minute);
	else if (hour < 24)
	  sprintf(timestring, "%d:%2.2d pm", hour-12, minute);
}

#ifdef BSD

char *tzname()
{
	/** Return the name of the timezone (three letters) by plowing about
	    in the various time structures on the Berkeley systems.  This is
	    a pretty grungy routine, actually! **/

	struct timeval  *current_time;
	struct timezone *time_zone;	
	char   *timezone();

	gettimeofday(current_time, time_zone);

	return (timezone(time_zone->tz_minuteswest, time_zone->tz_dsttime));
}

#endif
