/*   maynot -- mail notification program */
/*
    mn -- see man page. Written by Wim Lewis c. May 1990. 
    Reads a (berkeley-style) mailbox and produces various forms
of output, mainly for human consumption.

*/
/*
    options:

    -f   Formatting option. Formats available:
    -n   Notify option. Prints "You have mail", "You have new mail",
          "You have unread mail" or "You have mail" instead of other output.
    -N   -n to stderr, after normal output.
    -t,T Verbose verion of -n. Print number of messages in each category.
    -h   Suppress header & trailer lines.
    -w   Decrement line width.
    -ix  Include messages with status letters `x'.
    -ex  Exclude messages with status letters `x'.

     this comment is probably outdated    
*/
/*
    This program may be freely distributed, modified, hacked, warped, 
ignored, etc., as long as the author's name and this notice are kept
intact in the source, and as long as recipients are free to do likewise
*/

/* path for last-ditch attempt to find mailbox  */
/* customize if necessary */
#define MAILBOXPATHFORM "/usr/spool/mail/%s" 

#include<stdio.h>
#include<ctype.h>
#include<sys/ioctl.h>  /* for get-line-length call */
extern char *getenv(), *strcpy();

#define SUBOFF  8 /* length of string `Subject:' */
#define FROMOFF 5 /* length of string `From '    */
#define STATOFF 8 /* length of string `Status: ' */

/* output styles */
#define OFROM 0      /* old "from"(1)-style output */
#define NFROM 1      /* "nfrom"-style output */
#define SIMPLE 2     /* simple output, originally a debug mode */
#define ADDRONLY 3   /* address-only format */
#define QUIET 4      /* no output, for -t, -n switches */
#define NFROM_S 5    /* like NFROM but without continuation */

#define LINLEN 250  /* input line buffer size */

#define MAXNCOND 10 /* max # conditional phrases */
#define MAXCONDL 5  /* max conditional length */

#define DEFAULT_OWIDTH 80  /* assumed output screen width */

extern char *index(), *rindex(), *malloc();   /* minimal header files? */
char *dynamicize();

main(argc, argv)  int argc; char **argv;
{int state;         /* 0 = waiting for 'from', 1=in header, 2=in body */
 char buf[LINLEN];  /* holds lines read from mailbox */
 char *from, *sub;  /* holds addr, subj of current article */
 FILE *np = stderr; /* stream to place tally on */
 FILE *inp = NULL;  /* stream to input from */
 int noheader = 0;  /* whether to print decorative headers if appropriate*/
 int style = NFROM; /* output style */
 int notify = 0;    /* whether to notify read/unread/new mail */
 int stafl = 0;     /* flags "quiet"(Status:less) trans. from header to body */
 int unread=0, newmail=0;  /* counters for notify mode */
 int nummsgs=0;     /* number of messages */
 char cond[MAXNCOND][MAXCONDL];  /* conditional expressions */
 int numconds = 0;  /* number of conditional phrases */
 int owidth;        /* width of output device */
 int remains = 0;   /* remaining length on an NFROM summary line */
 int i;             /* miscellaneous index variable */


 /* first get the tty line length */
 {struct winsize ttyb; int fd;
  fd = open("/dev/tty", 0);
  if(fd < 1) owidth = DEFAULT_OWIDTH;
  else {
  ioctl(fd, TIOCGWINSZ, &ttyb);
  owidth = ttyb.ws_col;
  }
  if(owidth<20) owidth = DEFAULT_OWIDTH;  /* sanity check */
 }

 for(i=1; i<argc; i++)  /* process all arguments */
 {
  if(argv[i][0] != '-' || strlen(argv[i]) == 1)    /* filenames...*/
  {
   if(inp)
   {
    fprintf(stderr, "Extra filename arguments rejected: %s\n", argv[i]);
    continue;
   }
   inp = (argv[i][0] == '-' && argv[i][1] == 0) ? stdin : fopen(argv[i], "r");
   if(!inp)
   {
    fprintf(stderr, "Filename argument rejected -- %s: ", argv[i]);
    perror("fopen");
    inp = stdin;
   }
   continue;
  }
  switch(argv[i][1])
  {
   case 'h': noheader = 1; break;
   case 'f':
    switch(argv[i][2])   /* select output mode */
    {
     case 'n': style=NFROM;	break;
     case 'N': style=NFROM_S;	break;
     case 's': style=SIMPLE;	break;
     case 'a': style=ADDRONLY;	break;
     case 'x': style=QUIET;	break;
     case 'f':
     case 0:   style=OFROM;	break;
     default:  fprintf(stderr, "Format '%s' unknown.\n", argv[i]+2); break;
    }
    break;
   case 'n':
    style=QUIET;   /* falls through */
   case 'N':
    notify = 1;
    break;
   case 't':
    style=QUIET;    /* falls through */
   case 'T':
    notify = 2;
    break;
   case 'w':
    owidth--;      /* decrement output width */
    break;
   case 'e':
   case 'i':
    strncpy(cond[numconds++], argv[i]+1, MAXCONDL);
    cond[numconds-1][0] = (cond[numconds-1][0] == 'e')? 0 : 1;
    break;
   default:
    printf("Unrecognized option -%s\n", argv[i]);
    break;
  }
 }

 if(!inp   /* no filenames were specified on the command line */
  && !(getenv("MAIL") && (inp = fopen(getenv("MAIL"), "r"))) /* env var */
 )
 {char *cp;   /* try to open /usr/spool/mail/user */
  cp = getenv("USER");
  if(!cp) cp = getenv("LOGNAME");
  if(cp)
  {char mbuf[100];
   sprintf(mbuf, MAILBOXPATHFORM, cp);
   inp = fopen(mbuf, "r");
  }
  if(!inp)
  {
   fprintf(stderr, "No files specified on the command line, aborting...\n");
   exit(-1);
  }
 }

 state = 0;           /* initialize */
 from = sub = NULL;

 
 while(!feof(inp))
 {
   fgets(buf, LINLEN, inp);    /* get a line from the mailbox */
   if(*buf == '\n' && !buf[1]) /* blank lines are special */
   { 
    if(state==1) 
    {
     state=2;
     if(stafl != -1) stafl = 1;   /* flag end of header */
     else 
     {
      stafl = 0;
      continue; 
     }
    }
    else state=0;
   }
#ifdef DEBUG
      /*ACME universal debugging statement*/
   printf("%d %d %d %s", state,stafl,remains, buf); 
#endif
   if(!strncmp(buf, "From ", FROMOFF) && state==0) /* grab origin addr */
   { 
    state=1;               /* beginning of message header */
    if(from) free(from);
    if(style == OFROM) { from=dynamicize(buf); continue; }  /* special case */
    for(i = FROMOFF; buf[i] && !isspace(buf[i]); i++)
       ; /* search for first blank */
    buf[i] = (char)0;  /* terminate the string there */
    from = dynamicize(buf+FROMOFF);
   }
   if(state==1 && !(style==OFROM || style==ADDRONLY) &&
       !strncmp("Subject:", buf, SUBOFF)) /* grab the subj string */
   {
    if(sub) free(sub);
    for(i=strlen(buf) - 1; i > SUBOFF && isspace(buf[i]); i--)
       ; /* search for last nonblank */
    buf[i+1] = (char)0;  /* trim trailing blanks */
    for(i=SUBOFF; buf[i] && isspace(buf[i]); i++)
       ;
    sub = dynamicize(buf+i);    /* trim leading blanks too */
   }
   if((state==1 && !strncmp("Status: ", buf, STATOFF)) || stafl==1) /*end hdr*/
   {int flag; char tych;  /* tych == type character */
    if(stafl == 1)
     buf[STATOFF] = (char)0;
    else
    {
     for(i=strlen(buf)-1; i>STATOFF && isspace(buf[i]); i--)
       ; /* search for last nonblank */
     buf[i+1] = (char)0; /* trim blanks */
    }
    stafl --;

    flag = 1;
    for(i=0; i<numconds; i++)  /* evaluate the conditional */
    {int j;
     for(j=1; cond[i][j]; j++)
      if(!index(buf+STATOFF, cond[i][j])) goto nextc; /* kludge for break^2 */
     flag = cond[i][0];
     nextc:  ;
    }
    if(!flag) continue;   

    nummsgs++; tych = 'N';
    if(!index(buf+STATOFF, 'O'))  /* Old mail? */
     newmail++; else tych = 'U';
    if(!index(buf+STATOFF, 'R'))  /* read mail? */
     unread++;  else tych = (tych == 'U')?'O':'?';

    switch(style)   /* output this msg. in correct style */
    {
     case SIMPLE:
      printf("<%s> %s %s\n", buf+STATOFF, from, sub);
      break;
     case OFROM:
      printf("%s", from);
      break;
     case ADDRONLY:
      puts(from);
      break;
     case QUIET:
      break;
     case NFROM:
     case NFROM_S:
      /* Format of NFROM output:
          Columns 0 - owidth/3: Address
          4 columns: ':', ' ', status, ' '
          The rest: subject */
      /* The hard part: figuring out which part of a too-long address
          the user wants to see. Current algorithm: justify so that
          whatever is immediately before the last '@' sign is in view.
         This fails for addresses of the form  user%host%host%host@host. */
      if(nummsgs == 1 && !noheader)  /* print the header line before 1st msg */
      {
       for(i=0; i< (owidth/2) -4; i++) putchar('-');
       printf(" M A I L ");
       for(i=0; i< ((1+owidth)/2) -5; i++) putchar('-');
       putchar('\n');
      }
      if(remains) { putchar('\n'); remains = 0; }  /* null message body? */
     {char *cp;
      cp = rindex(from, '@');    /* find end of area to be shown */
      if(cp == NULL) cp = from+(strlen(from) -1);
      cp  -= (owidth/3);  /* find beginning of area to be shown */
      if(cp<from) cp=from;  /* don't overshoot */
      printf("%*.*s: %c %.*s", owidth/3, owidth/3, cp,
	     tych, owidth - ( (owidth/3) +5), sub);
     }
      if(style == NFROM_S || (strlen(sub) >= (owidth-((owidth/3)+5))))
      {
       remains = 0;     /* don't fill the rest of the line */
       putchar('\n');   /* and terminate it */
      }
      else 
      {
       remains = (owidth-((owidth/3)+5)) - strlen(sub);  /* calc remainder */
       putchar('/');    /* mark end of subject line */
      }
      break;
     default:
      puts("Unimplemented style"); break;  /* this should never happen ;-) */
    }  /* end of style-output switch() */
   }  /* end of if(Status:) test */
   if(remains > 0 && (state == 0 || state == 2)) /*need to fill rest of line?*/
   {char *cp; int spflag = 0;    /* stick this line onto end of line */
				/* (NFROM mode) */
    				/* spflag is used to compress blank space */
    for(cp = buf; *cp; cp++)
    {
	/* put char., compressing out blanks */
     if(!spflag && isspace(*cp)) { spflag = 1; putchar(' '); remains--; }
     if(spflag && !isspace(*cp)) spflag = 0;
     if(!spflag) { putchar(*cp); remains--; } 
     if(!remains)		/* and check for termination */
     {
      putchar('\n');
      break;
     }
    }  /* end of fill-lines loop */
   }  /* end of fill-line test */
 }  /* end of read-mailbox loop, now clean up */
 if(remains)  /* if we're still filling up a line */
  putchar('\n');  /* end that line */
 if((style == NFROM || style == NFROM_S) && nummsgs && !noheader)
  {for(i=0; i<owidth; i++) putchar('-');  /* draw second "header" line */
   putchar('\n'); }			  /* and terminate it */
 if(notify && nummsgs)     /* next we print out message tally */
 {                    /* the complex booleans are for English's grammar */
  if(notify == 2 && unread >= newmail) unread -= newmail;
  fputs("You have",np);
  if(notify == 2 && newmail) pnum(np, newmail);
  if(newmail) fputs(" new",np);
  if(newmail && notify == 2) fprintf(np, " message%s", newmail!=1? "s":"");
  if(newmail && unread && notify == 2) fputs(" and",np);
  if(unread && notify==2) pnum(np, unread);
  if(unread && ((notify==1 && newmail==0) || notify==2)) fputs(" unread",np);
  if(notify == 1 || !(newmail+unread)) fputs(" mail",np);
  if(notify == 2 && unread) fprintf(np, " message%s", unread!=1? "s":"");
  fputs(".\n",np);   /* Whew! */
 }
}

static char *nums[] = 
{ "no", "one", "two", "three", "four", "five", "six", "seven", "eight",
"nine", "ten" };

pnum(s, n) FILE *s; int n;  /* print num to stream preceded by space */
{
  fputc(' ',s);
  if(n < 0) fprintf(s,"%d",n);
  else if(n < 11) fputs(nums[n], s);
  else fprintf(s,"%d",n);
}

/* dynamicize() takes a string and moves it to dynamic memory, performing
   error checking (crashes out on error) */
char *dynamicize(str) char *str;
{char *cp;
  cp = malloc((unsigned)(strlen(str)+1));
  if(!cp) 
  {
    fputs("Out of memory", stderr);
    exit(-2);
  }
  strcpy(cp, str);
  return(cp);
}
