 /*
  * elmedit.c 
  * Mark Winsor & Marc Siegel
  * This is intended as a VERY simple ASCII editor for the ELM mailer.
  * It allows naive users to use ELM without learning how to use an editor.
  * It only handles 23 lines * 80 columns. There are no editing capabilities
  * other than cursor movement.
  * 
  * WARNING: If you invoke editor on a file with more than 23 lines, the 
  *          file will be truncated when you write it out. 
  *          In other words, one screenfull of text only!
  *     
  * You are free to use this code, provided that you do not
  * sell or otherwise profit from it. These comments must remain intact!
  *
  * Please send comments to:
  * Marc Siegel  {uunet}!wb3ffv!mas!smarc || smarc@mas.wb3ffv.AMPR.ORG
  * or
  * Mark Winsor  {uunet}!wb3ffv!mas!wmark
  */

#include <curses.h>
#include <signal.h>

#define BUFSIZE  1840
#define LINESIZE 80
#define CTRLJ    '\012'
#define CTRLL    '\014'
#define ESC      '\033'

int lastindex;

main(argc,argv)
int argc;
char *argv[];
{
    if (argc != 2) {
        (void) usage(argv[0]);
        (void) exit(1);
    }
    (void) startedit(argv[0],argv[1]);
    (void) bye();
}

startedit(progname,filename)
char *progname;
char *filename;
{
    FILE *output;
    char *editbuf;
    int i;

    if ((editbuf = (char *)malloc(BUFSIZE)) == NULL) {
        (void) fprintf(stderr,"\n%s: Cannot allocate %d bytes\n",
                 progname,BUFSIZE);
        (void) exit(3);
    }
    for (i = 0; i < BUFSIZE; i++) {
        *(editbuf + i) = ' ';
    }

    lastindex = 0;
    if ((output = fopen(filename,"r")) != NULL) {
        (void) loadbuf(output,editbuf);
        (void) fclose(output);
    }

    (void) edit(progname,output,editbuf);
    if (lastindex > 0) {
		if ((output = fopen(filename,"w")) == NULL) {
            (void) move(23,0);
			(void) clrtoeol();
            (void) mvprintw(23,0,"%s: Cannot open \"%s\"",progname,filename);
			(void) refresh();
            (void) bye();
        }
        (void) writebuf(output,editbuf);
    }
    (void) fclose(output);
    return(0);
}

edit(progname,output,editbuf)
char *progname;
FILE *output;
char *editbuf;
{
    int row;
    int col;
    int ch;
    int index;
    int bye();

    (void) initscr();
    (void) signal(SIGINT,bye);
    (void) signal(SIGQUIT,bye);
    (void) signal(SIGTERM,bye);
    (void) nonl();
    (void) noecho();
    (void) keypad(stdscr,TRUE);

    index = row = col = 0;
    (void) dispbuf(editbuf);

    for (;;) {
        row = index / LINESIZE;
        col = index % LINESIZE;
        (void) mvprintw(23,10,
                "ESC to end, DEL to abort      Row: %2d   Col: %2d",
				 row + 1,col + 1);
        (void) move(row,col);
        (void) refresh();
        ch = getch();
        switch(ch) {
            case(KEY_DOWN):
            case(CTRLJ):
                            index += LINESIZE;
                            if (index > (BUFSIZE - 1)) {
                                index -= BUFSIZE;
                            }
                            break;

            case(KEY_UP):
                            index -= LINESIZE;
                            if (index < 0) {
                                index += BUFSIZE;
                            }
                            break;

            case(KEY_RIGHT):
                            ++index;
                            if (index > (BUFSIZE - 1)) {
                                index = 0;
                            }
                            break;

            case(KEY_LEFT):
            case('\b'):
                            --index;
                            if (index < 0) {
                                index = BUFSIZE - 1;
                            }
                            break;

            case('\r'):
                            index = index + (80 - col);
                            if (index > (BUFSIZE - 1)) {
                                index = 0;
                            }
                            break;

			case(CTRLL):  
							(void) clearok(stdscr);
                            break;

            case(ESC):
                            return(0);

            default:
                            if (ch < ' ' || ch > '~') {
                                (void) beep();
                            }
                            else {
                                editbuf[index] = ch;
                                (void) mvprintw(row,col,"%c",ch);
                                (void) refresh();
                                if (index > lastindex) {
                                    lastindex = index;
                                }    
                                ++index;
                                if (index >= BUFSIZE) {
                                    index = 0;
                                }
                            }
        }
    }
}

writebuf(output,editbuf)
FILE *output;
char *editbuf;
{
    int index;
    int count;
    char linebuf[LINESIZE + 1];

    index = 0;
    for (;;) {
        (void) strncpy(linebuf,(editbuf+index),LINESIZE);
        linebuf[LINESIZE] = '\0';
        for (count = (LINESIZE-1); linebuf[count] == ' ' && count > 0; count--)
            ;
        linebuf[(count + 1)] = '\0';
        fprintf(output,"%s\n",linebuf);
        index += LINESIZE;
        if (index > lastindex) {
            return(0);
        }
    }
}

loadbuf(output,editbuf)
FILE *output;
char *editbuf;
{
    char line[LINESIZE + 2];
    int index;
    int len;
    int row;
    int col;
    int i;

    index = 0;
    while ((fgets(line,LINESIZE+2,output)) != NULL) {
        len = strlen(line) - 1;
        line[len] = '\0';
        for (i = 0; i < len; i++) {
            *(editbuf + index + i) = line[i];
        }
        index += LINESIZE;
        if (index > (BUFSIZE - 1)) {
            index -= LINESIZE;
            break;
        }
        lastindex = index + i;
    }
}

dispbuf(editbuf)
char *editbuf;
{
    char line[LINESIZE + 1];
    int index;
    int row;

    index = 0;
    while (index < BUFSIZE) {
        (void) memcpy(line,(editbuf + index),LINESIZE);
        line[LINESIZE] = '\0';
        row = index / LINESIZE;
        (void) mvprintw(row,0,"%s",line);
        index += LINESIZE;
    }
    (void) refresh();
}

usage(progname)
char *progname;
{
    (void) fprintf(stderr,"\n%s: Bad argument count\n",progname);
}

bye()
{
    (void) move(23,0);
    (void) clrtoeol();
    (void) refresh();
    (void) endwin();
    (void) exit(0);
}

/******* END OF ELMEDIT.C ************/
