/* $Header: dfile.c,v 2.1 89/06/09 12:25:24 network Exp $
 *
 * Filter destination(s) through delivery file(s).
 *
 * $Log:	dfile.c,v $
 * Revision 2.1  89/06/09  12:25:24  network
 * Update RCS revisions.
 * 
 * Revision 1.10  89/06/09  12:23:49  network
 * Baseline for 2.0 release.
 * 
 */

#include "deliver.h"
#include <sys/stat.h>

/*----------------------------------------------------------------------
 * Filter all valid destinations through the global delivery file.
 */

sys_dfile(dac, dav)
int     dac;
char    **dav;
{
	char    **fav;
	int     fac, a;
	struct stat st;

	/*
	 * If there is no global delivery file, forget it.
	 */

	if (stat(sys_deliver, &st) == -1)
	{
		if (verbose)
			message("no system delivery file\n");
		return -1;
	}

	/*
	 * If we've been asked not to run delivery files, forget it.
	 */

	if (!rundfiles)
	{
		if (verbose)
			message("system delivery file disabled\n");
		return -1;
	}

	/*
	 * Collect the arguments for the delivery file.
	 */

	fav = (char **) zalloc((dac + 3) * sizeof(char **));
	fav[0] = shell;
	fav[1] = sys_deliver;
	fac = 2;

	for (a = 0; a < dac; ++a)
	{
		char    *addr;

		addr = dav[a];
		if (valid_address(addr))
		{
			/* Let the delivery file handle valid addresses. */

			fav[fac++] = addr;
		}
		else
		{
			/* Note invalid address(es); report them later. */

			(void) dest(addr, (char *) NULL);
		}
	}

	fav[fac] = NULL;

	/*
	 * If there were any good names found, let loose the delivery
	 * file.  Note the meaning of "good" is "well-formed", not "valid".
	 * Thus the system delivery file has control over the handling of
	 * all local deliveries, not just those to valid users.
	 */

	if (fac > 2)
		(void) do_dfile(eff_ct, fav, (DEST *)NULL);

	free((char *) fav);

	return 0;
}

/*----------------------------------------------------------------------
 * Filter some undelivered destinations through the post-user
 * delivery file.
 */

post_dfile()
{
	DEST    *d;
	char    **fav;
	int     num_dests, fac;
	struct stat st;

	/*
	 * If there is no post-user delivery file, forget it.
	 */

	if (stat(post_deliver, &st) == -1)
	{
		if (verbose)
			message("no post-user delivery file\n");
		return -1;
	}

	/*
	 * If we've been asked not to run delivery files, forget it.
	 */

	if (!rundfiles)
	{
		if (verbose)
			message("post-user delivery file disabled\n");
		return -1;
	}

	/*
	 * Generate the delivery file argument list.
	 */

	num_dests = 0;
	for (d = first_dest(); d; d = next_dest(d))
		++num_dests;

	fav = (char **) zalloc((num_dests + 3) * sizeof(char **));
	fav[0] = shell;
	fav[1] = post_deliver;
	fac = 2;

	for (d = first_dest(); d; d = next_dest(d))
	{
		if ((d->d_class == CL_USER || d->d_class == CL_UUCP)
		 && (d->d_state == ST_WORKING
		  || (d->d_state == ST_ERROR && d->d_error == E_NSUSER)))
		{
			fav[fac++] = d->d_name;
			d->d_state = ST_HOLD;
		}
	}

	fav[fac] = NULL;

	if (fac > 2)
		(void) do_dfile(eff_ct, fav, (DEST *)NULL);

	free((char *) fav);

	return 0;
}

/*----------------------------------------------------------------------
 * Filter all user destinations through their local delivery files.
 */

user_dfiles()
{
	DEST    *d;
	int     nfound;

	/*
	 * If we've been asked not to run delivery files, forget it.
	 */

	if (!rundfiles)
	{
		if (verbose)
			message("user delivery files disabled\n");

		return -1;
	}


	/*
	 * Continue to loop through all addresses until no destination
	 * that needs expanding can be found.
	 */

	do {
		nfound = 0;
		for (d = first_dest(); d; d = next_dest(d))
		{
			if (d->d_class == CL_USER
			 && d->d_state == ST_WORKING
			 && !d->d_dfdone)
			{
				one_dfile(d);
				d->d_dfdone = TRUE;
			}
		}
	} while (nfound > 0);

	return 0;
}

/*----------------------------------------------------------------------
 * Run the delivery file (if any) for the specified destination.
 */

one_dfile(d)
DEST    *d;
{
	CONTEXT *ct;
	char    *fav[4];
	char    udel_path[100];
	struct stat st;

	if ((ct = name_context(d->d_name)) == NULL)
	{
		dest_err(d, E_CTLOST);
		return;
	}

	/*
	 * If user's home directory is missing, forget it.
	 * If user's home directory is writable to the world,
	 * executing the delivery file would allow a security breach!
	 * Thanks to Jon Zeeff for this hint...
	 */

	if (stat(ct->ct_home, &st) == -1
	 || (st.st_mode & S_IFMT) != S_IFDIR)
	{
		if (verbose)
			message("user %s: home directory %s is missing!\n",
				ct->ct_name, ct->ct_home);
		return;
	}

	if (st.st_mode & 02)
	{
		if (verbose)
			message("user %s: home directory is writable to the world!\n",
				ct->ct_name);
		return;
	}

	/*
	 * If there is no delivery file to execute, just return.
	 */

	(void) sprintf(udel_path, "%s/%s", ct->ct_home, user_deliver);
	if (stat(udel_path, &st) == -1)
	{
		if (verbose)
			message("%s has no delivery file\n", d->d_name);
		return;
	}

	/*
	 * Time to run the file!
	 * We put this dest on hold, so that it will be ignored unless
	 * the delivery file names it.
	 */

	d->d_state = ST_HOLD;

	fav[0] = shell;
	fav[1] = udel_path;
	fav[2] = d->d_name;
	fav[3] = NULL;
	(void) do_dfile(ct, fav, d);
}

/*----------------------------------------------------------------------
 * Process a delivery file.
 */

int
do_dfile(ct, av, d)
CONTEXT *ct;
char    **av;
DEST    *d;
{
	FILE    *fp;
	char    *name, *mailbox;

	if (!ct)
		return -1;

	if (! ok_context(ct))
	{
		if (d)
			dest_err(d, E_CTPERM);
		else
			message("No permissions to run as %s\n", ct->ct_name);

		return -1;
	}

	/* Copy the temp files again */

	if (copy_again() < 0)
		return -1;

	/* Allow the given user to own and read the copies */

	if (give_temps(ct) < 0)
		return -1;

	/* Here we go! */

	if (verbose)
		message("Processing delivery file as %s\n", ct->ct_name);

	if ((fp = ct_popenv(ct, shell, av, "r")) == NULL)
	{
		error("can't execute delivery file as %s\n", ct->ct_name);
		return -1;
	}

	/*
	 * Read the standard output of the delivery file.
	 */

	while (dfile_gets(fp, &name, &mailbox) >= 0)
	{
		DEST    *nd;

		nd = dest(name, mailbox);
		if (nd->d_state == ST_HOLD)
			nd->d_state = ST_WORKING;

		/*
		 * If the delivery file specified a mailbox, verify
		 * that the user whose delivery file is running has
		 * permissions for the requested context.
		 */

		if ((nd->d_state == ST_WORKING) && (mailbox != NULL))
		{
			CONTEXT *nct;

			if ((nct = name_context(name)) == NULL)
				dest_err(nd, E_CTLOST);
			else if (! ok_context(nct))
				dest_err(nd, E_CTPERM);
		}
	}

	return ct_pclose(fp);
}

/*----------------------------------------------------------------------
 * Get and parse a single delivery file output line.
 */

int
dfile_gets(fp, namep, mailboxp)
FILE    *fp;
char    **namep;
char    **mailboxp;
{
	char    *p, *q;
	static  char    buf[BUFSIZ];

	if (fgets(buf, GETSIZE(buf), fp) == NULL)
		return -1;

	if ((p = strchr(buf, '\n')) != NULL)
		*p = 0;
	else
	{
		int c;

		while ((c = fgetc(fp)) != '\n' && c != EOF)
			; /* keep reading */

		error("invalid line from delivery file: '%s'\n", buf);
		return -1;
	}

	/* Strip out all whitespace and eliminate duplicated slashes */

	p = q = buf;
	while (*p)
	{
		if (isspace(*p))
			++p;
		else if ((*q++ = *p++) == '/')
		{
			while (*p == '/')
				++p;
		}
	}
	*q = 0;

	/* Debugging message: display input line */

	if (verbose)
		message("\t'%s'\n", buf);

	if ((p = strchr(buf, ':')) != NULL)
	{
		*p++ = 0;
		if ((q = strchr(p, ':')) != NULL)
			*q = 0;
	}

	*namep = buf;
	*mailboxp = p;
	return 0;
}

/*----------------------------------------------------------------------
 * Make the temp files readable in the given context.
 * This is needed because the temps are readable by owner only.
 */

int
give_temps(ct)
CONTEXT *ct;
{
	int     err, t;

	if (!ct)
		return -1;

	err = 0;
	for (t = T_HDRCOPY; t <= T_BODYCOPY; ++t)
	{
		if (chmod(tfile[t], 0600) == -1)
		{
			syserr("can't chmod %s", tfile[t]);
			++err;
		}
		if (chown(tfile[t], ct->ct_uid, ct->ct_gid) == -1)
		{
			syserr("can't chown %s to %d/%d",
				tfile[t], ct->ct_uid, ct->ct_gid);
			++err;
		}
	}

	return err ? -1 : 0;
}
