/* $Header: main.c,v 2.1 89/06/09 12:25:32 network Exp $
 *
 * A program to deliver local mail with some flexibility.
 *
 * $Log:	main.c,v $
 * Revision 2.1  89/06/09  12:25:32  network
 * Update RCS revisions.
 * 
 * Revision 1.13  89/06/09  12:23:53  network
 * Baseline for 2.0 release.
 * 
 */

#include "deliver.h"
#include "patchlevel.h"
#include <signal.h>

/*
 * External data.
 */

/* Variables set by getopt() [blech] */

extern  int     optind, opterr;
extern  char    *optarg;

/*
 * Local data
 */

static  char    sys_dfl[] = SYS_DELIVER;
static  char    post_dfl[] = POST_DELIVER;
static  char    user_dfl[] = USER_DELIVER;

/*
 * Global data
 */

int     verbose         = FALSE;
int     dryrun          = FALSE;
int     rundfiles       = TRUE;
int     printaddrs      = FALSE;
int     leavetemps      = FALSE;
int     boxdelivery     = FALSE;

char    *progname       = "deliver";
char    version[32]     = "2.0";
char    *shell          = SHELL;

char    *sys_deliver    = sys_dfl;
char    *post_deliver   = post_dfl;
char    *user_deliver   = user_dfl;
char    *sender         = NULL;
char    *hostname       = NULL;

int     eff_uid         = -1;
int     eff_gid         = -1;
int     real_uid        = -1;
int     real_gid        = -1;

CONTEXT *eff_ct         = NULL;
CONTEXT *real_ct        = NULL;

int     tty_input       = FALSE;
SIGFLAG got_sig         = FALSE;

int     trust_user      = FALSE;
int     trust_delfiles  = FALSE;

char    *ttype[T_MAX]   = { "header", "body", "header copy", "body copy" };
char    *tfile[T_MAX]   = { NULL, NULL, NULL, NULL };
char    *tenv[T_MAX]    = { NULL, NULL, ENV_HEADER, ENV_BODY };
int     tfd[T_MAX]      = { -1, -1, -1, -1 };

/*
 * Local functions.
 */

static  SIGTYPE sighup(), sigint(), sigquit();

/*----------------------------------------------------------------------
 * The Program.
 */

main(argc, argv)
int     argc;
char    **argv;
{
	char    *p;
	int     u, c, errcount, copy;

	/* Make sure that stdout and stderr are interleaved correctly */

	Linebuf(stdout);
	Linebuf(stderr);

	/* Figure out the name used to invoke this program. */

	progname = basename(argv[0]);

	/* What version of the program is this? */

	(void) sprintf(version + strlen(version), ".%02d", PATCHLEVEL);

	/* Figure out the name of this host */

	if ((hostname = gethost()) == NULL)
	{
		hostname = "unknown";
		error("unable to determine host name; using \"%s\"\n",
		      hostname);
	}

	/* Find effective and real uids and gids. */

	eff_uid = geteuid();
	eff_gid = getegid();
	real_uid = getuid();
	real_gid = getgid();

	if (eff_uid != real_uid && eff_uid != 0)
	{
		error("if setuid, must be setuid root\n");
		leave(1);
	}

	/* Process environment: handle recursive invocation */

	if ((p = getenv(ENV_DFLAGS)) != NULL)
	{
		while (*p)
		{
			switch (*p++)
			{
			case 'v':
				verbose = TRUE;
				break;
			case 'd':
				verbose = TRUE;
				dryrun = TRUE;
				break;
			case 'A':
				printaddrs = TRUE;
				dryrun = TRUE;
				break;
			case 'n':
				rundfiles = FALSE;
				break;
			case 't':
				leavetemps = TRUE;
				break;
			}
		}
	}

	if ((p = getenv(ENV_SYSDEL)) != NULL && *p)
		sys_deliver = p;
	if ((p = getenv(ENV_POSTDEL)) != NULL && *p)
		post_deliver = p;
	if ((p = getenv(ENV_USERDEL)) != NULL && *p)
		user_deliver = p;
	if ((p = getenv(ENV_SENDER)) != NULL && *p)
		sender = p;
	if ((p = getenv(ENV_HOSTNAME)) != NULL && *p)
		hostname = p;

	/* Parse command line arguments */

	while ((c = getopt(argc, argv, "vdAntbs:p:u:r:h:")) != EOF)
	{
		switch (c)
		{
		case 'v':
			verbose = TRUE;
			break;
		case 'd':
			verbose = TRUE;
			dryrun = TRUE;
			break;
		case 'A':
			printaddrs = TRUE;
			dryrun = TRUE;
			break;
		case 'n':
			rundfiles = FALSE;
			break;
		case 't':
			leavetemps = TRUE;
			break;
		case 'b':
			boxdelivery = TRUE;
			break;
		case 's':
			if (*optarg)
				sys_deliver = optarg;
			break;
		case 'p':
			if (*optarg)
				post_deliver = optarg;
			break;
		case 'u':
			if (*optarg)
				user_deliver = optarg;
			break;
		case 'r':
			if (*optarg)
				sender = optarg;
			break;
		case 'h':
			if (*optarg)
				hostname = optarg;
			break;
		case '?':
			usage();
		}
	}

	/* If no destinations were given, forget it. */

	if (optind >= argc)
	{
		error("no recipients specified\n");
		usage();
	}

	/* Print a debugging message */

	if (verbose)
	{
		message("%s %s running on host %s\n",
			progname, version, hostname);
	}

	/* Do we trust our caller? */

	if (trusted_uid(real_uid))
		trust_user = TRUE;

	/* Do we trust our delivery files? */

	if (strcmp(sys_dfl, sys_deliver) == 0
	 && strcmp(post_dfl, post_deliver) == 0
	 && strcmp(user_dfl, user_deliver) == 0)
		trust_delfiles = TRUE;

	/* Renounce special privileges if something insecure was requested. */

	if (!trust_user && !trust_delfiles)
	{
		if (setgid(eff_gid = real_gid) == -1
		 || setuid(eff_uid = real_uid) == -1)
		{
			syserr("can't renounce setuid privileges");
			leave(1);
		}
	}

	/* Get the contexts of our effective and real uids. */

	if ((eff_ct = uid_context(eff_uid)) == NULL)
		error("invalid effective uid %d!?\n", eff_uid);

	if ((real_ct = uid_context(real_uid)) == NULL)
		error("invalid real uid %d!?\n", real_uid);

	if (!eff_ct || !real_ct)
		leave(1);

	if (verbose)
	{
		message("effective uid = %s (%d/%d); real uid = %s (%d/%d)\n",
			eff_ct->ct_name, eff_ct->ct_uid, eff_ct->ct_gid,
			real_ct->ct_name, real_ct->ct_uid, real_ct->ct_gid);
	}

	/* Let's be sane about the file creation mask. */

	u = umask(0);
	u &= ~0700;     /* Let's not deprive ourselves of permissions.  */
	u |= 022;       /* Let's be reasonably paranoid about writing.  */
	(void) umask(u);

	/*
	 * Where is the message coming from?
	 */

	if (isatty(0))
		tty_input = TRUE;

	/*
	 * If we are not going to deliver, or if we are receiving the
	 * message from a tty, catch signals so we can remove temp files.
	 * Otherwise, ignore signals.
	 */

	if (dryrun || tty_input)
		catch_sigs();
	else
		ignore_sigs();

	/*
	 * Create the temporary files and write the message to them.
	 */

	copy = copy_message();

	/*
	 * No more signals...
	 */

	ignore_sigs();

	/*
	 * ... but if we had already caught a signal,
	 *     or if copy_msg() had a problem, leave.
	 */

	if ((copy < 0) || got_sig)
	{
		if (got_sig)
			error("caught signal - exiting\n");
		leave(1);
	}

	/*
	 * Set up useful environment variables.
	 * Note that this must be done _after_ copy_message(),
	 * since that's where the temp files are created.
	 */

	setup_environ();

	/*
	 * Perhaps we should consider all arguments as mailbox names...
	 */

	if (boxdelivery)
	{
		int     a;

		if (verbose)
			message("mailbox delivery as %s\n", real_ct->ct_name);

		/*
		 * Consider all arguments as mailbox filenames.
		 */

		for (a = optind; a < argc; ++a)
			(void) dest(real_ct->ct_name, argv[a]);

		if (verbose)
			dumpdests("(should all be mailboxes)");
	}

	/*
	 * They're not mailbox names, so they should be mail addresses.
	 */

	else
	{
		/* Run all destinations though the system delivery file. */

		if (sys_dfile(argc - optind, argv + optind) >= 0)
		{
			if (verbose)
				dumpdests("after running system delivery file");
		}
		else
		{
			int     a;

			/*
			 * System delivery file is missing or ignored.
			 * Use the argument list verbatim.
			 */

			for (a = optind; a < argc; ++a)
				(void) dest(argv[a], (char *) NULL);

			if (verbose)
				dumpdests("as taken from argument list");
		}

		/*
		 * Run each user destination through his delivery file.
		 */

		if (user_dfiles() >= 0)
		{
			if (verbose)
				dumpdests("after running user delivery files");
		}

		/*
		 * Run each remaining destination though the post-user
		 * delivery file.
		 */

		if (post_dfile() >= 0)
		{
			if (verbose)
				dumpdests("after running post-user delivery file");
		}
	}

	/*
	 * Drop mail in mailbox(es).
	 */

	mbox_deliver();

	if (verbose)
		dumpdests("after delivery to all mailboxes");

	/*
	 * Send mail to UUCP address(es).
	 */

	uucp_deliver();

	if (verbose)
		dumpdests("after delivery to UUCP addresses");

	/*
	 * Report any errors, and leave.
	 */

	errcount = report_errors();

	/*
	 * All done.
	 */

	leave(errcount ? 1 : 0);
	/* NOTREACHED */
}

/*----------------------------------------------------------------------
 * Print a usage message and exit.
 */

usage()
{
	message("Usage: %s [-b][-A][-d][-v][-n][-t][-r from][-h host] args\n", progname);
	message("-b       All arguments are mailbox filenames.\n");
	message("         (Default: arguments are user names.)\n");
	message("-A       Resolve addresses but do not deliver.\n");
	message("-d       Be verbose but do not deliver.\n");
	message("-v       Be verbose and deliver.\n");
	message("-n       Do not run any delivery files.\n");
	message("-t       Do not remote temp files before exiting.\n");
	message("-s file  Specify the system delivery filename.\n");
	message("-p file  Specify the post-user delivery filename.\n");
	message("-u file  Specify the user delivery filename.\n");
	message("-r from  Specify the address to appear in the \"From \" line.\n");
	message("-h host  Specify the host name.\n");
	message("args     Either user addresses or mailboxes (-b).\n");
	leave(1);
}

/*----------------------------------------------------------------------
 * Clean up and exit.
 */

leave(code)
int     code;
{
	if (! leavetemps)
	{
		int     t;

		for (t = 0; t < T_MAX; ++t)
		{
			if (tfd[t] != -1)
				(void) close(tfd[t]);
			if (tfile[t] && unlink(tfile[t]) == -1)
				syserr("can't unlink %s", tfile[t]);
		}
	}

	exit(code);
}

/*----------------------------------------------------------------------
 * Catch signals.
 */

catch_sigs()
{
	if (signal(SIGHUP, SIG_IGN) != SIG_IGN)
		(void) signal(SIGHUP, sighup);
	if (signal(SIGINT, SIG_IGN) != SIG_IGN)
		(void) signal(SIGINT, sigint);
	if (signal(SIGQUIT, SIG_IGN) != SIG_IGN)
		(void) signal(SIGQUIT, sigquit);
}

/*----------------------------------------------------------------------
 * Ignore signals.
 */

ignore_sigs()
{
	(void) signal(SIGHUP, SIG_IGN);
	(void) signal(SIGINT, SIG_IGN);
	(void) signal(SIGQUIT, SIG_IGN);
}

static SIGTYPE
sighup()
{
	(void) signal(SIGHUP, sighup);
	got_sig = TRUE;
}

static SIGTYPE
sigint()
{
	(void) signal(SIGINT, sigint);
	got_sig = TRUE;
}

static SIGTYPE
sigquit()
{
	(void) signal(SIGQUIT, sigquit);
	got_sig = TRUE;
}

/*----------------------------------------------------------------------
 * Report any errors to stderr.
 * Return an error count.
 */

int
report_errors()
{
	DEST    *d;
	int     count = 0;

	for (d = first_dest(); d; d = next_dest(d))
	{
		if (d->d_state != ST_ERROR)
			continue;

		if (++count == 1)
		{
			error(
		    "delivery to the following address(es) failed on host %s\n",
				hostname);
		}

		message("\t\"%s\"", d->d_name);
		if (d->d_class == CL_MBOX)
			message(", mailbox \"%s\"", d->d_mailbox);
		message(": %s\n", derrmsg(d->d_error));
	}

	return count;
}

/*----------------------------------------------------------------------
 * Is the given uid trusted?
 */

int
trusted_uid(uid)
int     uid;
{
	CONTEXT *ct;
	char    **n;
	static char *t[] = { TRUSTED_USERS, 0 };

	for (n = t; *n; ++n)
	{
		if ((ct = name_context(*n)) != NULL && uid == ct->ct_uid)
			return TRUE;
	}

	return FALSE;
}

/*----------------------------------------------------------------------
 * Set up useful environment variables.
 */

setup_environ()
{
	char    flags[8];
	int     f = 0;

	flags[f++] = '-';
	if (verbose)
		flags[f++] = (dryrun ? 'd' : 'v');
	if (printaddrs)
		flags[f++] = 'A';
	if (leavetemps)
		flags[f++] = 't';
	flags[f] = 0;

	alloc_env(ENV_DFLAGS, (f > 1) ? flags : "");
	if (sys_deliver && *sys_deliver)
		alloc_env(ENV_SYSDEL, sys_deliver);
	if (user_deliver && *user_deliver)
		alloc_env(ENV_USERDEL, user_deliver);
	if (hostname && *hostname)
		alloc_env(ENV_HOSTNAME, hostname);
	if (sender && *sender)
		alloc_env(ENV_SENDER, sender);

	alloc_env("IFS", " \t\n");
	del_env("ENV");         /* in case SHELL is ksh */
}
