#ifndef lint
static char rcsid[] = "$Header: pxl.c,v 1.9 85/10/15 18:19:05 chris Exp $";
#endif

/* Routines for manipulating PXL files---Vax version */

/* The functions DMagFactor, GenPXLFileName, and ReadPXLFile could go in
   separate files, but it's likely that any program that uses any one
   function uses all three. */

#include "pxl.h"
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

extern int errno;

char *malloc (), *sprintf (), *getenv ();
long  lseek ();

/* Convert a magnification factor to floating point.  This is used in
   conjunction with the FONT_SLOP stuff to try to get the file names
   right, and may also be used by DVI reading programs to get slightly
   more accurate values for (mag/1000.0). */
double
DMagFactor (mag)
int mag;
{
    if (mag == 1095)
	return 1.095445;	/* stephalf */
    if (mag == 1315)
	return 1.314534;	/* stepihalf */
    if (mag == 2074)
	return 2.0736;		/* stepiv */
    if (mag == 2488)
	return 2.48832;		/* stepv */
    if (mag == 2986)
	return 2.985984;	/* stepiv */
 /* remaining mags have been ok */
    return (double) mag / 1000.;
}

/* Generate the name of a PXL font file, given the partial name `nm',
   at the magnification `magfactor', with design size `designsize',
   a global mag factor of `globalmag' (default 1000), and a fontname
   path of `path' (default PXLPATH).  Paths are defined as colon-
   separated strings (eg, "/usr/foo:/usr/bar").
   
   Font directories either contain all the fonts directly, or else each
   font is in its own subdirectory.  For example, amti10.* would be
   found in the `amti10' directory; ambx7.* in `ambx7', and so forth.
   The latter method is used iff there is a file in the main directory
   called `SUBDIR'. */
char *
GenPXLFileName (nm, magfactor, designsize, globalmag, path)
char *nm;
int   magfactor, designsize, globalmag;
char *path;
{
    double  dmag;
    register char  *s,
		   *p;
    register int    mag,	/* expected magnification */
                    slop;	/* current slop index */
    char   *slash;		/* path name separator */
    char   *fmt;		/* either oldfmt or newfmt */
    char    pbuf[256];		/* expansion area for path components */

    static char oldfmt[] = "%s%s%s%0.0s.%dpxl";
    static char newfmt[] = "%s%s%s/%s.%dpxl";
				/* the formats for generating names */
				/* arguments are path, /, nm, nm, sz */
    static char rv[256];	/* the generated name (and other uses) */

    if (globalmag == 0)
	globalmag = 1000;
    mag = (int) ((double) magfactor / (double) designsize * 1000.0 + .5);
    dmag = DMagFactor (mag);
    mag = (int) (DMagFactor (globalmag) * 1000.0 * dmag + .5);
    if (mag > 9999)
	mag = 9999;
    else if (mag < 0)
	mag = 0;

 /* For some reason I used to look for slashes in `nm', and suppress path
    expansion if there were any.  This seems awfully bogus, so I've changed
    it to suppress path expansion iff the font name starts with '/'.  Also,
    we no longer choke on font names with percent signs in them. */
    if (*nm == '/')
	slash = s = "";
    else {
	s = path;
	if (s == 0 || *s == 0)
	    s = PXLPATH;
	slash = "/";
    }

 /* Tread the path in s, trying for the file, and testing "nearby" names */
    while (s) {
	p = pbuf;
	while (*s) {
	    if ((*p = *s++) == ':')
		break;
	    p++;
	}
	*p = 0;
	if (*s == 0)
	    s = 0;

     /* choose new format if SUBDIR file exists, old format otherwise */
     /* but always use old format for names starting with / */
	if (*nm == '/')
	    fmt = oldfmt;
	else {
	    (void) sprintf (rv, "%s%sSUBDIR", pbuf, slash);
	    fmt = access (rv, 0) == 0 ? newfmt : oldfmt;
	}

	for (slop = 0; slop < FONT_SLOP; slop++) {
	    (void) sprintf (rv, fmt, pbuf, slash, nm, nm, mag + slop);
	    if (access (rv, 4) == 0)
		return rv;
	    if (slop) {
		(void) sprintf (rv, fmt, pbuf, slash, nm, nm, mag - slop);
		if (access (rv, 4) == 0)
		    return rv;
	    }
	}
    }

 /* Not found - give up and return expected name anyway */
 /* Someday, we may actually generate the PXL (or GF) file here. */
    (void) sprintf (rv, "%s.%dpxl", nm, mag);/* note: no path! */
    return rv;
}

/* Read in a PXL font file, performing all the necessary byte swapping.
   Return a pointer to a struct pxltail.  If 'getrast' is true, get the
   raster too. */

#define TAILSIZE (517 * 4)	/* size of pxl tail info */

struct pxltail *
ReadPXLFile (nm, getrast)
char *nm;
int   getrast;
{
    register char  *p;
    register int    f;
    struct stat stat;
#define px ((struct pxltail *) p)

 /* First open the file, and read the pxltail part */

    if ((f = open (nm, 0)) < 0)
	return 0;
    (void) fstat (f, &stat);
 /* There should be 4n bytes, with an absolute minimum of TAILSIZE + 4
    (+4 for the initial PXLID) */
    if (stat.st_size & 3 || stat.st_size < (TAILSIZE + 4)) {
	errno = EINVAL;
	return 0;
    }
    (void) lseek (f, (long) (-TAILSIZE), 2);
    p = malloc ((unsigned) TAILSIZE);
    if (p == 0)
	return 0;
    if (read (f, p, TAILSIZE) != TAILSIZE) {
	int     saverr = errno;
	free (p);
	errno = saverr;
	return 0;
    }

#ifdef vax
 /* Next, byte swap, since PXL file byte order is the opposite of VAX byte
    order */
    machint ((int *) p, TAILSIZE / 4);
#endif

 /* Finally, check the PXLID part, and if desired, read the character
    rasters */

    if (px -> px_pxlid != PXLID)
	error (0, 0, "Warning: strange PXL id (%d) in \"%s\"",
		px -> px_pxlid, nm);
    if (getrast) {
	int     rastsize = stat.st_size - (TAILSIZE + 4);

	if (ReadRasters (px -> px_info, f, rastsize, nm)) {
	    int     saverr = errno;
	    free (p);
	    errno = saverr;
	    return 0;
	}
    }
    (void) close (f);

    return px;
#undef px
}

/* Round n up to the nearest multiple of r (r must be a power of two) */
#define ROUND(n,r) (((n) + ((r) - 1)) & ~((r) - 1))

/* Read the raster info from a PXL file.  To avoid system call overhead, we
   read the entire thing into core, then break up the glyphs for the chinfo
   array.  NOTE: if malloc fails, we don't try very hard to clean up---most
   programs will die anyway if they fail to find a font.

   The reason for breaking up the characters in the first place is so that
   the storage allocated to any particular glyph may be freed.  The rotation
   code uses this to get rid of bits it no longer needs.  If they are not
   released, it is possible to run out of core when using many fonts. */
static
ReadRasters (ch, fd, sz, nm)
register struct chinfo *ch;
int fd, sz;
char *nm;
{
    register int    i;

    if (i = sz) {
	register char  *p,	/* glyph storage area */
		       *rp;	/* raster pointer */
	register int	rs;	/* raster size, in bytes */

    /* First, read all the glyph bits */
	if ((p = malloc ((unsigned) i)) == 0)
	    return -1;
	(void) lseek (fd, 4L, 0);
	if (read (fd, p, i) != i)
	    return -1;

    /* Next, break them up into the 128 characters */
	p -= 4;			/* to account for the initial PXLID */
	for (i = 128; --i >= 0; ch++) {
	    if (ch -> ch_rastoff) {
		rs = ch -> ch_height * (ROUND (ch -> ch_width, 32) >> 3);
		if ((rp = malloc ((unsigned) rs)) == 0)
		    return -1;
	     /* the << 2 accounts for the fact that ch_rastoff is for 32
		bit things, and we are using 8 bits (32/8 = 4 = 1<<2) */
		bcopy (&p[ch -> ch_rastoff << 2], rp, rs);
		ch -> ch_raster = rp;
	    }
	}

    /* Finally, release the temporary copy */
	free (p + 4);
    }
    else {			/* should be 'totally white' */
	for (i = 128; --i >= 0; ch++)
	    if (ch -> ch_rastoff || ch -> ch_height || ch -> ch_width ||
		    ch -> ch_yoffset || ch -> ch_xoffset)
		error (1, 0, "bad pxl file \"%s\"...help!", nm);
    }

    return 0;
}
