/*
	This routine is meant to act as an interface between drawimp
	(which knows nothing about how text is done on the Imagen or
	the format of a font file) and the textimp routine (which
	knows nothing about Macintoshs or MacDraw.
	It translates MacDraw font and style numbers into font
	specifications and also does ajustments for the weird way
	that the MacDraw text is placed in the object rectangles.
*/

#include <stdio.h>
#include "drawimp.h"

/* This table and the following constants are used to specify the font
   file names.  The constants should correspond to the strings in "fontname". 
   In other words, fontname[AMR] should be "amr". */
/* We have since changed over to the CM fonts.  Only the names in this
   array have been changed, the constants like AMR have been left as is. */
static char *fontname[] = { "cmr", "cmbx", "cmti", "cmsy",
			    "cmbsy", "cmmi", "cmmib" };
#define ANY	-2
#define NONE	-1
#define AMR 	0
#define AMBX	1
#define AMTI	2
#define AMSY	3
#define AMBSY	4
#define AMMI	5
#define AMBI	6
#define DEF_FONT AMR

#define MAX_SIZES   8

/* The following table maps the Mac font sizes into TeX font sizes.  The
   value of "f_size" is combined with the font name from the "fontname"
   array to make a string like "amr10".  The "d_size" and "a_size" fields
   are the "design size" and "at size" for the font in TeX sp units (65536
   sp's to a point) and are used to select the size of font to be used.  The
   d_size is just the f_size * 65536.  The a_size is selected to match one
   of the fonts available on the system.  The routines which find the
   font file will look for a file with a name like "amr10.nnnnpxl", where
   nnnn is the integer result of 
	(a_size / d_size) * (printer_resolution_in_DPI / 200) * 1000.
   The a_size values should be changed to cause the above equation to give
   a result that matches whatever font size is desired for that size of Mac
   font.  The values in the table below were selected to produce characters
   of about the right size on a 300DPI Imagen given the available font
   files.  Since I don't know what fonts are usually around for 240DPI
   devices so I can't tell if any of these values will map to the fonts that
   would be available for that resolution printer.

   TeX does all this in a much cleaner way, but this will do for now.
*/ 
static struct fontsize { 
	int f_size, d_size, a_size;
	int il_size, sp_size, tab_size;
	int shift_down, shift_right;
 } fontsize[MAX_SIZES] = {
	 0,        0,       0,	 0,	 0,	 0,	 0,	 0,
	 9,   589824,  589824,	12,      3,	 6,	 9,	 1, /* 9 */ 
	10,   655360,  655360,	12,	 3,      6,	 9,	 1, /* 10 */ 
	10,   655360,  786432,	16,	 4,	 8,     11,	 1, /* 12 */
	10,   655360,  943718,	19,	 4,	 8,	13,      1, /* 14 */ 
	10,   655360, 1132462,	23,	 6,	12,	16,	 2, /* 18 */
	10,   655360, 1630745,	29,	 8,	16,	20,	 2, /* 24 */
	10,   655360, 2174485,	46,	12,	24,	33,	 2  /* 36 */
 };

/* This table maps Mac font and character codes to TeX font and character
   codes.  The table contains three entries for each Mac character code.
   The entries give the TeX font and character for plain, bold, italic.
   Control characters such as carriage return, tab, and space are in the
   table with a 0x80 added to the ASCII code to flag them so textimp can
   handle them differently. */
static struct fontcode {
	short font;
	short code;
} fontcode[224][3] = {	/* order is: plain, bold, italic */
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	ANY,	0x89,	ANY,	0x89,	ANY,	0x89,	/* 09 tab */
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	ANY,	0x8D,	ANY,	0x8D,	ANY,	0x8D,	/* 0D carriage ret */
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	

	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	

	ANY,	0xA0,	ANY,	0xA0,	ANY,	0xA0,	/* 20 space */
	AMR,	0041,	AMBX,	0041,	AMTI,	0041,	/* 21 ! */
	AMR,	0175,	AMBX,	0175,	AMTI,	0175,	/* 22 " */
	AMR,	0043,	AMBX,	0043,	AMTI,	0043,	/* 23 # */
	AMR,	0044,	AMBX,	0044,	AMTI,	0044,	/* 24 $ */
	AMR,	0045,	AMBX,	0045,	AMTI,	0045,	/* 25 % */
	AMR,	0046,	AMBX,	0046,	AMTI,	0046,	/* 26 & */
	AMR,	0023,	AMBX,	0023,	AMTI,	0023,	/* 27 ' */
	AMR,	0050,	AMBX,	0050,	AMTI,	0050,	/* 28 ( */
	AMR,	0051,	AMBX,	0051,	AMTI,	0051,	/* 29 ) */
	AMR,	0052,	AMBX,	0052,	AMTI,	0052,	/* 2A * */
	AMR,	0053,	AMBX,	0053,	AMTI,	0053,	/* 2B + */
	AMR,	0054,	AMBX,	0054,	AMTI,	0054,	/* 2C , */
	AMR,	0055,	AMBX,	0055,	AMTI,	0055,	/* 2D - */
	AMR,	0056,	AMBX,	0056,	AMTI,	0056,	/* 2E . */
	AMR,	0057,	AMBX,	0057,	AMTI,	0057,	/* 2F / */

	AMR,	0060,	AMBX,	0060,	AMTI,	0060,	/* 30 0 */
	AMR,	0061,	AMBX,	0061,	AMTI,	0061,	/* 31 1 */
	AMR,	0062,	AMBX,	0062,	AMTI,	0062,	/* 32 2 */
	AMR,	0063,	AMBX,	0063,	AMTI,	0063,	/* 33 3 */
	AMR,	0064,	AMBX,	0064,	AMTI,	0064,	/* 34 4 */
	AMR,	0065,	AMBX,	0065,	AMTI,	0065,	/* 35 5 */
	AMR,	0066,	AMBX,	0066,	AMTI,	0066,	/* 36 6 */
	AMR,	0067,	AMBX,	0067,	AMTI,	0067,	/* 37 7 */
	AMR,	0070,	AMBX,	0070,	AMTI,	0070,	/* 38 8 */
	AMR,	0071,	AMBX,	0071,	AMTI,	0071,	/* 39 9 */
	AMR,	0072,	AMBX,	0072,	AMTI,	0072,	/* 3A : */
	AMR,	0073,	AMBX,	0073,	AMTI,	0073,	/* 3B ; */
	AMMI,	0074,	AMBI,	0074,	AMMI,	0074,	/* 3C < */
	AMR,	0075,	AMBX,	0075,	AMTI,	0075,	/* 3D = */
	AMMI,	0076,	AMBI,	0076,	AMMI,	0076,	/* 3E > */
	AMR,	0077,	AMBX,	0077,	AMTI,	0077,	/* 3F ? */

	AMR,	0100,	AMBX,	0100,	AMTI,	0100,	/* 40 @ */
	AMR,	0101,	AMBX,	0101,	AMTI,	0101,	/* 41 A */
	AMR,	0102,	AMBX,	0102,	AMTI,	0102,	/* 42 B */
	AMR,	0103,	AMBX,	0103,	AMTI,	0103,	/* 43 C */
	AMR,	0104,	AMBX,	0104,	AMTI,	0104,	/* 44 D */
	AMR,	0105,	AMBX,	0105,	AMTI,	0105,	/* 45 E */
	AMR,	0106,	AMBX,	0106,	AMTI,	0106,	/* 46 F */
	AMR,	0107,	AMBX,	0107,	AMTI,	0107,	/* 47 G */
	AMR,	0110,	AMBX,	0110,	AMTI,	0110,	/* 48 H */
	AMR,	0111,	AMBX,	0111,	AMTI,	0111,	/* 49 I */
	AMR,	0112,	AMBX,	0112,	AMTI,	0112,	/* 4A J */
	AMR,	0113,	AMBX,	0113,	AMTI,	0113,	/* 4B K */
	AMR,	0114,	AMBX,	0114,	AMTI,	0114,	/* 4C L */
	AMR,	0115,	AMBX,	0115,	AMTI,	0115,	/* 4D M */
	AMR,	0116,	AMBX,	0116,	AMTI,	0116,	/* 4E N */
	AMR,	0117,	AMBX,	0117,	AMTI,	0117,	/* 4F O */

	AMR,	0120,	AMBX,	0120,	AMTI,	0120,	/* 50 P */
	AMR,	0121,	AMBX,	0121,	AMTI,	0121,	/* 51 Q */
	AMR,	0122,	AMBX,	0122,	AMTI,	0122,	/* 52 R */
	AMR,	0123,	AMBX,	0123,	AMTI,	0123,	/* 53 S */
	AMR,	0124,	AMBX,	0124,	AMTI,	0124,	/* 54 T */
	AMR,	0125,	AMBX,	0125,	AMTI,	0125,	/* 55 U */
	AMR,	0126,	AMBX,	0126,	AMTI,	0126,	/* 56 V */
	AMR,	0127,	AMBX,	0127,	AMTI,	0127,	/* 57 W */
	AMR,	0130,	AMBX,	0130,	AMTI,	0130,	/* 58 X */
	AMR,	0131,	AMBX,	0131,	AMTI,	0131,	/* 59 Y */
	AMR,	0132,	AMBX,	0132,	AMTI,	0132,	/* 5A Z */
	AMR,	0133,	AMBX,	0133,	AMTI,	0133,	/* 5B [ */
	AMSY,	0156,	AMBSY,	0156,	AMSY,	0156,	/* 5C \ */
	AMR,	0135,	AMBX,	0135,	AMTI,	0135,	/* 5D ] */
	AMR,	0136,	AMBX,	0136,	AMTI,	0136,	/* 5E ^ */
	NONE,	0,	NONE,	0,	NONE,	0,	/* 5F _ */

	AMR,	0022,	AMBX,	0022,	AMTI,	0022,	/* 60 ` */
	AMR,	0141,	AMBX,	0141,	AMTI,	0141,	/* 61 a */
	AMR,	0142,	AMBX,	0142,	AMTI,	0142,	/* 62 b */
	AMR,	0143,	AMBX,	0143,	AMTI,	0143,	/* 63 c */
	AMR,	0144,	AMBX,	0144,	AMTI,	0144,	/* 64 d */
	AMR,	0145,	AMBX,	0145,	AMTI,	0145,	/* 65 e */
	AMR,	0146,	AMBX,	0146,	AMTI,	0146,	/* 66 f */
	AMR,	0147,	AMBX,	0147,	AMTI,	0147,	/* 67 g */
	AMR,	0150,	AMBX,	0150,	AMTI,	0150,	/* 68 h */
	AMR,	0151,	AMBX,	0151,	AMTI,	0151,	/* 69 i */
	AMR,	0152,	AMBX,	0152,	AMTI,	0152,	/* 6A j */
	AMR,	0153,	AMBX,	0153,	AMTI,	0153,	/* 6B k */
	AMR,	0154,	AMBX,	0154,	AMTI,	0154,	/* 6C l */
	AMR,	0155,	AMBX,	0155,	AMTI,	0155,	/* 6D m */
	AMR,	0156,	AMBX,	0156,	AMTI,	0156,	/* 6E n */
	AMR,	0157,	AMBX,	0157,	AMTI,	0157,	/* 6F o */

	AMR,	0160,	AMBX,	0160,	AMTI,	0160,	/* 70 p */
	AMR,	0161,	AMBX,	0161,	AMTI,	0161,	/* 71 q */
	AMR,	0162,	AMBX,	0162,	AMTI,	0162,	/* 72 r */
	AMR,	0163,	AMBX,	0163,	AMTI,	0163,	/* 73 s */
	AMR,	0164,	AMBX,	0164,	AMTI,	0164,	/* 74 t */
	AMR,	0165,	AMBX,	0165,	AMTI,	0165,	/* 75 u */
	AMR,	0166,	AMBX,	0166,	AMTI,	0166,	/* 76 v */
	AMR,	0167,	AMBX,	0167,	AMTI,	0167,	/* 77 w */
	AMR,	0170,	AMBX,	0170,	AMTI,	0170,	/* 78 x */
	AMR,	0171,	AMBX,	0171,	AMTI,	0171,	/* 79 y */
	AMR,	0172,	AMBX,	0172,	AMTI,	0172,	/* 7A z */
	AMSY,	0146,	AMBSY,	0146,	AMSY,	0146,	/* 7B { */
	AMSY,	0152,	AMBSY,	0152,	AMSY,	0152,	/* 7C | */
	AMSY,	0147,	AMBSY,	0147,	AMSY,	0147,	/* 7D } */
	AMR,	0176,	AMBX,	0176,	AMTI,	0176,	/* 7E ~ */
	NONE,	0,	NONE,	0,	NONE,	0,	/* 7F  */

	/* the next two groups (0x8* and 0x9*) are letters with things
	above or below them, like accent marks.  For now, just output
	the letter without the extra mark */

	AMR,	0101,	AMBX,	0101,	AMTI,	0101,	/* 80 A */
	AMR,	0101,	AMBX,	0101,	AMTI,	0101,	/* 81 A */
	AMR,	0103,	AMBX,	0103,	AMTI,	0103,	/* 82 C */
	AMR,	0105,	AMBX,	0105,	AMTI,	0105,	/* 83 E */
	AMR,	0116,	AMBX,	0116,	AMTI,	0116,	/* 84 N */
	AMR,	0117,	AMBX,	0117,	AMTI,	0117,	/* 85 O */
	AMR,	0125,	AMBX,	0125,	AMTI,	0125,	/* 86 U */
	AMR,	0141,	AMBX,	0141,	AMTI,	0141,	/* 87 a */
	AMR,	0141,	AMBX,	0141,	AMTI,	0141,	/* 88 a */
	AMR,	0141,	AMBX,	0141,	AMTI,	0141,	/* 89 a */
	AMR,	0141,	AMBX,	0141,	AMTI,	0141,	/* 8A a */
	AMR,	0141,	AMBX,	0141,	AMTI,	0141,	/* 8B a */
	AMR,	0141,	AMBX,	0141,	AMTI,	0141,	/* 8C a */
	AMR,	0143,	AMBX,	0143,	AMTI,	0143,	/* 8D c */
	AMR,	0145,	AMBX,	0145,	AMTI,	0145,	/* 8E e */
	AMR,	0145,	AMBX,	0145,	AMTI,	0145,	/* 8F e */

	AMR,	0145,	AMBX,	0145,	AMTI,	0145,	/* 90 e */
	AMR,	0145,	AMBX,	0145,	AMTI,	0145,	/* 91 e */
	AMR,	0151,	AMBX,	0151,	AMTI,	0151,	/* 92 i */
	AMR,	0151,	AMBX,	0151,	AMTI,	0151,	/* 93 i */
	AMR,	0151,	AMBX,	0151,	AMTI,	0151,	/* 94 i */
	AMR,	0151,	AMBX,	0151,	AMTI,	0151,	/* 95 i */
	AMR,	0156,	AMBX,	0156,	AMTI,	0156,	/* 96 n */
	AMR,	0157,	AMBX,	0157,	AMTI,	0157,	/* 97 o */
	AMR,	0157,	AMBX,	0157,	AMTI,	0157,	/* 98 o */
	AMR,	0157,	AMBX,	0157,	AMTI,	0157,	/* 99 o */
	AMR,	0157,	AMBX,	0157,	AMTI,	0157,	/* 9A o */
	AMR,	0157,	AMBX,	0157,	AMTI,	0157,	/* 9B o */
	AMR,	0165,	AMBX,	0165,	AMTI,	0165,	/* 9C u */
	AMR,	0165,	AMBX,	0165,	AMTI,	0165,	/* 9D u */
	AMR,	0165,	AMBX,	0165,	AMTI,	0165,	/* 9E u */
	AMR,	0165,	AMBX,	0165,	AMTI,	0165,	/* 9F u */

	AMSY,	0171,	AMBSY,	0171,	AMSY,	0171,	/* A0 dagger */
	AMR,	0027,	AMBX,	0027,	AMTI,	0027,	/* A1 degree */
	NONE,	0,	NONE,	0,	NONE,	0,	/* A2 cents */
	AMTI,	0044,	AMTI,	0044,	AMTI,	0044,	/* A3 British pound */
	AMSY,	0170,	AMBSY,	0170,	AMSY,	0170,	/* A4 ?? */
	AMSY,	0017,	AMBSY,	0017,	AMSY,	0017,	/* A5 filled dot */
	AMSY,	0173,	AMBSY,	0173,	AMSY,	0173,	/* A6 paragraph */
	AMR,	0031,	AMBX,	0031,	AMTI,	0031,	/* A7 ?? */
	NONE,	0,	NONE,	0,	NONE,	0,	/* A8 registered */
	NONE,	0,	NONE,	0,	NONE,	0,	/* A9 copyright */
	NONE,	0,	NONE,	0,	NONE,	0,	/* AA trademark */
	AMR,	0023,	AMBX,	0023,	AMTI,	0023,	/* AB accent */
	AMR,	0177,	AMBX,	0177,	AMTI,	0177,	/* AC two dots */
	NONE,	0,	NONE,	0,	NONE,	0,	/* AD not equal */
	AMR,	0035,	AMBX,	0035,	AMTI,	0035,	/* AE AE */
	AMR,	0037,	AMBX,	0037,	AMTI,	0037,	/* AF O with slash */

	AMSY,	0061,	AMBSY,	0061,	AMSY,	0061,	/* B0 infinity */
	AMSY,	0006,	AMBSY,	0006,	AMSY,	0006,	/* B1 plus-minus */
	AMSY,	0024,	AMBSY,	0024,	AMSY,	0024,	/* B2 <= */
	AMSY,	0025,	AMBSY,	0025,	AMSY,	0025,	/* B3 >= */
	AMMI,	0022,	AMBI,	0022,	AMMI,	0022,	/* B4 theta (??) Yen */
	AMMI,	0026,	AMBI,	0026,	AMMI,	0026,	/* B5 mu */
	AMMI,	0100,	AMBI,	0100,	AMMI,	0100,	/* B6 partial */
	AMR,	0006,	AMBX,	0006,	AMTI,	0006,	/* B7 Sigma */
	AMR,	0005,	AMBX,	0005,	AMTI,	0005,	/* B8 Pi */
	AMMI,	0031,	AMBI,	0031,	AMMI,	0031,	/* B9 pi */
	AMSY,	0163,	AMBSY,	0163,	AMSY,	0163,	/* BA integral */
	AMMI,	0013,	AMBI,	0013,	AMMI,	0013,	/* BB alpha (??) a underbar */
	AMMI,	0041,	AMBI,	0041,	AMMI,	0041,	/* BC omega (??) o underbar */
	AMR,	0012,	AMBX,	0012,	AMTI,	0012,	/* BD Omega */
	AMR,	0032,	AMBX,	0032,	AMTI,	0032,	/* BE ae */
	AMMI,	0014,	AMBI,	0014,	AMMI,	0014,	/* BF beta (??) o with slash) */

	AMR,	0076,	AMBX,	0076,	AMTI,	0076,	/* C0 ? upside down */
	AMR,	0074,	AMBX,	0074,	AMTI,	0074,	/* C1 ! upside down */
	AMSY,	0072,	AMBSY,	0072,	AMSY,	0072,	/* C2 not */
	AMSY,	0160,	AMBSY,	0160,	AMSY,	0160,	/* C3 radical */
	NONE,	0,	NONE,	0,	NONE,	0,	/* C4 ?? */
	AMSY,	0031,	AMBSY,	0031,	AMSY,	0031,	/* C5 approx. equal */
	AMSY,	0064,	AMBSY,	0064,	AMSY,	0064,	/* C6 triangle */
	AMSY,	0034,	AMBSY,	0034,	AMSY,	0034,	/* C7 << */
	AMSY,	0035,	AMBSY,	0035,	AMSY,	0035,	/* C8 >> */
	NONE,	0,	NONE,	0,	NONE,	0,	/* C9 ellipsis */
	NONE,	0,	NONE,	0,	NONE,	0,	/* CA non-brea. sp. */
	AMR,	0101,	AMBX,	0101,	AMTI,	0101,	/* CB A with accent */
	AMR,	0101,	AMBX,	0101,	AMTI,	0101,	/* CC A with tilde */
	AMR,	0117,	AMBX,	0117,	AMTI,	0117,	/* CD O with tilde */
	AMR,	0036,	AMBX,	0036,	AMTI,	0036,	/* CE OE */
	AMR,	0033,	AMBX,	0033,	AMTI,	0033,	/* CF oe */

	AMR,	0173,	AMBX,	0173,	AMTI,	0173,	/* D0 n-dash */
	AMR,	0174,	AMBX,	0174,	AMTI,	0174,	/* D1 m-dash */
	AMR,	0134,	AMBX,	0134,	AMTI,	0134,	/* D2 open quotes */
	AMR,	0042,	AMBX,	0042,	AMTI,	0042,	/* D3 close quotes */
	AMR,	0140,	AMBX,	0140,	AMTI,	0140,	/* D4 open quote */
	AMR,	0047,	AMBX,	0047,	AMTI,	0047,	/* D5 close quote */
	AMSY,	0004,	AMBSY,	0004,	AMSY,	0004,	/* D6 divide */
	AMSY,	0005,	AMBSY,	0005,	AMSY,	0005,	/* D7 diamond */
	AMR,	0171,	AMBX,	0171,	AMTI,	0171,	/* D8 y with umlaut */
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	
	NONE,	0,	NONE,	0,	NONE,	0,	

};
#define MAX_CHAR (sizeof(fontcode) / sizeof(struct fontcode))

int il_size, sp_size, tab_size;
static init = 0;
static char fontspec[40];
extern char *malloc();

mactext(s,startv,starth,style,font,size,spacing,align,angle)
char *s;
int startv,starth,style,font,size,spacing,align,angle;
{
	int id, i, v, h, cur_font, new_line;
	struct fontsize *fs;
	struct fontcode *fc;
	char *p, *b;
	if (init == 0) {
		fs = fontsize;	/* change things to Imagen units */
		for (i = 0; i <= MAX_SIZES; i++) {
			fs->il_size     = md2im(fs->il_size);
			fs->sp_size     = md2im(fs->sp_size);
			fs->tab_size    = md2im(fs->tab_size);
			fs->shift_down  = md2im(fs->shift_down);
			fs->shift_right = md2im(fs->shift_right);
			fs++;
		}
		init = 1;
	}
	if (size > 7)
		size = 7;
	fs = &fontsize[size];
	il_size = fs->il_size;
	sp_size = fs->sp_size;
	tab_size= fs->tab_size;
	v = startv + fs->shift_down;
	h = starth + fs->shift_right;
/*	fprintf(stderr,"startv=%d, starth=%d, size=%d, v=%d, h=%d\n",
		startv, starth, size, v, h); */
	if (style > 2)		/* only plain, bold, and italic */
		style = 0;	/* anything else is plain */
	b = malloc(strlen(s) + 1);
	new_line = 1;
	while (*s != '\0') {
		p = b;
		cur_font = NONE;
		while (*s != '\0') {
			i = *s & 0377;
			if (i >= MAX_CHAR)
				i = 0;
			fc = &fontcode[i][style];
			if (cur_font == NONE && fc->font != ANY)
				cur_font = fc->font;
			if (fc->font != ANY && fc->font != NONE &&
			    cur_font != fc->font)
				break;	/* changed fonts */
			if (fc->font != NONE)
				*p++ = fc->code;
			s++;
		}
		*p = '\0';
		if (cur_font == NONE)
			cur_font = DEF_FONT;
		sprintf(fontspec,"%s%d",fontname[cur_font],fs->f_size);
/*		fprintf(stderr,"%s %s\n",fontspec,b); */
		id = selectfont(fontspec, fs->a_size, fs->d_size);
		textimp(b,v,h,new_line);
		new_line = 0;	/* make textimp leave position as is */
	}
}
