/*
	drawimp - MacDraw to Impress translator

	This program will read a MacDraw file and output a sequence of
	Impress commands to draw the picture on an Imagen laser printer.
	Program is compatible with MacDraw files created
	by "MacDraw Version 1.7 3/18/85".  

	Drawimp has two processing modes.  The normal mode causes
	the MacDraw files listed on the command line to be processed
	in order listed with the Impress data written to stdout with
	ENDPAGE commands between data from each file.  This will put
	each file on separate pages.

	In TeX mode (with -t switch) each input file, is processed
	and the results are written to a separate output file with
	the extension (if any) changed to .imp.  The output files have
	no header or trailing ENDPAGE or EOF commands.

	The -d switch causes debugging information to be written to 
	stderr.  This consists of a list of objects and some info about
	them.

	The -p switch is used to select which pages of the MacDraw file
	to output.  Syntax is: -pr1-r2:c1-c2[,r1-r2:c1-c2]

	The -R switch is used to specify the Image resolution in dots per
	inch.  The default is set in drawimp.h

	A few examples:

		drawimp Picture.data >Picture.imp
		drawimp -p1:2 Picture.data >Picture.imp
		drawimp -d Picture.data >/dev/null
		drawimp -d Picture.data >Picture.imp
		drawimp -t Picture1.data Picture2.data
		drawimp -R240 Picture1.data | ipr

	Not (yet) implemented:
		Smoothed polygons
		PICT objects from MacPaint
		Clipping of objects that cross 8x10 inch page boundaries

	Note: This program has been used extensively on VAX systems but
	only for a short time on SUN systems.  It seems to work on SUNs
	but some compatibility bugs may still be left in it.  The only
	code that is dependent on the VAX byte ordering is in swapbyte()
	and is is conditionally compiled.  There may be other places where
	the program needs to be changed for other machines.

	Please send any bug reports to Weber%Brand@USC-Oberon.USC.EDU

	Allan Weber (Weber%Brand@USC-Oberon.USC.EDU)
	USC Signal and Image Processing Institute

	History:
	16-Apr-85	Version 1
	23-Apr-85	Version 2 - Fixed bug in arcs, added arrows,
			added rest of patterns, changed cvtpnt to use
			32-bit coordinates, added -p switch for page select
	28-Aug-85	Version 3 - Added text using TeX fonts and routines
			from the Unix TeX distribution.	 Added fix to arc
			routine from Richard Roy (dick@su-dit).
	16-Dec-85	Version 4
			Added -t switch to produce impress file without
			header, endpage, or eof bytes for including
			output file in TeX documents going to Imagen.
			Also moves whatever is on page to upper left corner.
			More fixes to arcs to handle filled arcs and arcs
			of other than 90 degrees.
			Implemented different radius corners for the rounded
			rectangles.
	10-Apr-86	Fixed bug in TeX mode stuff to shift picture
			to top-left.
	27-Jan-87       Ported to SUN systems.  Fixed a couple of bugs 
	                that showed up.
*/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "drawimp.h"

/* #define DEBUG */		/* define DEBUG to get point conversions
				   printed out */
/* #define NO_TEXT */		/* define NO_TEXT to build without
				   text capability */

FILE *infp, *outfp, *fontfp;
int gdepth, done;
int debug = 0;
int tex_output = 0;
int draw_top, draw_left;
int line_table[] = { 0, 0, 1, 8, 16, 20 };	/* line width table */
int ar_size[] = { 0, 0, 10, 15, 20, 25};	/* arrowhead size table */
int corner_rad[] = { 0, 0, 9, 14, 18, 23, 27 }; /* corner radius table */
int texture_table[37];
char *ProgName;
char fheader[HEADER_SIZE];
char tmp[MAX_RECORD];
char filename[80];
struct MDheader rheader;
struct srect cur_page;
double ar_angle, ar_size2[6], ar_radius[6];
extern char pattern[37][128];

/*
This array contains info about each page of the file begin drawn.  The objects
in the MacDraw file can occur in any order and we have to sort them out into
separate page for output.  The .flag indicates whether this page is to be 
output or not (from -p option).  The .font field is used to store the ImFamily
value from the textimp.c file that was in use the last time we were doing that
page.  By remembering which font was in use on a page, we don't have to send
the imPress SetFamily commands each time we go back to put more stuff on that
page if the font hasn't changed.  The .name field holds the name of the 
temporary file so we can delete it later.  The file pointer is in .fp and is
0 if no file open for that page yet.  The page_ptr variable points to the
entry for the page currently being worked on.
*/
struct page {		/* table of drawing pages */
	int flag;	/* flag for page in use */
	int font;	/* font in use on this page */
	char name[20];	/* file name of where the data went */
	FILE *fp;	/* file pointer */
} pagetbl[MAC_V_PAGES][MAC_H_PAGES], *page_ptr;

/* This table contains the values used to convert from MacDraw units
(1/72 inch) to the resolution of the Imagen.  The .res field is the
Imagen resolution in pixels/inch.  The .num and .den fields are the 
numerator and denoninator used to do the multiplication, as in:
	Imagen = MacDraw * .num / .den
The scaling factor is split into separate numerator and denominator
to allow conversion without overflow problems.
*/
struct {
	int res;
	int num,den;
} restbl[] = {
	240, 10, 3,
	300, 25, 6,
	-1, 0, 0
	};
int md2im_num, md2im_den;
int resolution = DEF_RES;

char *getpages();
int terminal(), text(), vector(), rectangle(), roundrect();
int ellipse(), arc(), curve(), polygon(), group();
int (*opfun[])() = {terminal, text, vector, vector, rectangle, roundrect,
		    ellipse, arc, curve, polygon, group };

char *USAGE = "[-Rres] [-d] [-t] [-ppages] in_file > out_file";

extern char *malloc();
extern char *mktemp();
extern char *strcpy();
extern FILE *impout();
#ifndef NO_TEXT
extern int ImFamily;	/* current Imagen font number from textimp */
#endif

main(argc,argv)
int argc;
char *argv[];
{	
	int i;
	int h,h1,h2;
	int v,v1,v2;
	int gotpages = 0;
	char *s,c;
	struct lrect r;
	struct page *pp;
	ProgName = argv[0];

	while(--argc > 0 && (*++argv)[0] == '-') {
		for (s = argv[0]+1; *s != '\0'; s++)
			switch(*s) {
			case 'd':	/* debug */
				debug = 1;
				break;
			case 'p':	/* page select */
				++s;
				while (*s != '\0') {
					v1 = 1; v2 = 5;
					s = getpages(s,&v1,&v2);
					if (*s != ':') {
						fprintf(stderr,
						"%s: error in page list\n",
						ProgName);
						argc = -1;
						while (*s != '\0')
							s++;
						break;
					}
					else
						s++;
					h1 = 1; h2 = 12;
					s = getpages(s,&h1,&h2);
					if (v1 == 0)
						v1 = 1;
					if (h1 == 0)
						h1 = 1;
					for (v = v1; v <= v2; v++)
						for (h = h1; h <= h2; h++)
						    pagetbl[v-1][h-1].flag=1;
					if (*s == ',')
						s++;
				}
				s--;	/* back pointer up one place */
				gotpages = 1;
				break;
			case 't':
				tex_output = 1;
				break;
			case 'R':	/* Image resolution */
 				s++;
				if (sscanf(s,"%d",&resolution) != 1) {
				    fprintf(stderr,"%s: Bad resolution - %s\n",
					ProgName,s);
				    argc = -1;
				}
				while (*s != '\0')
					s++;
				s--;
				break;
			default:
				fprintf(stderr,"%s: illegal option - %c\n",
				    ProgName,*s);
				argc = -1;
				break;
		}
	}

	if (argc <= 0) {
		usage();
		exit(1);
	}

	if (tex_output) {
		if (gotpages == 0)
			pagetbl[0][0].flag = 1;	/* only first page for tex */
		else {
			fprintf(stderr,
				"Cannot use -t and -p options at same time\n");
			usage();
			exit(1);
		}
	}
	else
		if (gotpages == 0) {
			for (v = 0; v < MAC_V_PAGES; v++)  /* do all */
				for (h = 0; h < MAC_H_PAGES; h++)
					pagetbl[v][h].flag = 1;
		}

	/* figure out what resolution and conversion factors to use */
	i = 0;
	while (restbl[i].res > 0) {
		if (restbl[i].res == resolution) {
			md2im_num = restbl[i].num;
			md2im_den = restbl[i].den;
			break;
		}
		i++;
	}
	if (md2im_num == 0) {
		fprintf(stderr,"%s: Unknown resolution - %d\n",
			ProgName,resolution);
		exit(1);
	}

	/* calculate some things for doing arrows */
	ar_angle = ARROW_ANGLE * 2 * 3.14159 / 360;
	for (i = 0; i <= 5; i++) {
		ar_radius[i] = md2im(ar_size[i]);
		ar_size2[i] = pow(ar_radius[i] * cos(ar_angle), (float) 2);
	}

	/* calculate some things for doing rectangle corners */
	for (i = 1; i <= 6; i++) {
		corner_rad[i] = md2im(corner_rad[i]);
	}

	if (tex_output == 0) {
		printf("@document(language impress)");
		impout(stdout);
		imP_set_abs_h(md2im(H_SHIFT));		/* shift origin */
		imP_set_abs_v(md2im(V_SHIFT));
		imP_set_hv_system(3,0,0);	/* set new origin */
	}

	while (argc-- > 0) {
		if ((infp = fopen(*argv,"r")) == NULL) {
			fprintf(stderr,"%s: can't open input file %s\n",
			    ProgName,*argv);
			exit(1);
		}
		if (tex_output) {
			makename(*argv,filename);
			if ((outfp = fopen(filename,"w")) == NULL) {
				fprintf(stderr,
				    "%s: can't open output file %s\n",
				    ProgName,filename);
				exit(1);
			}
			fontfp = outfp;	/* put font info at front */
		}
		else {
			outfp = stdout;
			fontfp = stdout;
		}
		argv++;

		page_ptr = NULL;

		fread(fheader,1,HEADER_SIZE,infp);	/* read header */
		makelrect(fheader+214,&r);
		if (debug) {
			fprintf(stderr,
				"Drawing rectangle: t=%d, l=%d, b=%d, r=%d\n",
				r.top.i,r.left.i,r.bottom.i,r.right.i);
		}

		draw_top = draw_left = 0;
		if (tex_output != 0)
			cvtpnt(&r.top, &r.left, &draw_top, &draw_left);

		/* read the MacDraw objects and process them */
		done = gdepth = 0;
		while (fread((char *)&rheader, sizeof(struct MDheader),
			1, infp) == 1) {
			c = rheader.code;
			if (c >= 0 && c <= 10)
				(opfun[c])();
			else {
				fprintf(stderr,"Unrecognized type - %d\n",c);
				break;
			}
			if (done)
				break;
		}
		fclose(infp);

		/* now output all the collected picture data */
		impout(outfp);
		if (tex_output)
			imP_set_hv_system(3,0,0);	/* set new origin */
		for (v = 0; v < MAC_V_PAGES; v++)
			for (h = 0; h < MAC_H_PAGES; h++) {
				pp = &pagetbl[v][h];
				if (pp->fp != NULL) {
					fseek(pp->fp,0L,0);
					while ((c = getc(pp->fp)) != EOF ||
					       feof(pp->fp) == 0)
						putc(c,outfp);
					fclose(pp->fp);
					pp->fp = NULL;
					unlink(pp->name);
					if (tex_output == 0)
						imP_endpage();
				}
			}
		
#ifndef NO_TEXT
		reset_text();	/* reset text for next file */
#endif
		if (tex_output) {
#ifndef NO_TEXT
			reset_fonts();	/* set fonts as not downloaded yet */
#endif
			reset_textures(); /* same for texture glyphs */
			fclose(outfp);
		}
	}
	/* finish the file off */
	if (tex_output == 0)
		imP_eof();
}

usage()
{
	fprintf(stderr,"Usage: %s %s\n",ProgName, USAGE);
}

/*****************************************************************************/

terminal()
{
	if (debug)
		fprintf(stderr, "Terminator: depth = %d\n",gdepth);
	if (gdepth > 0)
		gdepth--;
	else
		done = 1;
}

text()
{
	int n, doit;
	int style, font, size, spacing, align, angle;
	int top, left, bottom, right;
	int v[5], h[5], tv, th;
	char *s, *s0;
	struct lrect r;
	fread(tmp,sizeof(struct MDtext),1,infp);
	style   = makenum(1,tmp+4);
	font    = makenum(1,tmp+5);
	size    = makenum(1,tmp+6);
	spacing = makenum(1,tmp+7);
	align   = makenum(1,tmp+8);
	angle   = makenum(1,tmp+9);
	n       = makenum(2,tmp+10);
	makelrect0(tmp+12,&r);
	doit = dopage(&r.top,&r.left) != 0;
	if (debug && doit) {
		fprintf(stderr, "Text: %d characters, rect: t=%d, l=%d, b=%d, r=%d\n",
			n,r.top.i,r.left.i,r.bottom.i,r.right.i);
		fprintf(stderr, "\tstyle=%d, font=%d, size=%d, spacing=%d\n",
			style, font, size, spacing);
		fprintf(stderr, "\talign=%d, angle=%d, fill=%d\n",
			align, angle, rheader.fill);
	}
	if ((s = s0 = malloc(n+1)) == 0)
		fprintf(stderr,"Unable to malloc %d byte\n",n+1);
	while (n--)
		*s++ = getc(infp);
	*s = '\0';
	if (debug && doit)
		fprintf(stderr, "\"%s\"\n",s0);
	if (doit) {
		if (rheader.fill > 1) {
			cvtpnt(&r.top,&r.left,&top,&left);
			cvtpnt(&r.bottom,&r.right,&bottom,&right);
			h[0] = h[3] = h[4] = left;
			v[0] = v[1] = v[4] = top;
			h[1] = h[2] = right - 1;
			v[3] = v[2] = bottom - 1;
			imP_create_path(5,h,v);
			imP_fill_path(use_pattern(rheader.fill));
		}
		cvtpnt(&r.top,&r.left,&tv,&th);
#ifndef NO_TEXT
		mactext(s0,tv,th,style,font,size,spacing,align,angle);
#endif
	}
	free(s0);	
}

vector()
{
	struct lpoint pnt1, pnt2;
	int h[2], v[2], ha[3], va[3];
	int pat, ar, s;
	double dh, dv, t;
	fread(tmp,sizeof(struct MDvector),1,infp);
	makelpoint(tmp+0,&pnt1);
	makelpoint(tmp+8,&pnt2);
	if (dopage(&pnt1.v,&pnt1.h) == 0)
		return;
	if (debug) {
		fprintf(stderr, "Vector: From (%dv %dh) to (%dv %dh), ",
			pnt1.v.i,pnt1.h.i,pnt2.v.i,pnt2.h.i);
		fprintf(stderr, "line=%d, pen=%d, fill=%d, arrow=%d\n",
			rheader.line,rheader.pen,rheader.fill,rheader.special);
	}
	if (onpage(&pnt1.v,&pnt1.h) && onpage(&pnt2.v,&pnt2.h) &&
	    rheader.line > 1) {
		cvtpnt(&pnt1.v,&pnt1.h,v,h);
		cvtpnt(&pnt2.v,&pnt2.h,v+1,h+1);
		if (rheader.pen > 1)
			pat = use_pattern(rheader.pen);
		if ((ar = rheader.special) > 0) {
			dh = h[1] - h[0];
			dv = v[1] - v[0];
			s = rheader.line;
			if (dh*dh + dv*dv > ar_size2[s]) {
				t = atan2(dv,dh);
				if (ar == 1 || ar == 3) {
					make_arrow(h+1,v+1,t,s,ha,va);
					imP_create_path(3,ha,va);
					imP_fill_path(pat);
				}
				if (ar == 2 || ar == 3) {
					t += 3.14159;
					make_arrow(h,v,t,s,ha,va);
					imP_create_path(3,ha,va);
					imP_fill_path(pat);
				}
			}
		}
		imP_create_path(2,h,v);
		imP_set_pen(use_line(rheader.line));
		imP_draw_path(pat);
	}
}

rectangle()
{
	struct lrect r;
	int top, left, bottom, right;
	int h[5], v[5];
	fread(tmp,sizeof(struct MDbox),1,infp);
	makelrect(tmp,&r);
	if (dopage(&r.top,&r.left) == 0)
		return;
	if (debug) {
		fprintf(stderr, "Rectangle: t=%d, l=%d, b=%d, r=%d, ",
			r.top.i,r.left.i,r.bottom.i,r.right.i);
		fprintf(stderr, "line=%d, pen=%d, fill=%d, corner=%d\n",
			rheader.line,rheader.pen,rheader.fill,rheader.special);
	}
	if (onpage(&r.top,&r.left) && onpage(&r.bottom,&r.right)) {
		cvtpnt(&r.top,&r.left,&top,&left);
		cvtpnt(&r.bottom,&r.right,&bottom,&right);
		h[0] = h[3] = h[4] = left;
		v[0] = v[1] = v[4] = top;
		h[1] = h[2] = right - 1;
		v[3] = v[2] = bottom - 1;
		if (h[0] >= h[1])	/* vertical line? */
			imP_create_path(2,h+1,v+1);
		else if (v[1] >= v[2])	/* horizontal line? */
			imP_create_path(2,h,v);
		else			/* rectangle */
			imP_create_path(5,h,v);
		if (rheader.fill > 1)
			imP_fill_path(use_pattern(rheader.fill));
		if (rheader.line > 1) {
			imP_set_pen(use_line(rheader.line));
			imP_draw_path(use_pattern(rheader.pen));
		}
	}
}

roundrect()
{
	struct lrect r;
	int top, left, bottom, right, ra, rb, rad, corner;
	int h[9], v[9];
	fread(tmp,sizeof(struct MDbox),1,infp);
	makelrect(tmp,&r);
	if (dopage(&r.top,&r.left) == 0)
		return;
	if (debug) {
		fprintf(stderr, "RoundRect: t=%d, l=%d, b=%d, r=%d, ",
			r.top.i,r.left.i,r.bottom.i,r.right.i);
		fprintf(stderr, "line=%d, pen=%d, fill=%d, corner=%d\n",
			rheader.line,rheader.pen,rheader.fill,rheader.special);
	}
	if (onpage(&r.top,&r.left) && onpage(&r.bottom,&r.right)) {
		cvtpnt(&r.top,&r.left,&top,&left);
		cvtpnt(&r.bottom,&r.right,&bottom,&right);
		ra = (right-1 - left) / 2;
		rb = (bottom-1 - top) / 2;
		rad = corner_rad[rheader.special];
		if (ra > rad)
			ra = rad;
		if (rb > rad)
			rb = rad;
		corner = (ra != 0) || (rb != 0);
		h[0] = h[5] = h[8] = left + ra;
		h[1] = h[4] = right-1 - ra;
		h[2] = h[3] = right-1;
		h[6] = h[7] = left;
		v[0] = v[1] = v[8] = top;
		v[2] = v[7] = top + rb;
		v[3] = v[6] = bottom-1 - rb;
		v[4] = v[5] = bottom-1;
		if (h[0] < h[1])
			imP_create_path(2,h,v);
		imP_set_pum(1);
		if (corner) {
			imP_set_abs_h(h[1]);
			imP_set_abs_v(v[2]);
			imP_ellipse_arc(ra,rb,0,12288,16383);
		}
		if (v[2] < v[3])
			imP_create_path(2,h+2,v+2);
		if (corner) {
			imP_set_abs_h(h[4]);
			imP_set_abs_v(v[3]);
			imP_ellipse_arc(ra,rb,0,0,4096);
		}
		if (h[5] < h[4])
			imP_create_path(2,h+4,v+4);
		if (corner) {
			imP_set_abs_h(h[5]);
			imP_set_abs_v(v[6]);
			imP_ellipse_arc(ra,rb,0,4096,8192);
		}
		if (v[7] < v[6])
			imP_create_path(2,h+6,v+6);
		if (corner) {
			imP_set_abs_h(h[0]);
			imP_set_abs_v(v[7]);
			imP_ellipse_arc(ra,rb,0,8192,12288);
		}
		if (rheader.fill > 1)
			imP_fill_path(use_pattern(rheader.fill));
		if (rheader.line > 1) {
			imP_set_pen(use_line(rheader.line));
			imP_draw_path(use_pattern(rheader.pen));
		}
		imP_set_pum(0);
	}
}

ellipse()
{
	struct lrect r;
	int top, left, bottom, right, ra, rb;
	fread(tmp,sizeof(struct MDbox),1,infp);
	makelrect(tmp, &r);
	if (dopage(&r.top,&r.left) == 0)
		return;
	if (debug) {
		fprintf(stderr, "Ellipse: t=%d, l=%d, b=%d, r=%d, ",
			r.top.i,r.left.i,r.bottom.i,r.right.i);
		fprintf(stderr, "line=%d, pen=%d, fill=%d\n",
			rheader.line,rheader.pen,rheader.fill,rheader.special);
	}
	if (onpage(&r.top,&r.left) && onpage(&r.bottom,&r.right)) {
		cvtpnt(&r.top,&r.left,&top,&left);
		cvtpnt(&r.bottom,&r.right,&bottom,&right);
		imP_set_abs_h((left + right-1) / 2);
		imP_set_abs_v((top + bottom-1) / 2);
		ra = (right-1 - left) / 2;
		rb = (bottom-1 - top) / 2;
		imP_ellipse_arc(ra,rb,0,0,16383);
		if (rheader.fill > 1)
			imP_fill_path(use_pattern(rheader.fill));
		if (rheader.line > 1) {
			imP_set_pen(use_line(rheader.line));
			imP_draw_path(use_pattern(rheader.pen));
		}
	}	
}

arc()
{
	struct lrect r, r2;
	int top, left, bottom, right, ra, rb, ch, cv;
	int start, extent;
	int alpha0, alpha1;
	double fch, fcv, frh, frv, fa1, fa2, fh1, fh2, fv1, fv2;
	fread(tmp,sizeof(struct MDarc),1,infp);
	makelrect(tmp, &r);
	start  = makenum(2,tmp+16);
	extent = makenum(2,tmp+18);
	left = r.left.i;
	if (left > 32767)
		left -= 65536;
	right = r.right.i;
	if (right > 32767)
		right -= 65536;
	top = r.top.i;
	if (top > 32767)
		top -= 65536;
	bottom = r.bottom.i;
	if (bottom > 32767)
		bottom -= 65536;
	fch = (double)(left + right) / 2.0;
	fcv = (double)(top + bottom) / 2.0;
	frh = (double)(right - left) / 2.0;
	frv = (double)(bottom - top) / 2.0;
	fa1 = (double)(start) * 3.14159 / 180.0;
	fa2 = (double)(start + extent) * 3.14159 / 180.0;
/*	fprintf(stderr,"fch=%f, fcv=%f, frh=%f, frv=%f\n",fch,fcv,frh,frv);
	fprintf(stderr,"fa1=%f, fa2=%f\n",fa1,fa2); */
	fh1 = fch + frh * sin(fa1);
	fh2 = fch + frh * sin(fa2);
	fv1 = fcv - frv * cos(fa1);
	fv2 = fcv - frv * cos(fa2);
/*	fprintf(stderr,"fh1=%f, fh2=%f, fv1=%f, fv2=%f\n",fh1,fh2,fv1,fv2); */
	if (fv1 > fv2)
		{ r2.top.i = fv2; r2.bottom.i = fv1; }
	else
		{ r2.top.i = fv1; r2.bottom.i = fv2; }
	if (fh1 > fh2)
		{ r2.left.i = fh2; r2.right.i = fh1; }
	else
		{ r2.left.i = fh1; r2.right.i = fh2; }
	r2.top.f = r2.left.f = r2.bottom.f = r2.right.f = 0;
/*	fprintf(stderr, "top=%d, left=%d, bottom=%d, right=%d\n",
			r2.top.i,r2.left.i,r2.bottom.i,r2.right.i); */
	if (dopage(&r2.top,&r2.left) == 0)
		return;
	if (debug) {
		fprintf(stderr, "Arc: t=%d, l=%d, b=%d, r=%d, ",
			r2.top.i,r2.left.i,r2.bottom.i,r2.right.i);
		fprintf(stderr, "line=%d, pen=%d, fill=%d\n",
			rheader.line,rheader.pen,rheader.fill);
		fprintf(stderr, "\tstart = %d, extent = %d\n",start,extent);
	}
	if (onpage(&r2.top,&r2.left) && onpage(&r2.bottom,&r2.right)) {
		cvtpnt(&r.top,&r.left,&top,&left);
		cvtpnt(&r.bottom,&r.right,&bottom,&right);
		ch = (left + right-1) / 2;
		cv = (top + bottom-1) / 2;
		imP_set_abs_h(ch);
		imP_set_abs_v(cv);
		ra = (right-1 - left) / 2;
		rb = (bottom-1 - top) / 2;
/* changed 8/22/85 by rhr - apparently MacDraw allows arcs to be traversed
 * in negative direction (Imagen does not)! Debug output gave extent=65446!
 * which for 16 bit 2's complement would be about -90!  So, this fix assumes
 * all arcs are 90 degrees in extent, plus or minus, and adjusts start 
 * accordingly */
/* changed again 11/4/85 by AGW to allow arcs of other than 90 degrees */
		if (extent > 32768) {	/* negative direction? */
			extent = 65536 - extent;
			start = start - extent;
		}
		if (rb > ra)	/* v axis = major axis */
			start = (start + 270) % 360;
		alpha0 = ((start - 90) * 4096) / 90;
		if (alpha0 < 0)
			alpha0 += 16384;
		if (extent == 360)
			alpha1 = alpha0 + 16383;
		else
			alpha1 = alpha0 + ((extent * 4096) / 90);
		if (rheader.fill > 1) {
			imP_create_path(1, &ch, &cv);
			imP_set_pum(1);
			imP_ellipse_arc(ra,rb,0,alpha0,alpha1);
			imP_fill_path(use_pattern(rheader.fill));
			imP_set_pum(0);
		}
		if (rheader.line > 1) {
			imP_ellipse_arc(ra,rb,0,alpha0,alpha1);
			imP_set_pen(use_line(rheader.line));
			imP_draw_path(use_pattern(rheader.pen));
		}	
	}
}

curve()
{
	int n, n0, dx, dy;
	struct lpoint pnt;
	int *h, *h0, *v, *v0;
	int doit;
	struct lrect r;
	fread(tmp,sizeof(struct MDcurve),1,infp);
	n = makenum(2,tmp+4);
	makelrect(tmp+6, &r);
	doit = dopage(&r.top,&r.left) != 0;
	if (debug && doit) {
		fprintf(stderr, "Curve: n=%d, t=%d, l=%d, b=%d, r=%d, ",
			n,r.top.i,r.left.i,r.bottom.i,r.right.i);
		fprintf(stderr, "line=%d, pen=%d, fill=%d\n",
			rheader.line,rheader.pen,rheader.fill);
	}
	makelpoint(tmp+24,&pnt);
	/* null point at end gets discarded which gives room in array
	   for point to close region if needed */
	h = h0 = (int *) malloc((unsigned) sizeof(int) * n);
	v = v0 = (int *) malloc((unsigned) sizeof(int) * n);
	n0 = 0;
	while (--n) {
		if (debug && doit)
			fprintf(stderr, "\t%d: (%dv %dh)\n",
				n0,pnt.v.i,pnt.h.i);
		cvtpnt(&pnt.v,&pnt.h,v++,h++);
		n0++;
		dx = getc(infp);
		dy = getc(infp);
		if (dx > 127)
			dx -= 256;
		if (dy > 127)
			dy -= 256;
		pnt.h.i += dx;
		pnt.v.i += dy;
	}
	if (debug && doit)
		fprintf(stderr, "\t%d: (%dv %dh)\n",n0,pnt.v.i,pnt.h.i);
	if (tmp[22] == 1) {	/* check for closed curve */
		*h = *h0;
		*v = *v0;
		n0++;
	}
	if (doit) {
		if (onpage(&r.top,&r.left) && onpage(&r.bottom,&r.right)) {
			imP_create_path(n0,h0,v0);
			if (rheader.fill > 1)
				imP_fill_path(use_pattern(rheader.fill));
			if (rheader.line > 1) {
				imP_set_pen(use_line(rheader.line));
				imP_draw_path(use_pattern(rheader.pen));
			}
		}
	}
	free((char *) h0);
	free((char *) v0);
}

polygon()
{
	int n, n0;
	struct lpoint pnt;
	int *h, *h0, *v, *v0;
	int doit;
	struct lrect r;
	fread(tmp,sizeof(struct MDpolygon),1,infp);
	n = makenum(2,tmp+4);
	makelrect(tmp+6, &r);
	doit = dopage(&r.top,&r.left) != 0;
	if (debug && doit) {
		fprintf(stderr, "Polygon: n=%d, t=%d, l=%d, b=%d, r=%d, ",
			n,r.top.i,r.left.i,r.bottom.i,r.right.i);
		fprintf(stderr, "line=%d, pen=%d, fill=%d\n",
			rheader.line,rheader.pen,rheader.fill);
	}
	h = h0 = (int *) malloc((unsigned) sizeof(int) * (n+1));
	v = v0 = (int *) malloc((unsigned) sizeof(int) * (n+1));
	n0 = 0;
	while (n--) {
		fread(tmp,1,8,infp);
		makelpoint(tmp,&pnt);
		if (debug && doit)
			fprintf(stderr, "\t%d: (%dv %dh)\n",
				n0,pnt.v.i,pnt.h.i);
		cvtpnt(&pnt.v,&pnt.h,v++,h++);
		n0++;
	}
	if (tmp[22] == 1) {	/* check for closed polygon */
		*h = *h0;
		*v = *v0;
		n0++;
	}
	if (doit) {
		if (onpage(&r.top,&r.left) && onpage(&r.bottom,&r.right)) {
			imP_create_path(n0,h0,v0);
			if (rheader.fill > 1)
				imP_fill_path(use_pattern(rheader.fill));
			if (rheader.line > 1) {
				imP_set_pen(use_line(rheader.line));
				imP_draw_path(use_pattern(rheader.pen));
			}
		}
	}
	free((char *) h0);
	free((char *) v0);
}

group()
{
	struct lrect r;
	fread(tmp,sizeof(struct MDgroup),1,infp);
	makelrect(tmp, &r);
	gdepth += 1;
	if  (debug) {
		fprintf(stderr, "Group start: depth = %d, ", gdepth);
		fprintf(stderr, "rectangle: t=%d, l=%d, b=%d, r=%d\n",
			gdepth,r.top.i,r.left.i,r.bottom.i,r.right.i);
	}
}

/*****************************************************************************/

makenum(i,p)		/* make a number from bytes in 68000 order */
int i;
char *p;
{
	int n = 0;
	while (i--) 
		n = (n << 8) + (*p++ & 0xff);
	return(n);
}

makelpoint(cp, pp)	/* swap bytes 4 times to make a lpoint struct */
char *cp;
struct lpoint *pp;
{
	swapbyte(cp, (char *)pp, 4);
}

makelrect(cp, rp)	/* swap bytes 8 times to make a lrect struct */
char *cp;
struct lrect *rp;
{
	swapbyte(cp, (char *)rp, 8);
}

makelrect0(cp, rp)	/* same as makelrect except fractional part = 0 */
char *cp;
struct lrect *rp;
{
	swapbyte(cp+0, (char *)&rp->top.i, 1);
	swapbyte(cp+2, (char *)&rp->left.i, 1);
	swapbyte(cp+4, (char *)&rp->bottom.i, 1);
	swapbyte(cp+6, (char *)&rp->right.i, 1);
	rp->top.f = rp->left.f = rp->bottom.f = rp->right.f = 0;
}

/* MacDraw binary numbers are in 68000 byte order which is opposite from
   VAXes.  If we are running on a VAX, this routine copies n pairs of
   bytes from p1 to p2 swapping the order.  If not on a VAX, it just copies
   n pairs of bytes from p1 to p2.  Note: I have never tested drawimp on
   a non-VAX so there may be other things that need to be changed that
   I have forgotten about.   -  AGW
*/
swapbyte(p1, p2, n)
register char *p1, *p2;
register int n;
{
#ifdef vax
	while (n-- > 0) {
		*(p2+1) = *p1++;
		*p2 = *p1++;
		p2 += 2;
	}
#else
	bcopy(p1, p2, n*2);
#endif
}

cvtpnt(mv,mh,iv,ih)	/* convert point from MacDraw to Imagen coord. */
struct lcoord *mv, *mh;
int *iv, *ih;
{
	int a,b;
	a = mv->i;
	b = mv->f;
	if (a > 32767)
		a -= 65536;
	a -= cur_page.top;
	*iv = (md2im((a << 8) + (b >> 8 & 0xff)) >> 8) - draw_top;
	a = mh->i;
	b = mh->f;
	if (a > 32767)
		a -= 65536;
	a -= cur_page.left;
	*ih = (md2im((a << 8) + (b >> 8 & 0xff)) >> 8) - draw_left;
#ifdef DEBUG
	fprintf(stderr, "\tv.i=%d v.f=%d v=%d, h.i=%d h.f=%d h=%d\n",
		mv->i,mv->f,*iv,mh->i,mh->f,*ih);
#endif
}

md2im(x)	/* convert a value from MacDraw to Imagen */
int x;
{
	return(x * md2im_num / md2im_den);
}

onpage(pv,ph)
struct lcoord *pv, *ph;
{
	return(pv->i >= cur_page.top && pv->i <= cur_page.bottom &&
	       ph->i >= cur_page.left && ph->i <= cur_page.right);
}

use_pattern(f)
char f;
{
	if (f == 2)	/* white */
		return(0);
	else if (f == 3)	/* black */
		return(15);
	else {			/* something else */
		load_texture(f); 
		return(3);
	}
}

use_line(n)
char n;
{
	if (n < 1 || n > 5)
		return(1);
	else
		return(line_table[n]);
}

load_texture(n)
int n;
{
	FILE *oldfp;
	if (n < 1 || n > 36)
		return;
	if (texture_table[n] == 0) {
		oldfp = impout(fontfp);	/* send textures out to font file */
		imP_bgly(0,TEXTURE_FAMILY,n,32,32,0,32,32,pattern[n]);
		impout(oldfp);
		texture_table[n] = 1;
	}
	imP_set_texture(TEXTURE_FAMILY,n);
}

reset_textures()	/* mark all textures as not yet loaded */
{
	int i;
	for (i = 1; i <= 36; i++)
		texture_table[i] = 0;
}

dopage(pv, ph)
struct lcoord *pv, *ph;
{
	struct page *pp;
	int r,c;
	char s[20];
	r = pv->i / MAC_V_SIZE;
	c = ph->i / MAC_H_SIZE;
	pp = &pagetbl[r][c];
	if (r > MAC_V_PAGES || c > MAC_H_PAGES)
		return(0);
	else if (pp->flag == 0)
		return(0);
	else {
		if (pp->fp == NULL) {
			strcpy(s,mktemp(TEMP_PREFIX));
			sprintf(pp->name,"%s.%01d%02d",s,r+1,c+1);
			pp->fp = fopen(pp->name,"w+");
			pp->font = -1;	/* force a imP_set_family() */
		}
		impout(pp->fp);
#ifndef NO_TEXT
		if (page_ptr != NULL)
			page_ptr->font = ImFamily; /* font for last page */
		ImFamily = pp->font;	/* font in use on new page */
#endif
		page_ptr = pp;		/* points to current page */
		cur_page.bottom = (cur_page.top= r * MAC_V_SIZE) + MAC_V_SIZE;
		cur_page.right = (cur_page.left= c * MAC_H_SIZE) + MAC_H_SIZE;
		return(1);
	}
}

/*
This routine parses a page select specification of the form n-m where
n is the first one to do and m is the last one to do.  If either is
missing, the value in n or m is left unchanged.  If only one number 
is given, m and n will both be set equal to it.  The parameter s is
a pointer to the string and the function returns an updated pointer
to the next character in the string after doing the parsing.
*/
char *
getpages(s,n,m)
char *s;
int *n,*m;
{
	int x;
	if (isdigit(*s)) {
		x = 0;
		while (isdigit(*s))
			x = x * 10 + (*s++ - '0');
		*n = x;
	}
	if (*s == '-') {
		if (isdigit(*++s)) {
			x = 0;
			while (isdigit(*s))
				x = x * 10 + (*s++ - '0');
			*m = x;
		}
	}
	else
		*m = *n;
	return(s);
}

/*
This routine take the endpoint of the vector (*h, *v), its angle (t), and
the line size (n) as input and fills the ha[] and va[] arrays with the
three points needed to draw the arrowhead.  It also modifies the endpoint
location to move it inside the arrowhead.  If this isn't done, the line
will show around the arrow tip when using wide line sizes.
*/
make_arrow(h,v,t,n,ha,va)
int *h, *v, n, ha[3], va[3];
double t;
{
	double tp, tm ,r;
	r = ar_radius[n];
	tp = t + ar_angle;
	tm = t - ar_angle;
	ha[0] = *h;			/* make triangle for the arrow	*/
	va[0] = *v;
	ha[1] = *h - r * cos(tm);
	va[1] = *v - r * sin(tm);
	ha[2] = *h - r * cos(tp);
	va[2] = *v - r * sin(tp);
	*h -= r * cos(t) * 0.5;		/* move endpoint of vector back	*/
	*v -= r * sin(t) * 0.5;		/* away from tip of arrowhead	*/
}

/* Create a filename for storing output for later inclusion with a dvi file */
makename(s1, s2)
char *s1, *s2;
{
	int n;
	char *p;
	p = s1;
	while (*p != '\0') {	/* get rid of leading path name */
		if (*p == '/')
			s1 = p+1;
		p++;
	}
	p = s1 + strlen(s1);	/* point to null byte */

	while (p > s1 && *--p != '.')	/* back up looking for a . */
		;
	if (*p == '.' && strcmp(p, MAC_EXT) == 0) {
		n = p - s1;
		strncpy(s2, s1, n);
		s2[n] = '\0';
		strcat(s2, TEX_EXT);
	}
	else {		/* wrong or no extension so just add to end */
		strcpy(s2, s1);
		strcat(s2, TEX_EXT);
	}

}
