/*  util.c - general utility routines shared by multiple components  */

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>	/* needed by HP-UX */
#include <fcntl.h>	/* needed by HP-UX */
#include "libc.h"



/******************************  error exits  ******************************/



char *xprefix = 0;	/* set this to prefix messages with arg0 or whatever */



/*  mexit(s) - exit immediately, with message  */

void
mexit(s)
char *s;
{
    fflush(stdout);
    if (xprefix)
	fprintf(stderr,"%s: ",xprefix);
    fprintf(stderr,"%s\n",s);
    exit(1);
}



/*  pexit(s) - call perror(s) and exit immediately  */

void
pexit(s)
char *s;
{
    fflush(stdout);
    if (xprefix)
	fprintf(stderr,"%s: ",xprefix);
    perror(s);
    exit(1);
}



/***************************  memory allocation  ***************************/



/*  alloc(n) - allocate n bytes, with success guaranteed  */

char *
alloc(n)
int n;
{
    char *s;

    s = malloc((unsigned) n);
    if (!s)
	mexit("out of memory");
    return(s);
}



/*  salloc(s) - allocate and initialize string, with success guaranteed  */

char *
salloc(s)
char *s;
{
    return strcpy(alloc(strlen(s)+1),s);
}



/****************************  string checking  ****************************/



/*  strtail(s,t) - return pointer to tail substring t of s, or 0 if no match */

char *
strtail(s,t)
char *s, *t;
{
    int ls = strlen(s);
    int lt = strlen(t);
    if (ls < lt)
	return 0;
    s = s + ls - lt;
    if (strcmp(s,t) == 0)
	return s;
    else
	return 0;
}



/****************************  file opening  ****************************/



/*  mustopen(filename,type) - fopen() with guaranteed success */

FILE *
mustopen(filename,type)
char *filename, *type;
{
    FILE *fp;

    fp = fopen(filename,type);
    if (!fp)
	pexit(filename);
    return fp;
}



/****************************  exec assistance  ****************************/



int trcexec;	/* if set, trace execs on standard error output */



/*  spawn(path,args,dir,input,output) - vfork, chdir, redirect, exec, wait
 *
 *  input or output can be null if no redirection is needed.
 *  if an output file is specified it is used for both stdout and stderr.
 */

int
spawn(path,args,dir,input,output)
char *path, *dir, *input, *output;
char **args;
{
    int status;
    int pid;
    void doexec();

    fflush(stdout);
    fflush(stderr);
    pid = vfork();
    if (pid < 0)
	pexit("can't fork");
    else if (pid == 0) {
	/* we're the child process */
	if (dir)		/* change current directory if specified */
	    if (chdir(dir) != 0)
		pexit(dir);
	if (input != 0)  {	/* redirect stdin if specified */
	    close(0);
	    if (open(input,O_RDONLY,0) < 0)
		pexit(input);
	}
	if (output != 0)  {	/* redirect stdout if specified */
	    close(1);
	    if (open(output,O_WRONLY|O_CREAT|O_TRUNC,0666) < 0)
		pexit(output);
	    dup2(1,2);		/* dup output file for use also as stderr */
	}
	doexec(path,args);	/* exec the program */
	perror(path);		/* diagnose error if exec failed */
	exit(1);
    }
    wait(&status);
    return status;
}



/*  doexec(path,argv) - trace if requested, exec program, abort on error  */

void
doexec(path,argv)
char *path, **argv;
{
    char **argp;

    fflush(stdout);
    if (trcexec)  {
	fprintf(stderr,"(%s)",path);
	for (argp = argv; *argp; argp++)
	    fprintf(stderr," %s",*argp);
	fprintf(stderr,"\n");
    }
    fflush(stderr);
    execv(path,argv);
    pexit(path);
}
