/*  symbols.c - miscellaneous Symbol routines that don't alter the symtabs  */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"
#include "../util.h"



/*  new_symbol(kind) - allocate new Symbol structure, and initialize it  */

Symptr
new_symbol(kind)
Kind kind;
{
    Symptr s;
    static Symbol z;

    s = (Symptr) alloc (sizeof (Symbol));	/* allocate */
    *s = z;					/* zero all fields */
    s->s_kind = kind;				/* set kind as specified */
    s->s_type = T_NOTATYPE;			/* set rest to defaults */
    s->s_restrict = R_NOTARESTRICT;
    s->s_used = FALSE;
    s->s_invoked = R_NOTARESTRICT;
    s->s_predef = PRE_notapredef;
    s->s_segment = S_NOTASEGMENT;
    s->s_offset = OFF_UNK;
    s->s_desoff = OFF_UNK;
    s->s_reply = FALSE;
    s->s_impl = IM_NOTYETIMPL;
    s->s_how_imported = TK_NOTATOKEN;
    return (s);
}



/* copy_from_decl(to,from) - copy Symbol attributes.
 *
 * used to install parameters where type info comes from op declaration.
 * makes a copy of type related info of symbol table entry.
 */
void
copy_from_decl(to,from)
Symptr to,from;
{
    to->s_type = from->s_type;
    to->s_tdef = from->s_tdef;
    to->s_restrict = from->s_restrict;
    to->s_size = from->s_size;
    if (from->s_ranges)
	/* very important that get own copy of ranges. */
	add_range(to,
		  copy_nodes(from->s_ranges->r_dim1),
		  copy_nodes(from->s_ranges->r_dim2));
    else
	to->s_ranges = (Rangeptr) NULL;
}



/*  add_range(sym,dim1,dim2) - add ranges to symbol structure  */

void
add_range(sym,dim1,dim2)
Symptr sym;
Nodeptr dim1, dim2;
{
    Rangeptr f;

    f = (Rangeptr) alloc (sizeof (Range));
    f->r_dim1 = dim1;
    f->r_dim2 = dim2;
    sym->s_ranges = f;
}



/* check_used() - check for unused identifiers in current block.
 *
 * called at the end of each command_block and at the end of a resource.
 * note: we don't want to check for unused identifiers in the spec because they
 * can be used from outside the resource; thus, here is the right place to do
 * the check.  furthermore, don't want to incorporate the check to see
 * whether an operation is implemented here because we don't check spec stuff;
 * so easiest to keep that separate.
 *
 * note: a primary purpose for maintaining usage infor is to detect whether
 * an operation is assigned to a capability or passed as a parameter; such use
 * precludes it from being a semaphore operation.  (For simplicity, if an
 * operation is only compared, then it can't be a semaphore operation.)
 * Because it is easiest to do this for all identifiers, we also can warn about
 * variables that are declared but not used.
 * (note: we might want to combine this with how an operation is implemented.)
 */
void
check_used()
{
    Symptr sym;

    assert (st_cb != NULLSYM);
    for (sym = st_cb->s_next; sym != NULLSYM; sym = sym->s_next)
	switch (sym->s_kind) {
	    case K_CONST:
	    case K_VAR:
	    case K_PARAM:
	    case K_RESULT:
		if (! sym->s_used)
		    errmsg(E_WARN,"identifier declared but not used: %s",
			sym->s_name);
		break;
	    case K_OP:
	    case K_SEMAPHORE:
		if (sym->s_invoked == R_NOTARESTRICT && !sym->s_used) {
		    strcpy(yytext,sym->s_name);
		    if (sym->s_impl!=IM_NOTYETIMPL)
			errmsg(E_WARN,
			    "op declared and implemented, but never used: %s",
			    yytext);
		    else
			errmsg(E_WARN,"op declared but never used: %s",yytext);
		}
		break;
/* imports into local blocks are now allowed.  should probably catch
 * unused imports too, but to do so would need to change denotation code.  */
	    case K_IMPORT:
	    case K_NOTAKIND:
	    case K_TYPE:
	    case K_OPTYPE:
	    case K_LITERAL:
	    case K_PREDEF:
	    case K_ANON:
		/* do nothing. */
		break;
	    case K_BLOCK:
	    case K_FREE:
	    case K_FIELD: /* don't check aux table. */
	    default:
		boom("bad kind in check_used");
		/*NOTREACHED*/
	}
}
