/* sr.h -- general definitions for the sr compiler */




/*************************  Symbolic Constants  *************************/

#define NAMESIZE 32
#define FILE_NAME_SIZE (NAMESIZE+10)

/* various implementation limitations */
#define MAX_PROC		200  /* procs per resource */
#define MAX_NEST		20   /* syntactical nesting */
#define MAX_ALLOC		200  /* unfreed invocation blocks */

/* Errors types */
#define E_FATAL 0200
#define E_WARN  0100
#define ERROR(n,s) errmsg(n,s,NULLSTR)
#define WARN(s) errmsg(E_WARN,s,NULLSTR)
#define FATAL(s) errmsg(E_FATAL,s,NULLSTR)

/*  initial (illegal) value for labels in intermediate code  */
#define NOLAB 0

/*  Pseudo-type Bool for making C look a little like Pascal  (no comment...)  */
#define Bool int
#define FALSE 0
#define TRUE 1

/*  String storage overhead (length field, room for NUL) excluding alignment */
#define STR_OVH (INTSIZE+1)

/*  Null character pointer  (stdio's NULL isn't char* on all systems) */
#define NULLSTR ((char *) 0)

/*  Typedef for the type of chars used inside lex  */
#ifdef hpux
typedef unsigned char YYCHAR;
#else
typedef char YYCHAR;
#endif




/*************************  Code Generation  *************************/

#define ALIGN 4		/* keep stack aligned on ALIGN byte boundaries */
#define align(x) ((x+3) & ~3)
#define CHARSIZE 1
#define BOOLSIZE 1
#define INTSIZE 4
#define ENUMSIZE INTSIZE
#define VMSIZE INTSIZE
#define PTRSIZE 4
#define FILESIZE PTRSIZE
#define SEMSIZE 4
#define CL_SIZE 4
#define OFF_UNK (1<<30)




/*************************  Macros  *************************/

/*  IS_ORDERED(type) -- check that type is ordered. Does not check for scalar.*/
#define IS_ORDERED(t) ((t)==T_INT || (t)==T_CHAR || (t)==T_BOOL || (t)==T_ENUM)

/*  IS_SCALAR(sym) -- check that signature indicates a scalar  */
#define IS_SCALAR(sym) ((sym)->s_ranges == 0)

/*  IS_SIMPLE(sym,type) -- check that symbol indicates a scalar of given type */
#define IS_SIMPLE(sym,type) (IS_SCALAR(sym)&&(sym)->s_type==(type))

/*  NEWLAB -- allocate a new label, given that curr_label is declared  */
#define NEWLAB (curr_label++)

#define boom(x) kboom(x,__FILE__,__LINE__) /* diagnose compiler error */

#ifndef NDEBUG
#define assert(ex) {if(!(ex))assfail(__FILE__,__LINE__);}
#else /*NDEBUG*/
#define assert(ex)
#endif /*NDEBUG*/




/*************************  Enumerated Types  *************************/

/*  Enumerated types with names of the form "e_xxxxx" are processed by awk.
 *  Do not change the layout or this may break.  */



/*  Predef -- predefined functions.  actual list is read from predefs.h.  */

#ifdef __STDC__
#define premac(name,type) PRE_ ## name,
#else
#define premac(name,type) PRE_/**/name,
#endif

typedef enum {
#include "predefs.h"
#undef premac
} Predef;



/*  SRfile -- predefined files.  numbering agrees with runtime definitions but
 *  this is not enforced mechanically and probably should be.  */

typedef enum { F_STDIN, F_STDOUT, F_STDERR } SRfile;



/*  Where -- destination type for put_value()  */

typedef enum { TOS, TOS_IND } Where;



/*  Alloc_where -- used with descriptors  */

typedef enum {Use_fp_offset, Use_mem_alloc, Use_stack} Alloc_where;



/*  Result -- result of an operation  */
typedef enum {NO_RESULT, VAL_RESULT, ADDR_RESULT} Result;



/*  Token -- lexical token values and tree node types.  From tokens.h.  */

#ifdef __STDC__
#define tkmac(name,rsv,left,right,prec) TK_ ## name,
#else
#define tkmac(name,rsv,left,right,prec) TK_/**/name,
#endif

typedef enum e_token {
#include "tokens.h"
#undef tkmac
} Token;



/*  Kind -- categories of entries in symbol and literal tables  */

typedef enum e_kind {
	K_NOTAKIND,
	K_BLOCK,
	K_IMPORT,
	K_CONST,
	K_TYPE,
	K_VAR,
	K_FIELD,
	K_OPTYPE,
	K_OP,
	K_SEMAPHORE,
	K_VM,
	K_PARAM,
	K_RESULT,
	K_LITERAL,
	K_PREDEF,
	K_ANON,
	K_FREE,
} Kind;



/*  Type -- basic datassignables  */

typedef enum e_type {
	T_NOTATYPE,
	T_NOTYETATYPE,	/* doing declaration; haven't yet filled in type. */
	T_PREDEF,
	T_GLOBAL,		/* BLOCK types */
	T_SPEC,
	T_INIT,
	T_FINAL,
	T_PROC,
	T_INPUT,
	T_CMD,
	T_BOOL,			/* CONST, TYPE, VAR, PARAM, and LITERAL types */
	T_CHAR,
	T_FILE,
	T_INT,
	T_ENUM,
	T_REC,
	T_UNION,
	T_CAP,
	T_PTR,
	T_ANON,
	T_VOID,			/* OP and OPTYPE types */
	T_FUNC,
	T_STAR,			/* for TK_ARB nodes. */
	T_ANY,			/* for externals only */
	T_NULL,			/* null type */
	T_NOOP,			/* noop type */
	T_STRING,		/* string type */
} Type;


/*  Restrict -- restrictions  */

typedef enum e_restrict {
	R_NOTARESTRICT,
	R_PUBLIC,		/* TYPE restrictions */
	R_PRIVATE,
	R_VAL,			/* PARAM restrictions */
	R_RES,
	R_VALRES,
	R_CALL,			/* OP restrictions */
	R_SEND,
	R_CALLSEND,
} Restrict;



/* Segment -- segment indicators */

typedef enum e_segment {
	S_NOTASEGMENT,
	S_RESOURCE,	/* declared outside any proc, initial, or final */
	S_PROC,		/* declared inside a proc, initial, or final */
	S_INPUT,	/* in-statement parameter */
	S_REC,		/* indicates that this is a field in a record */
} Segment;



/*  Impl -- implementation methods.  note: a semaphore operation is marked
 *  as IM_INPUT until it is determined to be a semaphore operation;
 *  i.e., at the end of a resource body and before code generation.
 */
typedef enum {
	IM_NOTYETIMPL,
	IM_PROC,
	IM_PROCESS,
	IM_INPUT,
	IM_SEMAPHORE,
	IM_EXTERNAL,
} Impl;





/* Icode -- intermediate code type.  */
typedef enum {
	I_NO_TYPE,
	I_COMPONENT, 
	I_COMPONENT_END,
	I_PROC,
	I_PROC_END,
	I_INIT,
	I_INIT_END,
	I_FINAL,
	I_FINAL_END,
	I_BLOCK,
	I_BLOCK_END,
	I_DO_DECLARE,
	I_IMPORT,
	I_EXPR,
	I_LABEL,
	I_LOOPTOP,
	I_BRANCH,
	I_BRANCH_TRUE,
	I_BRANCH_FALSE,
	I_JT,
	I_JT_LAB,
	I_JT_END,
	I_CALL,
	I_REPLY,
	I_CO_START,
	I_CO_ARM,
	I_CO_PPC_END,
	I_CO_WAIT,
	I_CO_END,
	I_INP_ALL,
	I_INP_ALL_END,
	I_INP_START,
	I_INP_ACCESS,
	I_INP_ELSEACCESS,
	I_INP_GET_INV,
	I_INP_GET_NAMED_INV,
	I_INP_GET_NAMED_INV_NB,
	I_INP_RECEIVE,
	I_INP_SEM_P,
	I_INP_MATCH,
	I_INP_ELSEMATCH,
	I_INP_REMOVE,
	I_INP_DONE,
	I_INP_END,
	I_INP_SET_MIN,
	I_INP_UPDATE_MIN,
	I_INP_REMOVE_MIN,
	I_STOP,
	I_PROCESS_SEND,		/* marker to change ic list. */
	I_PROCESS_SEND_END,	/* "			     */
} Icode;





/*************************  Structure Definitions  *************************/



/*  String -- hold string constants */

typedef struct str_struct {
    int str_len;
    char str_data[1];
} Str;

typedef Str *Strptr;



/*  Symbol -- holds symbol and literal table entries  */

typedef struct symbol {		/* field meanings (examples in parens) */
	char *s_name;		/* variable name */
	Kind s_kind;		/* entry class (K_VAR, K_FIELD) */
	Type s_type;		/* underlying datassignable (T_CHAR, T_INT) */
	struct symbol *s_tdef;	/* additional info e.g. fields of a record */
	Restrict s_restrict;	/* restrictions (R_VAL, R_PRIVATE, R_SEND) */
	Bool s_used;		/* is symbol used? i.e., as lvalue|rvalue */
	Restrict s_invoked;	/* how op is invoked */
	struct node *s_value;	/* value of a constant; other uses?? */
	struct range *s_ranges;	/* array bounds struct; NULL if simple var */
	Segment s_segment;	/* where stored (S_RESOURCE, S_PROC) */
	int s_done;		/* to control recursion in s.t. walking */
	int s_offset;		/* offset from FP or block, or OFF_UNK */
	int s_size;		/* object size, or 0 if unknown */
	int s_dessize;		/* descriptor size, or 0 if none */
	int s_desoff;		/* descriptor offset, or OFF_UNK if none */
	Impl s_impl;		/* how implemented (IM_PROC, IM_INPUT) */
	Bool s_reply; 		/* true if operation (in|proc) has replies */
	struct class_st *s_class;/*for operations, pointer to class */
	struct symbol *s_cl_next;/*for operations, pointer to next member */
	Predef s_predef;	/* if predef func, which one? (PRE_length) */
	Token s_how_imported;	/* how imported (TK_EXTEND, TK_IMPORT) */
	struct symbol *s_next;	/* forward link */
	struct symbol *s_prev;	/* backward link */
} Symbol;

typedef Symbol *Symptr;

#define	NULLSYM	((Symptr) 0)
#define SIZE_UNK -1
#define SIZE_ARB -2



/*  Range -- both dimensions of an array.  */

typedef struct range {
	struct node *r_dim1, *r_dim2;
	struct range *r_next;	/* used only for keeping free list. */
} Range;

typedef Range *Rangeptr;



/*  Node -- holds values for a node in the expression tree */

typedef struct node {
	Token e_op;			/* operator or node type (TK_xxx)*/
	union e_lu {			/* left pointer or value */
	    struct node *e_node;		/* left child node */
	    Symptr e_sym;			/* symbol struct */
	    int e_int;				/* integer value */
	    SRfile e_file;			/* standard file number */
	    Strptr e_char;			/* character pointer */
	    struct class_st *e_class;		/* class structure */
	} e_left;
	struct node *e_r;		/* right pointer */
	Symptr e_sig;			/* signature of node */
} Node;

typedef Node *Nodeptr;

#define	NULLNODE ((Nodeptr) 0)

/* shorthand field references */
#define e_l	e_left.e_node
#define e_s	e_left.e_sym
#define e_i	e_left.e_int
#define e_str	e_left.e_char



/*  Local_block -- local blocks after popping from main symbol table  */

typedef struct local_block {	
	int l_nest;	
	Symptr l_st;
	struct local_block *l_next;
} Local_block;

typedef Local_block *Local_blockptr; 



/*  Class -- equivalence class stuff.
 *
 *  symtab entry of each member in a class points back to this class header.
 *  cl_members is useful for combining or for considering certain optimizations.
 */

typedef struct class_st {
	int cl_refcnt;			/* reference count */
	int cl_offset;			/* offset (used for runtime address) */
	Segment cl_segment;		/* segment (used for runtime address) */
	Symptr cl_first, cl_last;	/* first and last members in symtab */
	int cl_members;			/* member count */
	struct class_st *cl_next;	/* forward link */
	struct class_st *cl_prev;	/* backward link */
} Class;

typedef Class *Classptr; 

#define NULLCLASS ((Classptr) 0)




/*  Inst -- intermediate code instruction.  */

typedef struct inst {
	int i_line;
	Icode i_type;
	struct node *i_exp;
	int i_lab;
	struct inst *i_next, *i_back;
} Inst;

typedef Inst *Instptr;

#define NULLINST ((Instptr) 0)

/* Instlist -- list of Inst's, with header node and tail pointer. */
typedef struct {
	Inst il_head;
	Instptr il_tail;
} Instlist;

typedef Instlist *Instlistptr;

#define NULLINSTLISTPTR ((Instlistptr) 0)





/*  In -- one for each input statement in the resource.  */

typedef struct in_st {
	Instptr in_start;	/* where does this input statement belong? */
	int in_return_label;	/* return label within this input statement */
	int in_old_exit_label;	/* exit label for surrounding block */
	int in_old_next_label;	/* next label for surrounding block */
	int in_exit_label;	/* exit label for this input statement */
	int in_next_label;	/* next label for this input statement */
	Classptr in_class;	/* class for this input statement */
	struct leg_st *in_legs;	/* the legs */
	struct in_st *in_next;	/* next input statement structure */
	Bool else_present;	/* flags presence of an else leg */
	struct leg_st *else_leg;/* else leg */
} In;

typedef In *Inptr;

#define NULLIN ((Inptr) 0)




/*  Leg -- one for each leg of an input statement.  */
 
typedef struct leg_st {
	Symptr leg_sym;		/* pointer to operation in symbol table */
	Nodeptr leg_op;		/* node representing the operation */
	Instlist leg_quant;	/* intermediate code for quantifier */
	Instptr leg_quant_guts;	/* pointer to code for quantifier guts */
	Nodeptr leg_synch;	/* synchronization expression node */
	Nodeptr leg_sched;	/* scheduling expression node */
	Instlist leg_block;	/* intermediate code for T_INPUT block */
	Instlist leg_body;	/* intermediate code for body */
	Bool dup_op;		/* flags multiple occurrences of op in in */
	struct leg_st *leg_next;/* next leg */
} Leg;

typedef Leg *Legptr;

#define NULLLEG ((Legptr) 0)




/* assign_descriptor */

struct assign_descriptor {
    Nodeptr source;		/* expression tree of source of assignment */
    Nodeptr destination;	/* expression tree of destination */
    int temp_space;		/* offset of temp holding address of temp space
				   into which source expression is evaluated */
    Bool contig;		/* is this piece of memory contiguous? */
    Bool fixed_total;		/* is the total size fixed at compile time? */
    int total;			/* total size if fixed, offset of local */
    /* fields below here are valid iff contig is true */
    Bool fixed_slice_size;	/* do we know size at compile time */
    int slice_size;		/* slice size if fixed, else offset of local */
    Bool fixed_rows;		/* do we know how many rows at compile time? */
    int num_rows;		/* row count if fixed, else offset of local */
    Bool fixed_row_length;	/* is row length known at compile time? */
    int row_length;		/* row length if fixed, else offset of local */
    struct assign_descriptor *next;
};




/* struct to hold the state of the scanner */
/* unfortunately, this depends on the code lex generates. */

typedef struct scan_str {
    int put_back;
    YYCHAR *yysptr;
    YYCHAR yysbuf;
    FILE *yyin;
    int lineno,
    	line_number,
    	prev_number;
    Token tok;
} Scan;

typedef Scan *Scanptr;
