/* routines that deal with deciding whether operation are semaphore operation.
 */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"

static void sem_rest();
static Bool decl_in_res_body(), no_else_legs(), no_ss_exprs();

/*
 * the criteria for an operation being a semaphore operation are:
 *
 *   -- operation is in a class by itself and if an array
 *	has only one element
 * 	so don't need to select from pending invocations.
 *
 *   -- operation has no parameters and no result
 *	so RTS can represent by a counter instead of queue of invocations.
 *
 *   -- operation is an input operation.
 *
 *   -- operation is invoked only by send.
 *      (formerly, op had to be declared with send restrictor;
 *       now we check how it is actually invoked.)
 *
 *   -- no else leg in any of input statement that services the operation.
 *
 *   -- no synchronization and no scheduling expression in any of its usages.
 *	otherwise P operation would be conditional.
 *
 *   -- not exported (i.e., not declared in the spec).
 *	otherwise the RTS would need to support remote V operations.
 *	(this could be relaxed, but we expect it will cover common usages.)
 *
 *   -- not assigned to a capability variable (and not passed as a parameter).
 *	otherwise, would invocations would need to determine what kind of
 *	operation (regular or semaphore) is being invoked.
 *	although that is do-able, it is an unnecessary complication,
 *	especially because we don't expect this kind of usage to be useful.
 *
 *   -- declared at the resource level.
 *	semaphore local to a process don't seem to have much utility.
 *	furthermore, because we don't allow assignment to capabilities,
 *	such a semaphore would be even more useless.
 *	this also simplifies code generation because all local operations
 *	are treated the same;
 *	otherwise, would need to worry about destroying local semaphores.
 */



/* sem_analysis()
 * called after classes have been determined.
 * decides whether an (input) operation is a semaphore operation
 * and, if so, mark it as K_SEMAPHORE.
 */
void
sem_analysis()
{
	Classptr cl;

	/* run through all classes.
	 * for those that have only one member,
	 * see whether the class's operation meets the rest of the criteria.
	 */
		for (cl = classes; cl != NULLCLASS; cl = cl->cl_next) {
			assert (cl->cl_first != NULLSYM);

			/* operation is already an input operation by
			 * how classes work.
			 */
				assert (cl->cl_first->s_impl == IM_INPUT);

			if (input_singular(cl->cl_first)) {
				sem_rest(cl->cl_first);
			}
		}
}


/* sem_rest(sym)
 * given that sym is in a class by itself,
 * check the rest of the criteria.
 * sets sym->s_impl to IM_SEMAPHORE if it does.
 * the strategy is first to check the simplest criteria that would most
 * likely eliminate the operation from further consideration,
 * and then do the expensive ones only if necessary.
 */
static void
sem_rest(sym)
Symptr sym;
{
	/* invoked only by send
	 * or not invoked at all
	 * (the latter represents an important optimization:
	 *   a process that is going to block (forever)
	 *   on an operation that no one invokes does so quickly ;-)
	 */
		if    (sym->s_invoked != R_SEND
		    && sym->s_invoked != R_NOTARESTRICT)
			return;

	/* no parameters and no result. */
		if (sym->s_tdef != NULLSYM)
			return;

	/* not assigned to capability. */
		if (sym->s_used)
			return;

	/* declared at resource level but not exported. */
		if (! decl_in_res_body(sym))
			return;

	/* no else legs in any in statement. */
		if (! no_else_legs(sym))
			return;

	/* no synch and no sched expressions in any of its usages. */
		if (! no_ss_exprs(sym))
			return;

	sym->s_impl = IM_SEMAPHORE;
	sym->s_kind = K_SEMAPHORE;
}

/* decl_in_res_body(sym)
 * returns TRUE iff sym is declared at the resource level but not exported.
 * strategy is to search backwards in the symbol table until hit
 * a NULLSYM or a K_BLOCK.
 * if hit NULLSYM, then sym is in auxiliary table, so it was declared in spec.
 * if hit K_BLOCK, then sym is either in resource body or local to some block.
 * note: can't use s_segment == S_RESOURCE because that is only set at the
 * beginning of code generation.
 */
static Bool
decl_in_res_body(sym)
Symptr sym;
{
	assert (sym != NULLSYM && sym->s_prev != NULLSYM);

	for (; sym != NULLSYM && sym->s_kind != K_BLOCK; sym = sym->s_prev){
		/* do nothing, and do it quickly please! */
	}

	if (sym == NULLSYM)
		return (FALSE);
	else
		return ((Bool) (sym->s_type == T_SPEC));
}

/*  check for else legs in any in statement 
 *
 */

static Bool
no_else_legs(sym)
Symptr sym;
{
	Inptr in;

	assert (sym != NULLSYM);

	for (in = inlist(); in != NULLIN; in = in->in_next)
		if (in->in_legs->leg_sym == sym)
			if (in->else_present)
				return (FALSE);
	return (TRUE);
}



/* no_ss_exprs(sym)
 * returns TRUE iff all input statement containing sym
 * have no synchronization and no scheduling expressions.
 * strategy is to search through all input statements for this operation;
 * brute force, yes.
 * need to be careful about (yes, bizarre) input statements like:
 *
 *	in x() ->
 *	[] x() & z = 5 ->
 *	ni
 *
 * so we check if first operation in the input statement is the one we want.
 * if not, then we needn't look further because only 1 op in class for sym.
 * if so, then we check all legs.
 * note: instead of doing this checking repeatedly for each input statement,
 * it might be faster to make a one pass of all input statements and
 * mark them if any of their legs have synchronization or scheduling
 * expressions.
 * alternatively, this pass could be eliminated
 * if this info were kept as the input statement structure is built.
 */
static Bool
no_ss_exprs(sym)
Symptr sym;
{
	Inptr in;
	Legptr leg;

	assert (sym != NULLSYM);

	for (in = inlist(); in != NULLIN; in = in->in_next)
		if (in->in_legs->leg_sym == sym)
			for (leg = in->in_legs;
			     leg != NULLLEG;
			     leg = leg->leg_next)
				if (leg->leg_synch != NULLNODE
				    || leg->leg_sched != NULLNODE)
					return (FALSE);
	return (TRUE);
}
