#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"
#include "../srsys.h"

#define INFINITY 1000000


static int post[MAX_NEST*2];
static int posp = -1;

static int stoff(), get_resvar_offset(), varoffset(), get_param_offset();
static void do_do(), do_op(), do_optype(), do_block(), do_type();
static void do_var(), do_param(), assign_size(), do_ranges(), do_semaphore();
static void assign_spec(), op_size(), new_param_list(), end_param_list();
static void new_proc(), new_block(), end_block(), showoff();

static void reorder();


/* assign sizes and offsets to one resource. */
void
assign_offset()
{
    Local_blockptr lb;
    Symptr cur_blk;
    int nest = -INFINITY;
    int i;
     
    /* This is a kluge */
    reorder();
    
    posp = -1;
    segment = S_RESOURCE;
    
    assert(comp_sym);
    assert(comp_sym->s_kind == K_BLOCK && comp_sym->s_type == T_SPEC);
    assign_spec(comp_sym);
    
    for (lb=lb_top; lb; lb=lb->l_next) {
	cur_blk = lb->l_st;
	assert(cur_blk->s_kind == K_BLOCK);
	if (cur_blk->s_type == T_PREDEF) continue;
	
	for (i=lb->l_nest; i <= nest; ++i) {
	    end_block();
	    end_param_list();
	}
	new_block();
	nest = lb->l_nest;
	
	do_do(cur_blk);
    }
}

/* What we doing here is moving T_CMD blocks that occur at the resource
   level (outside any process) into INITIAL.  This is klugey, but it
   allows the code generator to produce, on the fly,  declarations
   for variables that will be needed in INITIAL, as long as they are
   inside a CMD block.
   We do this in two steps:
       1) Run through the lb list to find the initial block
       2) Run through the lb list again, looking for CMD blocks at nesting
          2.  Move them to after the INITIAL, and increase the nesting so
	      they appear to be inside.
   This will break if we ever allow code other than declarations at the top
   level of a resource.
   The only place such a T_CMD block should occur is
   for quantifier variables for process, e.g., process p(i:=1 to 2).
*/	  
static void
reorder()
{
    Local_blockptr lb, init=0, prev, next;
    
    for (lb=lb_top; lb; lb=lb->l_next) {
	if (lb->l_st->s_type == T_INIT)
	    init = lb;
    }
    assert(init);
    
    for (prev = 0, lb=lb_top;
	 lb;
	 prev = lb, lb=next) {
	next = lb->l_next;
	if (lb->l_nest == 2	/* at outer resource level */
	    && lb->l_st->s_type == T_CMD ) {
	    
	    /* detach lb from list */
	    assert(prev);
	    prev->l_next = lb->l_next;

	    /* put it after init */
	    lb->l_next = init->l_next;
	    init->l_next = lb;

	    /* fix the nesting level */
	    lb->l_nest = init->l_nest + 1;
	}
    }
}


/* do_something() to the symbol table beginning at *start*,
   depending on sym->s_kind */
static void
do_do(start)
Symptr start;
{
    char err[80];
    Symptr sym;
    
    for (sym=start; sym; sym=sym->s_next) {
	sym->s_segment = segment;
	switch (sym->s_kind) {
	case K_BLOCK:
	    do_block(sym);
	    break;
	case K_IMPORT:
	    assign_spec(sym);
	    break;
	case K_CONST:
	    do_var(sym);
	    break;
	case K_TYPE:
	    do_type(sym);
	    break;
	case K_VAR:
	    do_var(sym);
	    break;
	case K_PARAM:
	case K_RESULT:
	    do_param(sym);
	    break;
	case K_FIELD:
	    boom("K_FIELD at top level of symbol table");
	    break;
	case K_LITERAL:
	    break;
	case K_OPTYPE:
	    do_optype(sym->s_tdef->s_tdef);
	    break;
	case K_OP:
	    do_op(sym);
	    break;
	case K_SEMAPHORE:
	    do_semaphore(sym);
	    break;
	case K_ANON:
	    boom("K_ANON at top level of symbol table");
	    break;
	case K_FREE:
	    boom("K_FREE in symbol table");
	    break;
	case K_PREDEF:
	    break;
	default:
	    sprintf(err, "bad kind in symbol table: %d",
		    (int)sym->s_kind);
	    WARN(err);
	    break;
	}
    }
}



static void
do_op(sym)
Symptr sym;
{
    /* assign sizes/offsets to parameters */
	do_optype(sym->s_tdef);
     
    if (sym->s_impl == IM_INPUT && sym->s_segment != S_RESOURCE) {
	/* this must be a REALLY local op: not in any spec and declared
	   inside a proc  */
	sym->s_size = OP_CAP_SIZE;
	if (sym->s_ranges)
	    do_ranges(sym);
	/* assign offset if possible */
	if (sym->s_size != SIZE_UNK) {
	    sym->s_offset = stoff(sym->s_size);
	} else {
	    sym->s_offset = OFF_UNK;
	}
    }
    /* assign desoff if necessary */
    if (sym->s_dessize)
	sym->s_desoff = varoffset(sym->s_dessize);
    else
	sym->s_desoff = OFF_UNK;
	
    /* assign offset to class if necessary */
    if (sym->s_impl == IM_INPUT && sym->s_class->cl_refcnt == -1) {
	Classptr pcl = sym->s_class;
	pcl->cl_refcnt = 0;
	pcl->cl_segment = segment;
	pcl->cl_offset = varoffset(align(CL_SIZE));
    } 
}

static void
do_optype(sym)
Symptr sym;
{
    register Symptr formal;
    
    if (!sym || sym->s_done)
	return;
    sym->s_done = 1;
    
    if (!sym || !sym->s_next)	/* no parameters */
	return;
    new_param_list(INVOCATION_HEADER_SIZE);
    for (formal=sym->s_next; formal; formal=formal->s_next) {
	assert(formal->s_kind == K_PARAM || formal->s_kind == K_RESULT);
	do_param(formal);
    }
    end_param_list();
}


/* do bookkeeping necessary for blocks. (sym may have the wrong segment
   when it enters this routine -- the segment might change as a result of
   the block.)
*/
static void
do_block(sym)
Symptr sym;
{
    switch (sym->s_type) {
    case T_GLOBAL:
	new_param_list(0);
	break;
    case T_SPEC:
	assert(sym == comp_sym);
	assert(sym->s_done);
	new_param_list(0);
	break;
    case T_INIT:
    case T_FINAL:
    case T_PROC:
	segment = S_PROC;
	new_proc();
	new_param_list(INVOCATION_HEADER_SIZE);
	break;
    case T_INPUT:
	segment = S_INPUT;
	new_param_list(INVOCATION_HEADER_SIZE);
	break;
    case T_CMD:
	segment = S_PROC;
	new_param_list(INVOCATION_HEADER_SIZE);
	break;
    default:
	boom("funny type in do_block");
    }
    sym->s_segment = segment;
}


static void
do_type(sym)
Symptr sym;
{
    Symptr s;
    int ss, roff;
	    
    if (sym->s_done)
	return;
    sym->s_done = 1;
    
    if (sym->s_ranges)
	boom("ranges in do_type");
	
    sym->s_segment = segment;
    switch (sym->s_type) {
    case T_ANY:
	sym->s_size = SIZE_ARB;		/* should never be used */
	break;
    case T_BOOL:
	sym->s_size = BOOLSIZE;
	break;
    case T_CHAR:
	sym->s_size = CHARSIZE;
	break;
    case T_FILE:
	sym->s_size = FILESIZE;
	break;
    case T_INT:
	sym->s_size = INTSIZE;
	break;
    case T_ENUM:
	sym->s_size = ENUMSIZE;
	break;
    case T_REC:
    case T_STRING:
	if (! sym->s_tdef)
	    boom("T_REC with no s_tdef");
	roff = 0;
	sym->s_size = 0;
	for (s=sym->s_tdef->s_next; s; s = s->s_next) {
	    s->s_segment = segment;
	    assign_size(s);
	    ss = s->s_size;
	    if (ss != SIZE_UNK) {
		if (ss==1) {
		    s->s_offset = roff;
		    roff++;
		} else if (ss==2) {
		    if (roff%2) roff++;
		    s->s_offset = roff;
		    roff += 2;
		} else {
		    roff = (roff+3) & ~0x3;
		    s->s_offset = roff;
		    roff += ss;
		}
	    } else {
		s->s_offset = OFF_UNK;
		sym->s_size = SIZE_UNK;
	    }
		    
	    if (s->s_dessize) {
		s->s_desoff = varoffset(s->s_dessize);
	    } else {
		s->s_desoff = OFF_UNK;
	    }
		
	}
	roff = (roff+3) & ~0x3;
	if (sym->s_size != SIZE_UNK)
	    sym->s_size = roff;
	break;
    case T_CAP:
	{
	   if (!sym->s_tdef)
	       boom("T_CAP without an s_tdef");
	   switch (sym->s_tdef->s_kind) {
	   case K_OP:
	       sym->s_size = OP_CAP_SIZE;
	       if (sym->s_tdef->s_tdef) {
		   new_param_list(INVOCATION_HEADER_SIZE);
		   for (s=sym->s_tdef->s_tdef->s_next; s; s=s->s_next) {
		       switch (s->s_kind) {
		       case K_PARAM:
		       case K_RESULT:
			   do_param(s);
			   break;
		       default:
			   boom("bad kind in parameter list");
		       }
		   }
		   end_param_list();
	       }
	       break;
	   case K_IMPORT:
	   case K_BLOCK:
	       switch (sym->s_tdef->s_type) {
	       case T_SPEC:
	       case T_GLOBAL:
		   assert(sym->s_tdef->s_size != SIZE_UNK);
		   sym->s_size =sym->s_tdef->s_size;
		   break;
	       default:
		   boom("bad capability");
		   break;
	       }
	       break;
	   case K_VM:
		assert (sym->s_tdef->s_type == T_SPEC);
		sym->s_size = VMSIZE;
		break;
	   default:
	       boom("bad kind in T_CAP:");
	   }
	} break;
    case T_PTR:
	sym->s_size = PTRSIZE;
	assign_size(sym->s_tdef);
	break;
    default:
	boom("bad type in do_type");
    }
    
    /* if the size is not known, allocate space for a descriptor*/
    if (sym->s_size == SIZE_UNK) {
	sym->s_dessize = 2*INTSIZE;
	sym->s_desoff = varoffset(2*INTSIZE);
    }
}

/* assign size and offset to a variable */
static void
do_var(sym)
Symptr sym;
{
    /* RON */
    if (sym->s_done)
	return;
    sym->s_done = 1;
    
    sym->s_segment = segment;
    /* fill in size, dessize, desoff */
	assign_size(sym);
    /* fill in s_offset */
	if (sym->s_size != SIZE_UNK) {
	    sym->s_offset = varoffset(sym->s_size);
	} else {
	    sym->s_offset = OFF_UNK;
	}
    /* assign s_desoff */
	if (sym->s_dessize) {
	    sym->s_desoff = varoffset(sym->s_dessize);
	} else {
	    sym->s_desoff = OFF_UNK;
	}

    /* show results */
	if (dbflags['O'])
	    showoff(sym);
}


static void
do_param(sym)
Symptr sym;
{
    /* RON */
    if (sym->s_done)
	return;
    sym->s_done = 1;
    
    sym->s_segment = segment;
    assign_size(sym);
    if (sym->s_size != SIZE_UNK) {
	sym->s_offset = get_param_offset(align(sym->s_size));
    } else {
	sym->s_offset = OFF_UNK;
    }
    
    if (sym->s_type == T_STRING && sym->s_size == SIZE_UNK) {
	sym->s_dessize += INTSIZE;
	sym->s_desoff = get_param_offset(align(sym->s_dessize)) + INTSIZE;
    } else if (sym->s_dessize != 0) {
	sym->s_desoff = get_param_offset(align(sym->s_dessize));
    } else {
	sym->s_desoff = OFF_UNK;
    }

    if (dbflags['O'])
	showoff(sym);
}


static void
assign_size(sym)
Symptr sym;
{
    /* fill in s_size field */
	switch (sym->s_type) {
	case T_NULL:
	case T_NOOP:
	    break;
	case T_BOOL:
	    sym->s_size = BOOLSIZE;
	    break;
	case T_CHAR:
	    sym->s_size = CHARSIZE;
	    break;
	case T_FILE:
	    sym->s_size = FILESIZE;
	    break;
	case T_INT:
	    sym->s_size = INTSIZE;
	    break;
	case T_ENUM:
	    sym->s_size = ENUMSIZE;
	    break;
	case T_PTR:
	    sym->s_size = PTRSIZE;
	    do_type(sym->s_tdef);
	    break;
	case T_ANY:
	    sym->s_size = SIZE_ARB;	/* should never be used */
	    break;
	case T_REC:
	case T_STRING:
	    if (!sym->s_tdef)
		boom("T_REC without s_tdef field in do_var");
	    do_type(sym->s_tdef);
	    sym->s_size = sym->s_tdef->s_size;
	    break;
	case T_CAP:
	    if (!sym->s_tdef)
		boom("T_CAP without s_tdef field in do_var");
	    do_type(sym->s_tdef);
	    sym->s_size = sym->s_tdef->s_size;
	    break;
	default:
	    boom("funny type in assign_size");
	    break;
	}
    
    /* take care of ranges */
	if (sym->s_ranges)
	    do_ranges(sym);
    /* if size is unknown, but it has no (top-level) ranges */
	if (sym->s_size == SIZE_UNK && sym->s_dessize == 0)
	    sym->s_dessize = 2 * INTSIZE;
    
}

static void
do_ranges(sym)
Symptr sym;
{
    int ub, lb;
    Nodeptr dim1=sym->s_ranges->r_dim1,
    	    dim2=sym->s_ranges->r_dim2;
    
    sym->s_dessize = dim2 ? 6*INTSIZE : 4*INTSIZE;
    if (sym->s_size == SIZE_UNK)
	return;
    if (is_constant(dim1->e_r, &ub) && is_constant(dim1->e_l, &lb)) {
	sym->s_size *= ub-lb+1;
	if (sym->s_size < 0) sym->s_size = 0;
	if (dim2) {
	    if (is_constant(dim2->e_r, &ub) && is_constant(dim2->e_l, &lb)) {
		sym->s_size *= ub-lb+1;
		if (sym->s_size < 0) sym->s_size = 0;
	    } else
		sym->s_size = SIZE_UNK;
	}
    } else
	sym->s_size = SIZE_UNK;
}


static void
do_semaphore(sym)
Symptr sym;
{
    assert(sym->s_segment == S_RESOURCE);
    sym->s_size = SEMSIZE;
    sym->s_size =SEMSIZE;
    sym->s_offset = varoffset(SEMSIZE);
    sym->s_dessize = 0;
    sym->s_desoff = OFF_UNK;
}


/* assign sizes and offsets to everything in *spec*. */   
static void
assign_spec(spec)
Symptr spec;
{
    Symptr sym;
    int op_offset = RES_CAP_SIZE;
    
    if (spec->s_done)
	return;
    spec->s_done = 1;
    
    /* initialize things... */
    spec->s_segment = segment;
    
    /* assign sizes and offsets to exported ops */
	for (sym=spec->s_tdef->s_next; sym; sym=sym->s_next) {
	    if (sym->s_kind == K_OP) {
		op_size(sym);
		if (sym->s_size == SIZE_UNK)
		    boom("exported symbol with unknown size");
		sym->s_offset = op_offset;
		op_offset += sym->s_size;
	    }
	}
    spec->s_size = spec->s_tdef->s_size = op_offset;
    
    /* now do unexported ops, but not for imported components! */
	if (spec == comp_sym) {
	    for (sym=spec->s_next; sym; sym=sym->s_next) {
		if (sym->s_kind == K_OP) {
		    op_size(sym);
		    if (sym->s_size != SIZE_UNK) {
			sym->s_offset = op_offset;
			op_offset += sym->s_size;
		    } else
			sym->s_offset = OFF_UNK;
		    }
		}
	    res_var_size = op_offset;
	}

    /* now another pass to finish up spec and ops */
        new_param_list(CRB_HEADER_SIZE);
        do_do(spec->s_tdef->s_next);
        end_param_list();
    
	if (spec == comp_sym)
            do_do(spec->s_next);
}

static void
op_size(sym)
register Symptr sym;
{
    sym->s_segment = segment;
    sym->s_size = OP_CAP_SIZE;
    if (sym->s_ranges)
	do_ranges(sym);
    sym->s_desoff = OFF_UNK;
}


static int
varoffset(size)
int size;
{
    switch (segment) {
    case S_RESOURCE:
	return get_resvar_offset(align(size));
    case S_PROC:
    case S_INPUT:
	return stoff(align(size));
    case S_NOTASEGMENT:
	boom("S_NOTASEGMENT in varoffset");
	/*NOTREACHED*/
    case S_REC:
	boom("S_REC in varoffset");
	/*NOTREACHED*/
    default:
	boom("funny segment in varoffset");
	/*NOTREACHED*/
    }
}


/************************************/
/**** parameter offset generator ****/
/************************************/

static void
new_param_list(start)
int start;
{
    posp ++;
    if (posp == MAX_NEST) {
	boom("parameter lists nested too deeply");
	posp = 0;
    }
    post[posp] = start;
}

static void
end_param_list()
{
    if (posp < 0)
	boom("end_param_list: not in any param list");
    posp --;
}

static int
get_param_offset(size)
int size;
{
    register int ret;
    if (posp < 0)
	boom("get_param_offset: not in a param list");
    ret = post[posp];
    post[posp] += size;
    return ret;
}


/******************************/
/*** stack offset generator ***/
/******************************/
static int offst[MAX_NEST];
static int offsp = -INFINITY;

static void
new_proc()
{
    offsp = 0;
    offst[0] = 0;
    ++cur_proc;
}

static void
new_block()
{
    offsp ++;
    if (offsp == MAX_NEST) {
	boom("syntactic nesting too deep");
	offsp = 0;
    }
    if (offsp > 0)
	offst[offsp] = offst[offsp-1];
}

static void
end_block()
{
    offsp --;
}

static int
stoff(size)
int size;
{
    if (offsp < 0)
	boom("stoff: not in a proc");
    offst[offsp] += size;
    if (offst[offsp] > max_offset[cur_proc])
	max_offset[cur_proc] = offst[offsp];
    
    return offst[offsp] - size;
}

static int
get_resvar_offset(sz)
int sz;
{
    register int ret = res_var_size;
    res_var_size += sz; 
    return ret;
}



/*  showoff(sym) - print debugging output for sym showing offset, size, etc. */

static void
showoff(sym)
Symptr sym;
{
    printf("%08X   offset ",sym);
    if (sym->s_offset == OFF_UNK)
	printf("UNK");
    else
	printf("%3d",sym->s_offset);
    printf("   size %2d   %s%-8s %s  %-8s %s\n",
	sym->s_size, sym->s_ranges ? "[]" : "  ",
	sym->s_name ? sym->s_name : "[unnamed]",
	segtos(sym->s_segment), kindtos(sym->s_kind), typetos(sym->s_type));
}
