/* node.c -- node creation & destruction, constant folding */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "tokmacs.h"
#include "../util.h"

static Nodeptr alloc_node(), fold();


static Nodeptr free_node;		/* pointer to released nodes */


/* build and return a node with the given op and left and right nodes */
Nodeptr
bnode(op, l, r)
Token op;
Nodeptr l, r;
{
    union e_lu temp;
    temp.e_node = l;
    return make_node(op, temp, r);
}


/* build and return a node representing the given integer constant */
Nodeptr
numnode(n)
int n;
{
    union e_lu temp;
    temp.e_int = n;
    return make_node(TK_NUMBER, temp, NULLNODE);
}


/* build and return an identifier node, given a Symptr */
Nodeptr
idnode(s)
Symptr s;
{
    union e_lu temp;
    temp.e_sym = s;
    return make_node(TK_IDENTIFIER, temp, NULLNODE);
}


/* build and return a node using the symbol field on the left side */
Nodeptr
snode(op,s,n)
Token op;
Symptr s;
Nodeptr n;
{
    union e_lu temp;
    temp.e_sym = s;
    return make_node(op, temp, n);
}



/* build and return a node with the given op and left and right subtrees */
Nodeptr
make_node(op,left,right)
Token op;
union e_lu left;
Nodeptr right;
{
    Nodeptr p;
    int i, j;

    /* hope that nobody else is pointing to these nodes to be freed
       through constant folding/propagation */

    /* propagate only scalar constants for now, later think about vector
       constants and vectorization */

    if (PRECEDENCE(op)	/* if binary opr */
	&& left.e_node->e_sig->s_size == 1 && is_constant(left.e_node, &i) 
	&& right->e_sig->s_size == 1 && is_constant(right, &j)) {
	    
    /* fold constants */

	switch (op) {
	    case TK_EQ:		return fold (TK_BOOLEAN, left, right, (i == j));
	    case TK_NE:		return fold (TK_BOOLEAN, left, right, (i != j));
	    case TK_GT:		return fold (TK_BOOLEAN, left, right, (i > j));
	    case TK_LT:		return fold (TK_BOOLEAN, left, right, (i < j));
	    case TK_GE:		return fold (TK_BOOLEAN, left, right, (i >= j));
	    case TK_LE:		return fold (TK_BOOLEAN, left, right, (i <= j));
	    case TK_XOR:	return fold (TK_NUMBER,  left, right, (i ^ j));
	    case TK_OR:		return fold (TK_NUMBER,  left, right, (i | j));
	    case TK_AND:	return fold (TK_NUMBER,  left, right, (i & j));
	    case TK_LSHIFT:	return fold (TK_NUMBER,  left, right, (i << j));
	    case TK_RSHIFT:	return fold (TK_NUMBER,  left, right, (i >> j));
	    case TK_PLUS:	return fold (TK_NUMBER,  left, right, (i + j));
	    case TK_MINUS:	return fold (TK_NUMBER,  left, right, (i - j));
	    case TK_STAR:	return fold (TK_NUMBER,  left, right, (i * j));
	    case TK_DIV:	return fold (TK_NUMBER,  left, right, (i / j));
	    case TK_MOD:	return fold (TK_NUMBER,  left, right, (i % j));
	}			
    } else if ((op == TK_NOT || op == TK_UMINUS)
	&& left.e_node->e_sig->s_size == 1 && is_constant(left.e_node, &i)) {
	switch (op) {
	    case TK_NOT:
		if (left.e_node->e_sig->s_type == T_BOOL)
		    return fold (TK_NUMBER,  left, right, !i);
		else
		    return fold (TK_NUMBER,  left, right, ~i);
	    case TK_UMINUS:
		    return fold (TK_NUMBER,  left, right, -i);
	}
    }

    p = alloc_node();

    p->e_op = op;
    p->e_left = left;
    p->e_r = right;

    make_sig(p);
    return (p);
}


/* fold integer constants */
static Nodeptr
fold(op,left,right,value)
Token op;
union e_lu left;
Nodeptr right;
int value;
{
    release_node(left.e_node);
    release_node(right);
    left.e_int = value;
    return make_node(op, left, NULLNODE);
}


static Nodeptr
alloc_node()
{
	register Nodeptr p;

	if (free_node) {
	    p = free_node;
	    free_node = p->e_l;
	} else {
	    p = (Nodeptr)alloc(sizeof(struct node));
	}
	return p;
}

/* put e and all descendants on the free list for reuse */
void
release_node(e)
Nodeptr e;
{
	Token t;

	if (e == NULLNODE) 
		return;
	t = e->e_op;
	if (LEFT_CHILD(t) == N_NODE)
		release_node(e->e_l);
	if (RIGHT_CHILD(t) == N_NODE)
		release_node(e->e_r);
	e->e_l = free_node;
	free_node = e;
}




Nodeptr
copy_nodes(a_node)
Nodeptr a_node;
{
	Nodeptr nu_node;
	Token t;

	if (a_node == NULLNODE)
		return NULLNODE;
	nu_node = alloc_node();
	*nu_node = *a_node;
	t = a_node->e_op;
	if (LEFT_CHILD(t) == N_NODE)
		nu_node->e_l = copy_nodes(a_node->e_l);
	if (RIGHT_CHILD(t) == N_NODE)
		nu_node->e_r = copy_nodes(a_node->e_r);
	return nu_node;
}
