/* makesig.c -- make a signature for a new node */

#include <stdio.h>

#include "sr.h"
#include "funcs.h"
#include "globals.h"

static Bool cast_compatible();
static char *argsig_compare();
static Symptr sig_error();
static void binary_sig();

/* the use of the symbol structure to represent a signature differs
    slightly from the use of the symbol structure in other contexts:

	s_size field represents the array size of the object.
	    for objects larger than a single denotation
		SIZE_UNK and SIZE_ARB indicate that somewhere
		beneath them is located an object with the
		an unknown or arbitrary size.  Beneath
		SIZE_ARB nodes, there may exist objects of
		both unknown and arbitrary size.  For all of
		these objects, the s_value field is not used.
	    for objects parsed as a single denotation, the
		s_size and s_value fields are used below:
	    if s_size is SIZE_UNK, then s_value points
		to an expression which can be evaluated at
		run time to obtain the array size
	    if s_size is SIZE_ARB, then before this expr
		tree is handed over to the code generator,
		we should create an expression tree which
		can be evaluated to obtain the size at run
		time.  if this expression tree evaluates
		to a compile-time constant >= 0, then set
		the s_size field to the value of that
		constant and free the expr tree.  if it
		evaluates to a constant < 0, then fatal
		error.  otherwise, the size of this object
		is not known at compile time.  set the s_size
		field to SIZE_UNK.
	s_name - for named objects (identifiers), the name
		of the signature is the same as the name of
		the id.  this allows equivalence by strcmp(),
		ron's favorite method for matching resource
		capabilities and specs.
	s_kind - used somewhat inconsistently.  need to think
		about this some more.  used to be, that this
		was always set to K_TYPE.  then we found that
		occasionally we need to set it to K_OP...
	s_type - the type of this variable (this is really
		what the signature is all about)
	s_tdef - supplementary information about the type,
		meaningful only for certain values of s_type.
*/



/* assume e_sig field is valid for any node built with make_node.
    this field points to a Symbol with s_type set to reflect type
    of object, s_tdef pointing to a user-defined type definition if
    necessary, and s_size reporting the number of elements
*/

void
make_sig(op)
Nodeptr op;
{
    Symptr sym, at;
    Nodeptr left, right, top;
    struct assign_descriptor desc;

    left = op->e_l;
    right = op->e_r;

    switch (op->e_op) {
    case TK_RUNTIMESIZE:
    case TK_PBADDR:
    case TK_NUMBER:
    case TK_LB1:
    case TK_LB2:
    case TK_UB1:
    case TK_UB2:
    case TK_LENGTH:
	op->e_sig = new_sig(T_INT);
	break;
    case TK_CLASS:
    case TK_BLOCK:
	op->e_sig = NULLSYM;		/* no sig needed, nothing goes above */
	break;
    case TK_INPUT:
	if (IS_SCALAR(left->e_sig) && left->e_sig->s_kind == K_OP)
	    op->e_sig = NULLSYM; 	/* nothing goes above */
	else
	    op->e_sig = sig_error("input");
	break;
    case TK_INDEX:
	top = (left->e_op == TK_PERIOD) ? left->e_r : left;
	assert(top->e_op == TK_IDENTIFIER);
	sym = top->e_s;
	
	if (sym->s_ranges) {		/* array slice or index */
	    if (!contiguous(&desc,op))
		ERROR(E_FATAL+4,"non-contiguous slice");
	    fix_subs(sym,right);
	    op->e_sig = new_symbol(K_TYPE);
	    *(op->e_sig) = *(left->e_sig);
	    if (!right->e_l->e_r)	/* if not slice */
		op->e_sig->s_ranges = 0;
	    if (check_bounds(op->e_sig, sym->s_ranges, right) == FALSE) {
		/* free signature? */
		op->e_sig = sig_error("index");
		return;
	    }
	} else if (sym->s_type == T_STRING) {
	    /* the type of a substring is char or char vector */
	    op->e_op = TK_SUBSTR;
	    op->e_sig = new_sig(T_CHAR);
	    if (right->e_l->e_r)	/* substring, not single char */
		add_range(op->e_sig,	/* need range to make it a vector */
		    bnode(TK_RANGE,numnode(1),bnode(TK_ARB,NULLNODE,NULLNODE)),
		    NULLNODE);
		op->e_sig->s_ranges = op->e_sig->s_ranges;  /* just need !=0 */
	    fix_substr(top,right);
	    if (check_bounds(op->e_sig,
 	      sym->s_tdef->s_tdef->s_next->s_next->s_ranges,right) == FALSE) {
		op->e_sig = sig_error("substring index");
		return;
	    }
	} else {
	    FATAL("subscripted object is not an array");
	    op->e_sig = left->e_sig;
	}
	break;
    case TK_NEW:
	sym = new_symbol(K_ANON);
	sym->s_type = op->e_s->s_type;
	sym->s_tdef = op->e_s;
	sym->s_size = op->e_s->s_size;
	op->e_sig = new_symbol(K_TYPE);
	op->e_sig->s_type = T_PTR;
	op->e_sig->s_tdef = sym;
	op->e_sig->s_size = 1;
	op->e_sig->s_restrict = R_NOTARESTRICT;
	break;
    case TK_HAT:
	if (!IS_SIMPLE(left->e_sig,T_PTR)) {
	    op->e_sig = sig_error("indirection");
	    return;
	}
	/* need to make a new signature so as to own the size field */
	op->e_sig = new_symbol(left->e_sig->s_kind);
	op->e_sig->s_type = left->e_sig->s_tdef->s_type;
	op->e_sig->s_tdef = left->e_sig->s_tdef->s_tdef;
	op->e_sig->s_size = left->e_sig->s_size;
	if (op->e_sig->s_tdef != NULLSYM
	&& op->e_sig->s_tdef->s_kind == K_ANON
	&& op->e_sig->s_tdef->s_type == T_PTR)
	    op->e_sig->s_tdef = op->e_sig->s_tdef->s_tdef; /* skip extra entry*/
	if (op->e_sig->s_type == T_ANY)
	    op->e_sig = sig_error("can't dereference an `any' pointer");
	break;
    case TK_CAST:
	if (!right)  {
	    op->e_sig = sig_error(NULLSTR);	/* errs diagnosed elsewhere */
	    return;
	}
	if (op->e_s->s_kind != K_TYPE && op->e_s->s_kind != K_ANON) {
	    op->e_sig = sig_error("very unreasonable cast");
	    return;
	}
	if (! cast_compatible(op, op->e_s, right->e_sig)) {
	    op->e_sig = sig_error("unreasonable cast");
	    return;
	}
	op->e_sig = new_symbol(right->e_sig->s_kind);

	switch (op->e_s->s_type) {
	    case T_VOID:
	    case T_FUNC:
		boom("bad signature");
		/*NOTREACHED*/
	    case T_REC:
	    case T_UNION:
	    case T_SPEC:
		op->e_sig->s_type = op->e_s->s_type;
		op->e_sig->s_tdef = op->e_s;
		break;
	    default:
		op->e_sig->s_type = op->e_s->s_type;
		op->e_sig->s_tdef = op->e_s->s_tdef;
		break;
	}
	op->e_sig->s_restrict = R_NOTARESTRICT;
	op->e_sig->s_size = right->e_sig->s_size;
	op->e_sig->s_value = right->e_sig->s_value;
	break;
    case TK_QUESTION:
	if (IS_SCALAR(left->e_sig) && left->e_sig->s_kind == K_OP)
	    op->e_sig = new_sig(T_INT);
	else
	    op->e_sig = sig_error("question mark -- bad argument");
	break;
    case TK_PERIOD:
	if (left->e_sig->s_type == T_CAP) {
	    Symptr typdef;

	    typdef = left->e_sig->s_tdef;
	    if ((typdef->s_kind != K_BLOCK && typdef->s_kind != K_IMPORT)
		|| typdef->s_type != T_SPEC) {
		op->e_sig = sig_error("period -- bad res cap");
		return;
	    } else if (right->e_sig->s_kind != K_OP
		       && right->e_sig->s_kind != K_OPTYPE
		       || (at_lookup(typdef->s_tdef,
				     right->e_sig->s_name) == NULLSYM)) {
		op->e_sig = sig_error("period -- bad op");
		return;
	    }
	    op->e_sig = right->e_sig;
	} else if (left->e_sig->s_type == T_REC ||
		   left->e_sig->s_type == T_UNION) {
	    if ((right->e_op != TK_IDENTIFIER) ||
		(right->e_s->s_kind != K_FIELD))
		op->e_sig = sig_error("right side of period");
	    else if (at_lookup(left->e_sig->s_tdef->s_tdef,
			       right->e_s->s_name) == NULLSYM)
		op->e_sig = sig_error("period: field not found");
	    else
		op->e_sig = right->e_sig;
	} else
	    op->e_sig = sig_error("period");
	break;
    case TK_ARB:
	op->e_sig = new_sig(T_STAR);
	break;
    case TK_CHRLIT:
	op->e_sig = new_sig(T_CHAR);
	break;
    case TK_STRLIT:
	at = new_symbol(K_TYPE);
	sym = at_install(at, "len", K_FIELD);
	sym->s_type = T_INT;
	sym = at_install(at, "str", K_FIELD);
	sym->s_type = T_CHAR;
	add_range(sym,
	    bnode(TK_RANGE, numnode(1), numnode(op->e_str->str_len)),
	    NULLNODE);
	sym = at_install(at, "pad", K_FIELD);
	sym->s_type = T_CHAR;
	op->e_sig = new_sig(T_STRING);
	op->e_sig->s_tdef = new_sig(T_STRING);
	op->e_sig->s_tdef->s_tdef = at;
	break;
    case TK_BOOLEAN:
	op->e_sig = new_sig(T_BOOL);
	break;
    case TK_FILE_CONST:
	op->e_sig = new_sig(T_FILE);
	break;
    case TK_NOOP:
    case TK_NULL:
	op->e_sig = new_symbol(K_LITERAL);
	op->e_sig->s_type = (op->e_op == TK_NOOP) ? T_NOOP : T_NULL;
	op->e_sig->s_tdef = NULLSYM;
	op->e_sig->s_size = 1;
	op->e_sig->s_restrict = R_NOTARESTRICT;
	break;
    case TK_IMPORTED_CONST:
    case TK_TEMPLATE:
    case TK_IDENTIFIER:
    case TK_FORMAL:
	sym = op->e_s;
	op->e_sig = new_symbol(sym->s_kind);
	op->e_sig->s_name = sym->s_name;
	if (sym->s_type == T_VOID || sym->s_type == T_FUNC) {
	    op->e_sig->s_type = sym->s_type;
	    op->e_sig->s_tdef = sym->s_tdef;
	    /* kind won't be right if this is a capability */
	    op->e_sig->s_restrict = sym->s_restrict;
	    op->e_sig->s_kind = K_OP;
	} else if (sym->s_type == T_REC
		   || sym->s_type == T_STRING
		   || sym->s_type == T_UNION
		   || sym->s_type == T_SPEC
		   || sym->s_type == T_ENUM
		   || sym->s_type == T_GLOBAL) {
	    op->e_sig->s_type = sym->s_type;
	    op->e_sig->s_tdef = sym->s_tdef;
	    op->e_sig->s_restrict = R_NOTARESTRICT;
	} else if (sym->s_tdef) {
	    op->e_sig->s_type = sym->s_tdef->s_type;
	    op->e_sig->s_tdef = sym->s_tdef->s_tdef;
	    if (op->e_sig->s_tdef)
		op->e_sig->s_restrict = (sym->s_type == T_CAP)?
		    sym->s_tdef->s_tdef->s_restrict:
		    sym->s_tdef->s_restrict;
	    else
		op->e_sig->s_restrict = R_NOTARESTRICT;
	} else {
	    op->e_sig->s_type = sym->s_type;
	    op->e_sig->s_tdef = NULLSYM;
	    op->e_sig->s_restrict = R_NOTARESTRICT;
	}
	range_size(sym, op->e_sig, (Bool) (op->e_op == TK_IDENTIFIER));
	break;
    case TK_RANGE:
	if (range_compatible(op, (left)? left->e_sig: NULLSYM,
			     (right)? right->e_sig: NULLSYM) == FALSE)
	    op->e_sig = sig_error("ranges");
	else
	    op->e_sig = left->e_sig;
	break;
    case TK_UMINUS:
	if (IS_SIMPLE(left->e_sig,T_INT))
	    op->e_sig = left->e_sig;
	else
	    op->e_sig = sig_error("unary minus");
	break;
    case TK_NOT:
	if (IS_SCALAR(left->e_sig)
	    && (left->e_sig->s_type == T_INT || left->e_sig->s_type == T_BOOL))
	    op->e_sig = left->e_sig;
	else
	    op->e_sig = sig_error("operand of NOT");
	break;
    case TK_OR:
    case TK_AND:
    case TK_XOR:
	if (IS_SCALAR(left->e_sig) && IS_SCALAR(right->e_sig)) 
	    binary_sig(op);
	else 
	    op->e_sig = sig_error("non-scalar boolean operation");
	break;
    case TK_PLUS:
	if (!IS_SCALAR(left->e_sig) || !IS_SCALAR(right->e_sig)) 
	    op->e_sig = sig_error("non-scalar addition");
	else if (left->e_sig->s_type == T_PTR		/* ptr + int */
		&& left->e_sig->s_tdef->s_type != T_ANY
		&& right->e_sig->s_type == T_INT)
	    op->e_sig = left->e_sig;
	else if (right->e_sig->s_type == T_PTR		/* int + ptr */
		&& right->e_sig->s_tdef->s_type != T_ANY
		&& left->e_sig->s_type == T_INT)
	    op->e_sig = right->e_sig;
	else						/* int + int  (etc.) */
	    binary_sig(op);
	if (left->e_op == TK_ADDR || right->e_op == TK_ADDR)
	    FATAL("can't handle addition of synthesized pointer");
	    /* (because the size field is wrong at code generation time) */
	break;
    case TK_MINUS:
	if (!IS_SCALAR(left->e_sig) || !IS_SCALAR(right->e_sig)) 
	    op->e_sig = sig_error("non-scalar subtraction");
	else if (right->e_sig->s_type == T_PTR		/* ptr - ptr */
		&& left->e_sig->s_type == T_PTR
		&& (left->e_sig->s_tdef->s_type != T_ANY
		    || right->e_sig->s_tdef->s_type != T_ANY))
	    op->e_sig = new_sig(T_INT);
	else if (left->e_sig->s_type == T_PTR		/* ptr - int */
		&& left->e_sig->s_tdef->s_type != T_ANY
		&& right->e_sig->s_type == T_INT)
	    op->e_sig = left->e_sig;
	else
	    binary_sig(op);				/* int - int  (etc.) */
	if (left->e_op == TK_ADDR || right->e_op == TK_ADDR)
	    FATAL("can't handle subtraction of synthesized pointer");
	break;
    case TK_STAR:
    case TK_DIV:
    case TK_MOD:
    case TK_LSHIFT:
    case TK_RSHIFT:
	if (IS_SCALAR(left->e_sig) && IS_SCALAR(right->e_sig))
	    binary_sig(op);
	else
	    op->e_sig = sig_error("non-scalar arithmetic");
	break;
    case TK_ASSIGN:
    case TK_CONST:
	if (check_assign(left, right) == FALSE)
	    op->e_sig = sig_error("assignment");
	break;
    case TK_SWAP:
	if (check_assign(left, right) == FALSE
	    || check_assign(right, left) == FALSE
	    || left->e_sig->s_type != right->e_sig->s_type) /* char:=:string */
	    op->e_sig = sig_error("swap");
	break;
    case TK_CONCAT:
	if (is_string(left->e_sig) && is_string(right->e_sig))
	    op->e_sig = new_sig(T_STRING);
	else
	    op->e_sig = sig_error("concatenation of non-strings");
	break;
    case TK_VECTORIZE:
	if (!left)
	    op->e_sig = right->e_sig;
	else {
	    if (sigcmp(left->e_sig, right->e_sig, left, right, 0) == FALSE)
		op->e_sig = sig_error("vectorization");
	    else {
		op->e_sig = new_symbol(K_TYPE);
		op->e_sig->s_type = right->e_sig->s_type;
		op->e_sig->s_tdef = right->e_sig->s_tdef;
		op->e_sig->s_restrict = right->e_sig->s_restrict;
		if (right->e_sig->s_size == SIZE_ARB &&
		    left->e_sig->s_size == SIZE_ARB)
		    op->e_sig =
			sig_error("too many arbitrarily-sized components");
		else if (right->e_sig->s_size == SIZE_ARB ||
			 left->e_sig->s_size == SIZE_ARB)
		    op->e_sig->s_size = SIZE_ARB;
		else if (right->e_sig->s_size == SIZE_UNK ||
			 left->e_sig->s_size == SIZE_UNK)
		    op->e_sig->s_size = SIZE_UNK;
		else
		    op->e_sig->s_size =
			left->e_sig->s_size + right->e_sig->s_size;
	    }
	}
	break;
    case TK_CLONE:
	if (!IS_SIMPLE(left->e_sig,T_INT))
	    op->e_sig = sig_error("invalid replication factor");
	else {
	    int i;

	    op->e_sig = new_symbol(K_TYPE);
	    op->e_sig->s_type = right->e_sig->s_type;
	    op->e_sig->s_tdef = right->e_sig->s_tdef;
	    op->e_sig->s_restrict = right->e_sig->s_restrict;
	    if (is_constant(left, &i) == FALSE)
		op->e_sig->s_size = SIZE_UNK;
	    else if (right->e_sig->s_size == SIZE_UNK)
		op->e_sig->s_size = SIZE_UNK;
	    else if (right->e_sig->s_size == SIZE_ARB)
		op->e_sig->s_size = SIZE_ARB;
	    else
		op->e_sig->s_size = right->e_sig->s_size * i;
	}
	break;
    case TK_INCREMENT:
    case TK_DECREMENT:
	if (!IS_SCALAR(left->e_sig))
	    op->e_sig = sig_error("non-scalar increment/decrement");
	else if (!IS_ORDERED(left->e_sig->s_type)
		&& (left->e_sig->s_type != T_PTR
		    || left->e_sig->s_tdef->s_type == T_ANY))
	    op->e_sig = sig_error("increment/decrement");
	else
	    op->e_sig = left->e_sig;
	break;
    case TK_EQ:
    case TK_NE:
    case TK_LT:
    case TK_LE:
    case TK_GT:
    case TK_GE:
	if (is_string(left->e_sig) && is_string(right->e_sig))  {
	    op->e_sig = new_sig(T_BOOL);
	    break;
	} else if (!check_size(left, right))
	    op->e_sig = sig_error("boolean expression, bad sizes");
	else if ((op->e_op == TK_EQ || op->e_op == TK_NE)
		&&(left->e_sig->s_type==T_NULL||left->e_sig->s_type==T_NOOP)
		&&(right->e_sig->s_type==T_NULL||right->e_sig->s_type==T_NOOP)){
	    int i;

	    i = (right->e_sig->s_type == left->e_sig->s_type);
	    if (op->e_op == TK_NE)
		i = !i;
	    release_node(right);
	    release_node(left);

	    op->e_op = TK_BOOLEAN;
	    op->e_i = i;
	    op->e_sig = new_sig(T_BOOL);
	} else {
	    if (sigcmp(left->e_sig, right->e_sig, left, right, 0) == FALSE)
		op->e_sig = sig_error("in boolean expression");
	    else if (!IS_SCALAR(left->e_sig) || !IS_SCALAR(right->e_sig))
		op->e_sig = sig_error("non-scalar boolean");
	    else if (!IS_ORDERED(left->e_sig->s_type)
		     && (op->e_op != TK_EQ && op->e_op != TK_NE))
		op->e_sig = sig_error("invalid type for arithmetic compare");
	    else
		op->e_sig = new_sig(T_BOOL);
	}
	break;
    case TK_DESTROY:
	if (left->e_sig->s_type == T_CAP || left->e_sig->s_type == T_NOOP) {
	    Symptr typdef;

	    if (left->e_sig->s_type == T_NOOP) {
		op->e_sig = new_symbol(K_TYPE);
		replace(left, TK_NOOP, TK_FILE_NOOP, NULLSYM);
	    } else if ((typdef = left->e_sig->s_tdef) == NULLSYM
		       || (typdef->s_kind != K_BLOCK
			   && typdef->s_kind != K_IMPORT
			   && typdef->s_kind != K_VM)
		       || typdef->s_type != T_SPEC)
		op->e_sig = sig_error("destroy");
	    else
		op->e_sig = new_symbol(K_TYPE);
	    op->e_sig->s_type = T_VOID;
	    op->e_sig->s_restrict = R_NOTARESTRICT;
	}
	else
	    op->e_sig = sig_error("destroy");
	break;
    case TK_CREATE:
	if (left->e_sig->s_type != T_SPEC || !IS_SCALAR(left->e_sig)) {
	    op->e_sig = sig_error("create");
	    break;
	}
	/*
	 * note: can't do easily in TK_ON because don't have resource
	 * node there.
	 */
	{			/* make sure create arguments are ok. */
	    Symptr res;
	    char *msg;

	    res = left->e_s;
	    assert (res->s_type == T_SPEC);
	    assert (res->s_tdef != NULLSYM);
	    if (msg = argsig_compare(res->s_tdef->s_next,right->e_r)) {
		       /* (s_next skips bogus entry at start of aux table) */
		op->e_sig = sig_error(msg);
	    } else {
		op->e_sig = new_symbol(left->e_sig->s_kind);
		op->e_sig->s_name = left->e_s->s_name;
		op->e_sig->s_type = T_CAP;
		op->e_sig->s_tdef = left->e_sig;
		op->e_sig->s_restrict = left->e_sig->s_restrict;
		op->e_sig->s_size = 1;
	    }
	}
	break;
    case TK_ON:
	if (!left)
	    op->e_sig = NULLSYM;
	else if (left->e_sig->s_type != T_CAP
		|| left->e_sig->s_tdef->s_kind != K_VM)
	    op->e_sig = sig_error("resource location");
	else if (!IS_SCALAR(left->e_sig))
	    op->e_sig = sig_error("non-scalar vmcap");
	else
	    op->e_sig = left->e_sig;
	break;
    case TK_CREVM:
	{
	Symptr res;

	res = op->e_s;
	assert(res->s_kind == K_VM);
	assert(res->s_type == T_SPEC);
	assert(res->s_tdef == NULLSYM);
	op->e_sig = new_symbol(K_VM);
	op->e_sig->s_name = res->s_name;
	op->e_sig->s_type = T_CAP;
	op->e_sig->s_tdef = res;
	op->e_sig->s_restrict = R_NOTARESTRICT;
	op->e_sig->s_size = 1;
	if (right != NULLNODE && !IS_SIMPLE(right->e_sig,T_INT))
	    op->e_sig = sig_error("physical-machine");
	break;
	}
    case TK_CALL:
    case TK_SEND:
    case TK_CO_CALL:
    case TK_CO_SEND:
	if (op->e_op == TK_CALL || op->e_op == TK_CO_CALL) {
	    if (left->e_sig->s_restrict == R_SEND)
		FATAL("can't call a {send} operation");
	} else {
	    if (left->e_sig->s_restrict == R_CALL)
		FATAL("can't send to a {call} operation");
	}
	op->e_sig = new_symbol(K_TYPE);
	*(op->e_sig) = *(left->e_sig);
	if (op->e_sig->s_type == T_CAP)
	    op->e_sig->s_restrict = left->e_sig->s_tdef->s_restrict;
	else
	    op->e_sig->s_restrict = R_NOTARESTRICT;
	/* check for variable or result parameters in a SEND.
	 * note: would be nice to check SEND to value-returning op too,
	 * but can't distinguish here between implicit and explicit calls.
	 */
	if (op->e_op == TK_SEND || op->e_op == TK_CO_SEND)
	    for (sym = op->e_l->e_l->e_sig->s_tdef;
		 sym != NULLSYM;sym = sym->s_next)
		if (sym->s_kind == K_PARAM && sym->s_restrict != R_VAL)
		    WARN(
	"var or res parameter(s) in send invocation\n\twill not be changed");
	break;
    case TK_INVOKE:
	/*  assume tree looks like:
	    TK_INVOKE
	    /     \
	    /       arg_list
	    operation
	    specification
	    */
	if (left->e_sig->s_type != T_FUNC &&
	    left->e_sig->s_type != T_VOID &&
	    !((left->e_sig->s_type == T_CAP
	       && left->e_sig->s_tdef->s_kind == K_OP)
	      || left->e_sig->s_type == T_NULL
	      || left->e_sig->s_type == T_NOOP))
	    op->e_sig = sig_error("invoke");
	else if (left->e_sig->s_size != 1 &&
		 left->e_sig->s_size != SIZE_UNK)
	    op->e_sig = sig_error("size mismatch");
	else {
	    Symptr fsig;
	    char *msg;

	    op->e_sig = new_symbol(K_TYPE);
	    if (left->e_sig->s_type == T_NULL) {
		op->e_sig = sig_error("invoke");
		break;
	    } else if (left->e_sig->s_type == T_NOOP) {
		op->e_sig->s_type = T_VOID;
		replace(left, TK_NOOP, TK_OPCAP_NOOP, NULLSYM);
		op->e_sig->s_restrict = R_CALLSEND;
		break;
	    }
	    fsig = (left->e_sig->s_type == T_CAP)?
		left->e_sig->s_tdef: left->e_sig;

	    assert(fsig->s_restrict == R_CALL
		   || fsig->s_restrict == R_SEND
		   || fsig->s_restrict == R_CALLSEND);
	    op->e_sig->s_restrict = fsig->s_restrict;
	    if (left->e_op == TK_IDENTIFIER && left->e_s->s_kind == K_PREDEF) {
		if (predef_args(op) == FALSE)
		    op->e_sig = sig_error("argument list");
		break;
	    }

	    if (fsig->s_type == T_VOID) {
		op->e_sig->s_size = 1;
		op->e_sig->s_type = T_VOID;
		op->e_sig->s_tdef = NULLSYM;
	    } else {
		Symptr findresult;
		Bool foundit;

		for (findresult = fsig->s_tdef, foundit = FALSE;
		    	!foundit && findresult;) {
		    if (findresult->s_kind == K_RESULT)
			foundit = TRUE;
		    else
			findresult = findresult->s_next;
		}
		if (!foundit)
		    boom("result not found");
		assert(findresult->s_type != T_VOID &&
		       findresult->s_type != T_FUNC);
		if (findresult->s_type == T_REC
		    || findresult->s_type == T_UNION
		    || findresult->s_type == T_SPEC) {
		    op->e_sig->s_type = findresult->s_type;
		    op->e_sig->s_tdef = findresult->s_tdef;
		} else if (findresult->s_tdef) {
		    op->e_sig->s_type = findresult->s_tdef->s_type;
		    op->e_sig->s_tdef = findresult->s_tdef->s_tdef;
		} else {
		    op->e_sig->s_type = findresult->s_type;
		    op->e_sig->s_tdef = NULLSYM;
		}
		range_size(findresult, op->e_sig, FALSE);
	    }
	    /* (s_next skips over bogus entry at start of auxiliary table) */
	    if (msg = argsig_compare((fsig->s_tdef)?
			       fsig->s_tdef->s_next: NULLSYM,
			       right))
		op->e_sig = sig_error(msg);
	}
	break;
    case TK_LIST:
	if (right == NULLNODE)		/* just a parenthesized expr */
	    op->e_sig = left->e_sig;
	else
	    op->e_sig = NULLSYM;
	break;
    case TK_ADDR:
	op->e_sig = new_symbol(left->e_sig->s_kind);
	op->e_sig->s_type = T_PTR;
	op->e_sig->s_tdef = left->e_sig;
	op->e_sig->s_size = 1;
	op->e_sig->s_restrict = left->e_sig->s_restrict;
	break;
    default:
	boom("signature boom");
	/*NOTREACHED*/
    }
}



/*  binary_sig(op) - make signature for a "normal" binary operator  */

static void
binary_sig(op)
Nodeptr op;
{
    Nodeptr left, right;

    left = op->e_l;
    right = op->e_r;
    if (sigcmp(left->e_sig, right->e_sig, left, right, 0) == FALSE)
	op->e_sig = sig_error("binary arithmetic operator");
    else if (!IS_ORDERED(left->e_sig->s_type))
	op->e_sig = sig_error("invalid type for arithmetic");
    else
	op->e_sig = left->e_sig;
}




/*  new_sig(t) - make a new signature of type t  */

Symptr
new_sig(t)
Type t;
{
    Symptr s;
    s = new_symbol(K_TYPE);
    s->s_type = t;
    s->s_size = 1;
    s->s_restrict = R_NOTARESTRICT;
    return s;
}



static Symptr
sig_error(msg)
char *msg;
{
	static Bool error_init = TRUE;
	static Symptr bad_sig;

	if (msg)
	    ERROR(E_FATAL+3,msg);
	if (error_init) {
	    bad_sig = new_symbol(K_NOTAKIND);
	    bad_sig->s_type = T_INT;
	    bad_sig->s_restrict = R_NOTARESTRICT;
	    error_init = FALSE;
	}
	return bad_sig;
}	



/* this routine checks arguments to invocations and to create.
 * returns NULL if okay or error message if not.
 *
 * note that the parameters in symbol table ("template")
 * for resource parameters might not be adjacent if one
 * is an anonymous enumeration type.
 * For example, resource a(x:enum(red, blue, green); y:int)
 * the literal red, blue, green will be between x and y.
 * Although such a parameter makes the resource silly (essentially useless)
 * we'll allow it because it is pretty easy to work around.
 * Note that this doesn't occur for operation parameters because
 * enumeration literals there don't go in the parameter's aux table.
 * Also note that operations and such defined in the spec will
 * appear in the aux table before the parameters.
 */
static char *
argsig_compare(template, arglist)
Symptr template;
Nodeptr arglist;
{
    Symptr rightsig;
    Nodeptr destination;
    char *retval = 0;

    /* skip to the next parameter; NULLSYM or result means end of the list */
    while (template != NULLSYM
	&& template->s_kind != K_PARAM
	&& template->s_kind != K_RESULT)
		template = template->s_next;

    if (template == NULLSYM || template->s_kind == K_RESULT)
	if (arglist)
	    return "too many arguments";
	else
	    return NULL;	/* end of list, and all is okay */

    if (!arglist)
	return "not enough arguments";
    if (template->s_type == T_ANY)
	return argsig_compare(template->s_next, arglist->e_r);

    rightsig = arglist->e_l->e_sig;

    /* so go ahead and be clumsy about it.
     * i.e., it would be nicer if we didn't have to build a signature
     * each time a parameter is checked.
     * how about using some lazy evaluation?
     * i.e., build first time and hang it off the symbol table entry.
     * but in the meantime, let's get even more clumsy,  some day we'll
     * need to come back and clean this up
     */

    destination = snode(TK_TEMPLATE, template, NULLNODE);
    if (!sigcmp(destination->e_sig, rightsig, destination, arglist->e_l, 0))
	retval = "incompatible argument";
    else if (destination->e_sig->s_restrict != rightsig->s_restrict)
	retval = "parameter restriction violation";
    else if (!check_size(destination, arglist->e_l))
	retval = "argument size";
    else {
	/* okay so far, move on to next argument */
	release_node(destination->e_s->s_value);
	retval = argsig_compare(template->s_next, arglist->e_r);
    }
    release_node(destination);
    return retval;
}


/* return true iff type2 can be reasonably cast into type1. */
static Bool
cast_compatible(op, type1, type2)
Nodeptr op;
Symptr type1, type2;
{
    Symptr l;
    Nodeptr r;

    if (type1 == NULLSYM) return FALSE;
    switch (type1->s_type) {
	case T_CHAR:
	case T_INT:
	case T_BOOL:
	case T_ENUM:
	    return IS_ORDERED(type2->s_type);
	case T_PTR:
	    return type2->s_type == T_PTR;
	case T_REC:
	    if (op->e_r->e_op != TK_LIST)
		return FALSE;
	    /* check each field in turn */
	    for (l=op->e_s->s_tdef->s_next, r=op->e_r;
		 l && r;
		 l=l->s_next, r=r->e_r)
		    if (!sigcmp(l, r->e_l->e_sig, NULLNODE, r->e_l, 0))
			return FALSE;
	    return !l && !r;
	default:
	    return FALSE;
    }
}


void
replace(e, old_tok, new_tok, sig)
Nodeptr e;
Token old_tok, new_tok;
Symptr sig;
{
    assert (e);
    if (e->e_op == TK_LIST)
	e = e->e_l;
    assert (e->e_op == old_tok);
    e->e_op = new_tok;
    if (sig != NULLSYM)
	e->e_sig = sig;
}
