/* isit.c -- miscellaneous inquiry functions */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"

static Bool contig_ndx();



/*  assignable(sig) - check if assignable type.
 *
 *  Returns the name of a C type (e.g. "int") if the signature is a type
 *  assignable by C; else returns NULL.
 */
char *
assignable(sig)
Symptr sig;
{
    if (sig->s_size != 1)
	return NULL;
    switch (sig->s_type)  {
	case T_INT:
	case T_ENUM:
	case T_PTR:
	case T_FILE:
	    return "int";
	case T_CHAR:
	case T_BOOL:
	    return "char";
	default:
	    return NULL;
    }
}



/*  return true if e has an address. */
/*  indexing is addressable only if memory is contiguous */
Bool
is_addressable(e)
Nodeptr e;
{   
    Nodeptr func;
    struct assign_descriptor temp;

    switch (e->e_op) {
	case TK_INDEX:
	    return contiguous(&temp, e);
	case TK_IDENTIFIER:
	    return (Bool)(e->e_s->s_kind != K_LITERAL);
	case TK_IMPORTED_CONST:
	case TK_PERIOD:
	case TK_HAT:
	case TK_FORMAL:
	case TK_STRLIT:
	case TK_SUBSTR:
	case TK_CONCAT:
	    return TRUE;
	case TK_CALL:
	case TK_CO_CALL_COPY_BACK:
	    func = e->e_l->e_l;
	    if (func->e_op == TK_IDENTIFIER && func->e_s->s_kind == K_PREDEF)
		return is_predef_addressable(func->e_s);
	    else
		return (Bool)(e->e_l->e_sig->s_type != T_VOID);
	default:
	    return FALSE;
    }
}



Bool
is_lvalue(e)
Nodeptr e;
{
    return ((Bool)(e->e_op == TK_IDENTIFIER ||
		  e->e_op == TK_FORMAL ||
    		  e->e_op == TK_PERIOD ||
		  e->e_op == TK_INDEX ||
		  e->e_op == TK_SUBSTR ||
	   	  e->e_op == TK_HAT));
}



/* does s (note: Symptr!) indicate a type usable as a string? */
Bool
is_string(s)
Symptr s;
{
    return ((s->s_type == T_STRING && IS_SCALAR(s))
	||  (s->s_type == T_CHAR && !IS_SCALAR(s)));
}




Bool
is_constant(node, value)
Nodeptr node;
int *value;
{   Symptr sym;

    if (node == NULLNODE)
	return FALSE;
    else if (node->e_op == TK_NUMBER) {
        *value = node->e_i;
        return TRUE;
    } else if (node->e_op == TK_CHRLIT) {
	*value = node->e_i;
	return TRUE;
    } else if (node->e_op == TK_BOOLEAN) {
	*value = node->e_i;
	return TRUE;
    } else if (node->e_op != TK_IDENTIFIER &&
	       node->e_op != TK_IMPORTED_CONST) {
        return FALSE;
    } else {
        sym = node->e_s;
/** We have a conflict in the use of the s_value field here.... */
        if (sym->s_kind == K_CONST) {
            return is_constant(sym->s_value, value);
        }
        else if (sym->s_kind == K_LITERAL) {
            *value = sym->s_offset;
            return TRUE;
        } else {
            return FALSE;
        }
    }
}



/* return TRUE if e is a constant node */
Bool
is_a_const(e)
Nodeptr e;
{
    return (Bool) (e->e_op == TK_IDENTIFIER && e->e_s->s_kind == K_CONST);
}



/* does e represent a contiguous chunk of memory, e known to be an lvalue */
/*  if contiguous, fill in fixed_total/total
    else
	fill in fixed_total/total, fixed_slice_size/slice_size, fixed_rows/
		num_rows, fixed_row_length/row_length
*/
Bool
contiguous(desc, e)
struct assign_descriptor *desc;
Nodeptr e;
{
	/* non-contiguous "reduces" to:
	    1st dimension > 1 && either bound of second
	    dimension doesn't reach extremeties (or
	    extremeties are not known at compile time)
	or, some bounds are not known at compile time */

	if (e->e_op != TK_INDEX)
	{
		assert((e->e_op == TK_IDENTIFIER || e->e_op == TK_PERIOD
			|| e->e_op == TK_NUMBER || e->e_op == TK_CHRLIT
			|| e->e_op == TK_BOOLEAN || e->e_op == TK_STRLIT
			|| e->e_op == TK_FILE_CONST || e->e_op == TK_NULL
			|| e->e_op == TK_NOOP || e->e_op == TK_HAT)
			&& e->e_sig->s_size != SIZE_ARB);
/* may want to assert that if op is TK_NULL,TK_NOOP, type is not T_NULL, T_NOOP
*/
		if (e->e_sig->s_size == SIZE_UNK)
			desc->fixed_total = FALSE;
		else
		{	desc->fixed_total = TRUE;
			desc->total = e->e_sig->s_size * get_ob_size(e->e_sig);
		}
		return TRUE;
	}
	else 
	{   Nodeptr top;
			/* find ranges of item to be indexed */
			/* the left side of TK_INDEX is always an identifier */
	    if ((top = e->e_l)->e_op == TK_PERIOD)
		top = top->e_r;
	    assert(top->e_op == TK_IDENTIFIER);
/* what if get_ob_size() return 0 (unknown)? */
	    return contig_ndx(desc, get_ob_size(top->e_sig),
		top->e_s->s_ranges, e->e_r);
	}
}



static Bool
contig_ndx(desc, unit_size, ranges, indices)
struct assign_descriptor *desc;
int unit_size;
Rangeptr ranges;
Nodeptr indices;
{	int i,j,k,l;

		/* have already checked that compile time bounds and
			dimensionality are consistent with declarations
		*/
	assert(ranges);
	if (ranges->r_dim2 == NULLNODE)
	{
		if (indices->e_l->e_r == NULLNODE)
		{	desc->fixed_total = TRUE;
			desc->total = unit_size;
		}
		else if (unit_size != SIZE_UNK
			&& is_constant(indices->e_l->e_l, &i)
			&& is_constant(indices->e_l->e_r, &j))
		{	
			desc->fixed_total = TRUE;
			desc->total = (j-i+1) * unit_size;
			assert(desc->total > 0);
		}
		else
			desc->fixed_total = FALSE;

		return TRUE;		/* 1-dimensional arrays are 
						always contiguous */
	}
	else if (indices->e_l->e_r == NULLNODE || 
		(is_constant(indices->e_l->e_l, &i) 
			&& is_constant(indices->e_l->e_r, &j)
			&& i == j))
	{
		if (unit_size != SIZE_UNK && indices->e_r == NULLNODE 
			|| indices->e_r->e_l->e_r 
			    == NULLNODE)
		{	desc->fixed_total = TRUE;
			desc->total = unit_size;
		}
		else if (unit_size != SIZE_UNK && is_constant(
			indices->e_r->e_l->e_l, &i)
			&& is_constant(
				indices->e_r->e_l->e_r, &j))
		{	desc->fixed_total = TRUE;
			desc->total = (j-i+1) * unit_size;
			assert(desc->total > 0);
		}
		else
			desc->fixed_total = FALSE;

		return TRUE;
					/* essentially, take a 1-dimensional
						slice */
	}
	else	/* need more than one (or an unknown number) of rows */
	{
		if (is_constant(indices->e_l->e_l, &i)
			&& is_constant(indices->e_l->e_r, &j))
		{	desc->fixed_rows = TRUE;
			desc->num_rows = j - i + 1;
			assert(desc->num_rows > 0);
		}
		else
			desc->fixed_rows = FALSE;

		/* if rows extend to their extremes, then contiguous */
		if ((is_constant(ranges->r_dim2->e_l, &i) 
		    && is_constant(ranges->r_dim2->e_r, &j) 
		    && is_constant(
		       indices->e_r->e_l->e_l, &k))
		  && ((i == j 
		    && indices->e_r->e_l->e_r == NULLNODE)
		    || (indices->e_r->e_l->e_r != NULLNODE 
			&& is_constant(
			  indices->e_r->e_l->e_r, &l) 
			  && i == k && j == l)))
		{
			if (unit_size != SIZE_UNK && desc->fixed_rows)
			{	desc->fixed_total = TRUE;
				desc->total = desc->num_rows * (j-i+1) 
					* unit_size;
			}
			else
				desc->fixed_total = FALSE;
			return TRUE;
		}
		else
		{
			if (unit_size != SIZE_UNK &&
				indices->e_r->e_l->e_r 
				== NULLNODE)
			{	desc->fixed_slice_size = TRUE;
				desc->slice_size = unit_size;
			}
			else if (unit_size != SIZE_UNK && is_constant(
			  indices->e_r->e_l->e_l, &k)
			  && is_constant(
			    indices->e_r->e_l->e_r, &l))
			{	desc->fixed_slice_size = TRUE;
				desc->slice_size = (l-k+1) * unit_size;
			}
			else
				desc->fixed_slice_size = FALSE;

			if (unit_size != SIZE_UNK &&
			  is_constant(ranges->r_dim2->e_l, &i)
			  && is_constant(ranges->r_dim2->e_r, &j))
			{	desc->fixed_row_length = TRUE;
				desc->row_length = (j-i+1) * unit_size;
				assert(desc->row_length > 0);
			}
			else
				desc->fixed_row_length = FALSE;

			if (desc->fixed_rows && desc->fixed_slice_size)
			{	desc->fixed_total = TRUE;
				desc->total = desc->num_rows *
				    desc->slice_size;
			}
			else
				desc->fixed_total = FALSE;
			return FALSE;
		}
	}
}
