/* expr.c -- read and parse an expression, returning root of the tree */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "tokmacs.h"
#include "globals.h"

static Nodeptr binary_node(), unary_node(), term(), vector_element(), new();
static struct node *exprx();

				/* 5. Expressions, Signatures,
				      and Type Checking */

				/* 5.1 Expressions */

Nodeptr
expr()
{
    Nodeptr lside;
    Token operator;

    lside = term();
    operator = get_token();
    if (PRECEDENCE(operator))
	return exprx(lside, operator, 1);
    else {
	putback();
	return lside;
    }
}


/* exprx -- parse an expression composed of operations with precedence >= n */
static Nodeptr
exprx(lside, operator, n)
Nodeptr lside;
Token operator;
int  n;
{
    Token nxt_op;
    Nodeptr rside;

    /* get an operand, peek at next token to see if it's a binary operator */
    for (;;) {
	if (PRECEDENCE(operator) > n) {
	    lside = exprx(lside, operator, n+1);
	    operator = get_token();
	    if (PRECEDENCE(operator) < n) {
		putback();
		return lside;
	    }
	    /* else, PRECEDENCE(operator) == n */
	}
	rside = term();
	nxt_op = get_token();
	if (!PRECEDENCE(nxt_op)) {
	    putback();
	    return binary_node(operator, lside, rside);
	} else if (PRECEDENCE(nxt_op) > n) {	
	    lside = binary_node(operator, lside, exprx(rside, nxt_op, n+1));
	    operator = get_token();
	    if (PRECEDENCE(operator) < n) {
		putback();
		return lside;
	    }
	} else if (PRECEDENCE(nxt_op) == n) {
	    lside = binary_node(operator, lside, rside);
	    operator = nxt_op;
	} else	/* PRECEDENCE(nxt_op) < n */ {
	    putback();
	    return binary_node(operator, lside, rside);
	}
    }
}


/* term() -- parse term.  returns NULLNODE if bogus term. */
static Nodeptr
term()
{
    Nodeptr n;
    Symptr s;

    get_token();
    switch (tok) {
	case TK_IDENTIFIER:
	/* for casts of builtins: */
	case TK_BOOL:
	case TK_CHAR:
	case TK_INT:
	/* for record constructor (which is like a cast): */
	case TK_REC:
	    putback();
			n = denotation(TK_CALL);
	    if (!n)
		ERROR(E_FATAL+6,"denotation");
	    return (n);
	case TK_NEW:
	    return new();
	case TK_ADDR:
	    return unary_node(TK_ADDR, term());
	case TK_NOT:
	    return unary_node(TK_NOT,term());
	case TK_MINUS:
	    return unary_node(TK_UMINUS,term());
	case TK_CHRLIT:
	case TK_STRLIT:
	case TK_NUMBER:
	case TK_BOOLEAN:
	case TK_FILE_CONST:
	    return (yynode);
	case TK_LEFTPAREN:
	    n = vector_element();
	    if (get_token()==TK_COMMA || n->e_op==TK_CLONE) {
		putback();
		n=bnode(TK_VECTORIZE,NULLNODE,n); 
		while (get_token() == TK_COMMA)
		    n = bnode(TK_VECTORIZE, n, vector_element());
	    }
	    putback();
	    mustbe(TK_RIGHTPAREN,")");
	    return (n);
	case TK_NOOP:
	case TK_NULL:
	    return (bnode(tok,NULLNODE,NULLNODE));
	case TK_QUESTION:
	    mustbe(TK_IDENTIFIER, "op_identifier");
	    if ((s = st_lookup(tk_str)) == NULLSYM) {
		ERROR(E_FATAL+6, "denotation");
		putback();
		return (NULLNODE);
	    } else if (get_token() == TK_LEFTBKET) {
		n = idnode(s);
		n = binary_node(TK_INDEX, n, indices());
		return (unary_node(TK_QUESTION, n));
	    } else {
		putback();
		return (unary_node(TK_QUESTION, idnode(s)));
	    }
	default:
	    errmsg(E_FATAL,"invalid term in expression: %s",yytext);
	    return (NULLNODE);
    }
}


/* vector_element() -- parse parenthesized expression */
static Nodeptr
vector_element()
{
    Nodeptr p;

    if (maybe(TK_LEFTBKET)) {
	p = expr();
	mustbe(TK_RIGHTBKET, "right bracket");
	p = binary_node(TK_CLONE, p, expr());
    } else {
	p = expr();
    }
    if (p == NULLNODE)
	return bnode(TK_NULL,NULLNODE,NULLNODE);
    else
	return p;
}

/* farg() -- parse actual argument list.  resulting structure looks like:

	---> LIST ---> LIST ---> ... ---> LIST ...---> LIST ---> (NULL)
	     |         |	          |            |
	     arg_1     arg_2              arg_(n-1)    arg_n

    here, `--->' is e_r and `|' is e_left
*/
Nodeptr
farg()
{
    Nodeptr top, end, q;

    top = NULLNODE;
    if (maybe(TK_RIGHTPAREN))
	return (top);
    do {
	if ((q = expr()) != NULLNODE) {
	    if (top == NULLNODE) {
		top = end = unary_node(TK_LIST,q);
	    } else {
		end->e_r = unary_node(TK_LIST,q);
		end = end->e_r;
	    }
	}
    } while (maybe(TK_COMMA));
    mustbe(TK_RIGHTPAREN, ")");
    return (top);
}

/* int_expr() -- call expr and verify that signature of returned expr is int */
Nodeptr
int_expr()
{
    Nodeptr n;
    n = expr();
    if (n != NULLNODE) {
	assert(n->e_sig != NULLSYM);
	if (n->e_sig->s_type != T_INT || n->e_sig->s_ranges)
	    FATAL("scalar int expression required");
    }
    return n;
}

	
/* bool_expr() -- parse Boolean expression.
 * just call expr() and check signature of returned expression.
 */
Nodeptr
bool_expr()
{
    Nodeptr n;

    n = expr();
    if (n != NULLNODE) {
	assert (n->e_sig != NULLSYM);
	if (!IS_SIMPLE(n->e_sig,T_BOOL))
	    FATAL("(scalar) Boolean expression required");
    }
    return (n);
}


/* ot_expr() -- ordered type expression.
 * Usually, just call expr() and check signature of returned expression.
 */
Nodeptr
ot_expr()
{
    Nodeptr n;

    n = expr();
    if (n == NULLNODE)
	return (n);
    assert (n->e_sig != NULLSYM);
    if (!IS_SCALAR(n->e_sig))
	FATAL("scalar required");
    else if (!IS_ORDERED(n->e_sig->s_type))
	FATAL("ordered type expression required");
    return (n);
}


/* new() -- parse new(type) */
static Nodeptr
new()
{
    Symptr s;
    Nodeptr e;

    s = new_symbol(K_TYPE);
    e = NULLNODE;
    mustbe(TK_LEFTPAREN,"(");
    switch (get_token()) {
	case TK_BOOL:
	    s->s_type = T_BOOL;
	    s->s_size = BOOLSIZE;
	    break;
	case TK_CHAR:
	    s->s_type = T_CHAR;
	    s->s_size = CHARSIZE;
	    break;
	case TK_FILE:
	    s->s_type = T_FILE;
	    s->s_size = FILESIZE;
	    break;
	case TK_INT:
	    s->s_type = T_INT;
	    s->s_size = INTSIZE;
	    break;
	case TK_STRING:
	    mustbe(TK_LEFTPAREN,"(");
	    if (maybe(TK_STAR))
		FATAL("new(string(*)) is illegal");
	    else
		e = int_expr();
	    mustbe(TK_RIGHTPAREN,")");
	    s->s_type = T_STRING;
	    break;
	case TK_IDENTIFIER:
	    s = st_unqual_lookup(tk_str,2);
	    if (s->s_kind == K_BLOCK || s->s_kind == K_IMPORT) {
		mustbe(TK_PERIOD,".");
		mustbe(TK_IDENTIFIER,"type identifier");
		if ((s = at_lookup(s->s_tdef, tk_str)) == NULLSYM) {
		    ERROR(E_FATAL+2,NULLSTR);
		    return (NULLNODE);
		}
	    }
	    if (s->s_kind != K_TYPE)
		ERROR(E_FATAL+2,tk_str);
	    break;
	case TK_ENUM:
	case TK_REC:
	case TK_UNION:
	case TK_CAP:
	case TK_PTR:
	case TK_ANY:
	    FATAL("new() requires specific, named type");
	default:
	    FATAL("new() requires a type");
    }
    mustbe(TK_RIGHTPAREN,")");
    return snode(TK_NEW,s,e);
}


/* binary_node(op,left,right) -- construct a new node if args are not null */
static Nodeptr
binary_node(op, left, right)
Token op;
Nodeptr left, right;
{
    if (left == NULLNODE || right == NULLNODE)
	return NULLNODE;
    else
	return bnode(op, left, right);
}


/* unary_node(op,left) -- construct and return a new node if left is not null */
static Nodeptr
unary_node(op, left)
Token op;
Nodeptr left;
{
    if (left == NULLNODE)
	return NULLNODE;
    else
	return bnode(op, left, NULLNODE);
}
