/*  copy.c -- generate code for data movement in various forms  */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"
#include "../srsys.h"

static void gen_create(), gen_vector(), gen_store();
static void str_to_str(), chr_to_str(), str_to_chr();




/* generate code for assignment */

void
gen_assign(l,r)
Nodeptr l, r;
{
    int vp;

    if (!is_lvalue(l)) {
	FATAL("LHS of assignment is not an lvalue.");
    } else switch (r->e_op) {
	case TK_CREATE:
	    gen_create(l, r);
	    break;
	case TK_CREVM:
	    cprintf("sr_crevm(%a,",l);
	    if (r->e_r)
		cprintf("%e)",r->e_r);
	    else
		cprintf("sr_my_machine)");
	    break;
    	case TK_VECTORIZE:	/* vector constructor: */
	    vp = temp_alloc(PTRSIZE);
	    cprintf("LA(%d)=%a",vp,l);
	    gen_vector(r,l->e_sig,vp);		/* gen items */
	    temp_free(PTRSIZE);
	    break;
	case TK_RESCAP_NULL:
	case TK_RESCAP_NOOP:
	    cprintf("sr_build_rcap(%a,%d,&sr_%s_ocap)", l,
		r->e_sig->s_tdef->s_size,
		(r->e_op == TK_RESCAP_NOOP) ? "no" : "nu");
	    break;
	case TK_CAST:
	    if (r->e_s->s_type == T_REC)
		gen_rec(l,r);
	    else
		gen_store(l,r,l->e_sig);
	    break;
	default:
	    gen_store(l,r,l->e_sig);
	    break;
    }
}



/* generate code for a create statement */

static void
gen_create(capvar, e)
Nodeptr capvar, e;
{
    int crbptr, st_size, blsize;
    Nodeptr actuals;
    Symptr formals;

    assert(e->e_op == TK_CREATE);
    assert(e->e_l);
    assert(e->e_r);
    assert(is_addressable(capvar));
    
    crbptr = temp_alloc(PTRSIZE);
    formals = e->e_l->e_s->s_tdef->s_next;
    actuals = e->e_r->e_r;

    /* define external resource index */
    cprintf("%hextern int RN%s;\n",e->e_l->e_s->s_name);
    
    /* allocate crb */
    blsize = temp_alloc(INTSIZE);
    st_size = 
	pb_alloc(formals, actuals, NULLNODE, CRB_HEADER_SIZE, crbptr,
		 RTS_OWN,blsize);

    cprintf(";\n{struct crb_st *p=(struct crb_st *)LA(%d);\n",crbptr);

    /* copy packet size into crb_size */
    cprintf("p->crb_size=LO(%d);\n",blsize);

    /* fill in resource pattern index */
    cprintf("p->rpatid=RN%s;\n",e->e_l->e_s->s_name);

    /* fill in rcp */
    cprintf("p->rcp=(rescap *)%a;\n",capvar);

    /* fill in rc_size */
    cprintf("p->rc_size=%z;\n",capvar);

    /* fill in machine number */
    if (e->e_r->e_l)
	cprintf("p->vm=I(%a)",e->e_r->e_l);
    else
	cprintf("p->vm=sr_my_vm");
    
    copy_list(formals, actuals, NULLNODE, crbptr, CRB_HEADER_SIZE + st_size);

    /* call create and clean up the stack */
    cprintf(";\nsr_create(p);\n}");
}



/*  generate vector constructor e, storing at *LA(vp) with signature s  */

static void
gen_vector(e,s,vp)
Nodeptr e;
Symptr s;
int vp;
{
    Symbol s1;			/* scalar variant of s */
    Node n;

    s1 = *s;
    s1.s_size = 1;
    s1.s_ranges = 0;
    n.e_op = TK_ALOCAL;
    n.e_i = vp;
    n.e_r = 0;
    n.e_sig = &s1;
    assert(e->e_op == TK_VECTORIZE);
    if (e->e_l)
	gen_vector(e->e_l,&s1,vp);
    e = e->e_r;
    if (s->s_type != e->e_sig->s_type)
	FATAL("incompatible vector assignment");

    cprintf(",\n");

    if (e->e_op == TK_CLONE) {
	/* if clone, store once and then duplicate */
	cprintf("%e>0?(",e->e_l);
	gen_store(&n,e->e_r,&s1);
	cprintf(",%a=sr_clone(%a,%Z,%e)):\"\"",&n,&n,&s1,e->e_l);
     } else {
	/* else just store and bump address */
	gen_store(&n,e,&s1);
	cprintf(",%a+=%Z",&n,&s1);
    }
}



/*  gen_store(a,e,s) -- store e at address of a using signature s  */

static void
gen_store(a,e,s)
Nodeptr a, e;
Symptr s;
{
    if (IS_SIMPLE(s,T_STRING))
	if (e->e_sig->s_type == T_STRING)
	    str_to_str(a,e,s);		/* string := string */
	else
	    chr_to_str(a,e);		/* string := char[] */
    else if (s->s_type == T_CHAR && e->e_sig->s_type == T_STRING)
	str_to_chr(a,e,s);		/* char[] := string */
    else if (assignable(s))
	cprintf("%e=%e",a,e);
    else
	cprintf("memcpy(%a,%a,%z)",a,e,e);
}




/*  str_to_str(a,e,s) -- copy string e to string a using signature s
 *
 *  Strings are always in memory, and always aligned, because any use of a
 *  string in an expression (including a substring) is cast to array of char.
 *  Easiest and probably fastest thing to do is to blindly copy as much of the
 *  source string as will fit, including the header field giving the "current"
 *  string size.
 */

static void
str_to_str(a,e,s)
Nodeptr a, e;
Symptr s;
{
    int lsz = get_ob_size(s);	/* get left size */
    int rsz = get_size(e);	/* get right size */
    assert(e->e_sig->s_size == 1);
    /* if both maxima known, use the smaller, else use current size */
    if (lsz != SIZE_UNK && rsz != SIZE_UNK)
	if (lsz <= rsz)
	    cprintf("memcpy(%a,%a,%d)",a,e,lsz);
	else
	    cprintf("memcpy(%a,%a,%d)",a,e,rsz);
    else
	cprintf("memcpy(%a,%a,%d+%l)",a,e,INTSIZE,e);
}



/*  str_to_chr(a,e,s) -- store string e into char array a of signature s */

static void
str_to_chr(a,e,s)
Nodeptr a, e;
Symptr s;
{
    assert(e->e_sig->s_size == 1);
    if (s->s_size != SIZE_UNK && s->s_size != SIZE_ARB)	/* constant size? */
	cprintf("memcpy(%a,%d+%a,%d)",a,INTSIZE,e,s->s_size);
    else
	cprintf("memcpy(%a,%d+%a,%l)",a,INTSIZE,e,e);
}



/*  chr_to_str(a,e) -- store char expr e into string a  */

static void
chr_to_str(a,e)
Nodeptr a, e;
{
    assert(e->e_sig->s_type == T_CHAR);
    if (e->e_sig->s_size == 1)
	cprintf("*(%a+%d)=%e,I(%a)=1",a,INTSIZE,e,a);
    else
	cprintf("memcpy(%d+%a,%a,%z),I(%a)=%z",INTSIZE,a,e,e,a,e);
}



/* generate code for swap statements */

void
gen_swap(l,r)
Nodeptr l, r;
{
    int temp;

    if (!is_lvalue(l))
	FATAL("LHS of :=: is not an lvalue.");
    else if (!is_lvalue(r))
	FATAL("RHS of :=: is not an lvalue.");
    else if (assignable(l->e_sig) && assignable(r->e_sig)) {
	temp = temp_alloc(INTSIZE);
	cprintf("LO(%d)=%e,\n%e=%e,\n%e=LO(%d)",temp,l,l,r,r,temp);
	temp_free(INTSIZE);
    } else if (l->e_sig->s_type == T_STRING)
	cprintf("sr_swap(%a,%a,0)",l,r);
    else
	cprintf("sr_swap(%a,%a,%z)",l,r,l);
}




/* generate code to build record e at address given by tree a */

void
gen_rec(a, e)
Nodeptr a, e;
{
    Node n;
    Nodeptr r;
    Symptr f;

    n.e_op  = TK_ALOCAL;
    n.e_r   = 0;
    n.e_i = temp_alloc(PTRSIZE);
    cprintf("(");
    for (f = e->e_s->s_tdef->s_next, r=e->e_r; r&&f; r=r->e_r,f=f->s_next) {
	n.e_sig = f;
	if (f->s_offset != OFF_UNK)
	    cprintf(",%a=%a+%d,",&n,a,f->s_offset);
	else
	    cprintf(",%a=%a+(int)%A,",&n,a,f);	/* %A actually is an offset */
	/* should use left side sig, but it doesn't exist */
	/* this will fail if string:=char[] or v.v */
	gen_store(&n,r->e_l,r->e_sig);
    }
    cprintf(")");
    temp_free(PTRSIZE);
}
