/*  comp.c -- System Components and Procs.  */

#include <stdio.h>
#include "../util.h"
#include "sr.h"
#include "funcs.h"
#include "globals.h"

static Bool res_spec(), res_body_guts();
static void global_comp(), body_comp(), res_comp(), res_params();
static void spec_declr(), body(), res_body(), fake_res_body();
static void res_initial(), res_final(), res_make_initial(), res_initial_final();
static void proc(), process(), proc_guts(), send_to_process();

/* some of these might have changed because of new resource components.
 * think about this later; also 0 should be "not a token", so change
 * find_follow.
 */

static Token fl_sys_comp[]
	= {TK_EOF,TK_GLOBAL,TK_RESOURCE,TK_BODY,TK_NOTATOKEN};
static Token fl_res_declr[]
	= {TK_END,TK_INITIAL,TK_PROC,TK_PROCESS,TK_FINAL,TK_NOTATOKEN};
/* next probably isn't quite right. */
static Token fl_par[] = {TK_RIGHTPAREN,TK_COMMA,TK_SEMICOLON,TK_NOTATOKEN};


/* think about simplifying these below - are they needed? */
static Token fl_proc[] = {TK_INITIAL,TK_FINAL,TK_PROC,TK_PROCESS,
				TK_END,TK_NOTATOKEN};


static void optional_end_follower();
static void global_declr();
static void copy_quant();
static Nodeptr process_args();

				/* 1. System Components (Compilation Units) */
void
srparse()
{
    switch (get_token()) {
	case TK_GLOBAL:
	    global_comp();
	    break;
	case TK_BODY:
	    body_comp();
	    break;
	case TK_RESOURCE:
	    res_comp();
	    break;
	default:
	    errmsg(E_FATAL,
		"expected 'resource', 'body', or 'global'; found '%s'", yytext);
	    find_follow(fl_sys_comp);
	    break;
	}

    if (inter_depth == 0)
	tidy_sym();
}

				/* 1.1 Global Components */

/* note: decls go in aux table, just like decls for spec. */
static void
global_comp()
{
    parsing_body = FALSE;
    mustbe(TK_IDENTIFIER,"global_identifier");
    if (dbflags['T'])
	printf("global %s\n", tk_str);

    if (!(comp_sym = st_install(tk_str,K_BLOCK))) {
	find_follow(fl_sys_comp);
	return;
    }
    comp_sym->s_type = T_GLOBAL;
    comp_name = comp_sym->s_name;

    comp_sym->s_tdef = main_table = new_symbol(K_NOTAKIND);
    if (inter_depth == 0)
	start_inter(TK_GLOBAL);
    global_declr();
    main_table = NULLSYM;

    if (inter_depth == 0)
	end_inter(TK_END);

    mustbe(TK_END,"globals end");
    optional_end_follower(TK_IDENTIFIER,comp_name);
    pop_block();
}


static void
global_declr()
{
    while (TRUE) {
	if (maybe(TK_CONST))
	    var_decl(K_CONST);
	else if (maybe(TK_TYPE))
	    type_declr();
	else if (maybe(TK_OPTYPE))
		optype_declr();
	else
	    break;

	/* skip over optional semicolon as terminator. */
	maybe(TK_SEMICOLON);
    }
}


static void
body_comp()
{
    assert(tok == TK_BODY);
    mustbe(TK_IDENTIFIER,"body_identifier");
    comp_name = tk_str;
    if (dbflags['T'])
	printf("body %s\n", comp_name);
    body(TK_BODY);
    pop_block();
}

static void
res_comp()
{

    assert(tok == TK_RESOURCE);
    if (res_spec()) { 			/* concrete */
	if (!maybe(TK_SEPARATE)) {
	    if (dbflags['T'])
		printf("body %s\n", comp_name);
	    body(TK_RESOURCE);
	}
    } else {				 /* abstract */
	/* do this here, after end_inter, so wi_fd
	 * doesn't get extra token if not optional name.  */
	optional_end_follower(TK_IDENTIFIER,comp_name);
    }
    pop_block();
}

/* returns true iff concrete resource. */
static Bool
res_spec(){
    Bool concrete;

    assert (tok == TK_RESOURCE);
    parsing_body = FALSE;
    mustbe(TK_IDENTIFIER,"resource identifier");
    if (dbflags['T'])
	printf("spec %s\n", tk_str);

    if (!(comp_sym = st_install(tk_str,K_BLOCK))) {
	find_follow(fl_sys_comp);
	return FALSE;
    }
    comp_sym->s_type = T_SPEC;
    comp_name = comp_sym->s_name;

    comp_sym->s_tdef = main_table = new_symbol(K_NOTAKIND);

    if (inter_depth == 0) start_inter(TK_RESOURCE);

    if (maybe(TK_LEFTPAREN)) {
	concrete = TRUE;
    } else {
	spec_declr();
	switch (get_token()) {
	    case TK_BODY:
		mustbe(TK_IDENTIFIER, "resource identifier after `body'");
		if (strcmp(tk_str,comp_name) != 0)
		    errmsg(E_WARN,
			"mismatched component name: found '%s'",yytext);
		mustbe(TK_LEFTPAREN, "need `(' after `body' and id after spec");
		concrete = TRUE;
		break;
	    case TK_END:
		concrete = FALSE;
		break;
	    default:
		errmsg(E_FATAL, "expected 'body' or 'end', found '%s'", yytext);
		concrete = FALSE;
		break;
	}
    }

    if (concrete)
	res_params();

    main_table = NULLSYM;
    if (inter_depth == 0)
	end_inter(concrete?TK_RESOURCE:TK_END);

    return concrete;
}

static void
res_params()
{
    /* resource parameters, like proc formals but unlike
     * operation parameters, go in the "main" symbol table.  */
    assert (tok == TK_LEFTPAREN);
    if (maybe(TK_RIGHTPAREN))
	return;
    do {
	param_spec(TK_RESOURCE);
    } while (semi_sep());
    putback();
    mustbe(TK_RIGHTPAREN,")");
}

static void
spec_declr()
{
    for (;;) {
	switch (tok = get_token()) {
	    case TK_IMPORT:
		import_declr();
		break;
	    case TK_EXTEND:
		extend_declr();
		break;
	    case TK_CONST:
		var_decl(K_CONST);
		break;
	    case TK_TYPE:
		type_declr();
		break;
	    case TK_OPTYPE:
		optype_declr();
		break;
	    case TK_OP:
	    case TK_SEM:
	    case TK_EXTERNAL:
		op_declr(tok);
		break;
	    default:
		putback();
		return;
	}
	/* skip over optional semicolon as terminator. */
	maybe(TK_SEMICOLON);
    }
}

/* invoked from within here and now also from statement level. */
void
import_declr()
{
    Symptr sym;

    assert (tok == TK_IMPORT);
    do {
	if (get_token() == TK_IDENTIFIER) {
	    if ((sym = st_lookup(tk_str)) == NULLSYM) {
		sym = import_inter(tk_str,TK_IMPORT);
		if (parsing_body && sym) {
		    emit(I_IMPORT,idnode(sym),NOLAB);
		}
	    } else if (sym->s_how_imported == TK_EXTEND)
		/* was implicitly imported, promote to explicitly imported */
		sym->s_how_imported = TK_IMPORT;
	    else
		/* this is fatal to prevent circularities.
		 * note: this handles cases where s_how_imported==TK_IMPORT,
		 * e.g. `import a; import a'
		 * and when s_how_imported==TK_NOTATOKEN,
		 * e.g., `op goo(); import goo'
		 * and `resource self; import self'
		 */
		ERROR(E_FATAL+7,"import item");
	} else {
	    ERROR(E_WARN+6, "bad item on import list");
	}
    } while (maybe(TK_COMMA));
}

/* handle extend.
 * do the implicit import for the extended component too.
 */
void
extend_declr()
{
    Symptr sym;

    assert (tok == TK_EXTEND);
    assert (!parsing_body);
    do {
	if (get_token() == TK_IDENTIFIER) {
	    if ((sym = st_lookup(tk_str)) == NULLSYM)
		sym = import_inter(tk_str,TK_EXTEND);
	    else if (sym->s_kind != K_IMPORT)
		errmsg(E_FATAL, "extend of an already declared id: %s", tk_str);
	    if (sym) {
		/* use sym->s_name as import_inter changes tk_str. */
		/* if sym is NULLSYM then error given above. */
		extend_inter(sym->s_name);
	    }
	} else {
	    ERROR(E_WARN+6, "bad item on extend list");
	}
    } while (maybe(TK_COMMA));
}

/* do the body for either separate body or combined resource. */
static void
body(which)
Token which;
{
    assert (which == TK_BODY || which == TK_RESOURCE);
    curr_label = 1;
    stack_init();

    if (start_body_inter(which)) {
	/* setup input stmt and classes for body.
	 * do it here after spec and possibly extends.
	 */
	input_init();
	class_spec_ops(); /* put each op in spec in its own class. */
	res_body();
	end_body_inter();
    } else {
	fake_res_body(which);
    }
    optional_end_follower(TK_IDENTIFIER,comp_name);
}


				/* 1.3 Resource Bodies*/
static void
res_body()
{
    Bool did_initial;

    parsing_body = TRUE;
    emit(I_COMPONENT,NULLNODE,NOLAB);

    did_initial = res_body_guts();

    /* ensure there's an initial, to make icode.c and code generator simpler. */
    if (! did_initial)
	res_make_initial();

    /* take care of input statements. */
    do_input();

    /* check for unused identifiers.
     * do it before I_COMPONENT_END because that triggers
     * code generation, which takes a while and would cause
     * a noticeable pause between error messages.
     */
		check_used();

    emit(I_COMPONENT_END,NULLNODE,NOLAB);
}


/* Parses the innards of a resource body.
 * Note: procs, initial, final, and declarations can appear in ANY order;
 * in particular, declarations can appear after procs.
 * Declaration before use is still enforced.
 * returns TRUE iff user specified initial was present in body.
 */
static Bool
res_body_guts()
{
    Bool did_initial, did_final;

    did_initial = did_final = FALSE;
    if (dbflags['T'])
	printf("resource body guts\n");

    get_token();
    while (tok != TK_END) {
	if (dbflags['S'])
	    printf("Loop. tok %s\n",tokentos(tok));
	switch (tok) {
	    case TK_CONST:
		var_decl(K_CONST);
		break;
	    case TK_TYPE:
		type_declr();
		break;
	    case TK_OP:
	    case TK_SEM:
	    case TK_EXTERNAL:
		op_declr(tok);
		break;
	    case TK_OPTYPE:
		optype_declr();
		break;
	    case TK_VAR:
		var_decl(K_VAR);
		break;
	    case TK_IMPORT:
		import_declr();
		break;
	    case TK_INITIAL:
		if (did_initial) {
		    WARN("multiple initial ignored");
		    find_follow(fl_res_declr);
		} else {
		    did_initial = TRUE;
		    res_initial();
		}
		break;
	    case TK_FINAL:
		if (did_final) {
		    WARN("multiple final ignored");
		    find_follow(fl_res_declr);
		} else {
		    did_final = TRUE;
		    res_final();
		}
		break;
	    case TK_PROC:
		proc();
		break;
	    case TK_PROCESS:
		process();
		break;
	    default:
		WARN("bad resource-level declaration");
		find_follow(fl_res_declr);
		putback();
		break;
	}

	/* skip over optional semicolon as terminator. */
	maybe(TK_SEMICOLON);
	    get_token();
    }
    return (did_initial);
}

/* fake_res_body(mytok)
 * this routine is for error recovery.
 * mytok tells what kind of component we are parsing.
 * it skip over the rest of a body,
 * and does the minimum so that we can continue compilation.
 */
static void
fake_res_body(mytok)
Token mytok;
{
    Symptr sym;

    assert(mytok == TK_BODY  || mytok == TK_RESOURCE);
    FATAL("skipping rest of body");
    find_follow(fl_sys_comp);
    /* error recovery: if necessary, make a new block in st so that res_comp()
     * has something to pop.  such is necessary for a separate body
     * that couldn't import its spec.
     * note that for a combined resource, the block is already there.
     */
    if (mytok == TK_BODY && st_cb->s_name != comp_name) {
	sym = st_install(NULLSTR,K_BLOCK);
	sym->s_type = T_SPEC;
    }
}


static void
res_initial()
{
    res_initial_final(T_INIT,I_INIT,I_INIT_END,TRUE);
}


static void
res_final()
{
    res_initial_final(T_FINAL,I_FINAL,I_FINAL_END,TRUE);
}


/* build a fake initial.  called if there was no user specified initial.  */
static void
res_make_initial()
{
    res_initial_final(T_INIT,I_INIT,I_INIT_END,FALSE);
}


/* generate code for initial or final.
 * parameters tell which things to generate;
 * do_guts tells whether this is a real initial/final
 * and should do proc guts, or if we are making an initial.
 */
/* note well:
 * I_INIT (I_FINAL, I_PROC) must be emitted before the I_BLOCK!!!!
 * and I_INIT_END must be emitted after I_BLOCK_END
 * in the below.
 */
/* initial and final are handled just like a proc.
 * in particular, we make a symbol table operation for a (fake)
 * initial/final operation.
 * such uniformity makes the code generator's job easier.
 */

static void
res_initial_final(block_type,ic_start,ic_end,do_guts)
Type block_type;
Icode ic_start, ic_end;
Bool do_guts;
{
    int old_return_label;
    Symptr block_sym, sym;

    assert((block_type==T_INIT && ic_start==I_INIT && ic_end==I_INIT_END)
	|| (block_type==T_FINAL && do_guts &&
	    ic_start==I_FINAL && ic_end==I_FINAL_END));

    sym = new_symbol(K_NOTAKIND);
    emit(ic_start, idnode(sym), NOLAB);

    block_sym = block_begin(block_type);
    block_sym->s_tdef = sym;

    if (do_guts) {
	old_return_label = return_label; return_label = NEWLAB;
	proc_guts(block_type==T_INIT?TK_INITIAL:TK_FINAL,NULLSTR);
	emit(I_LABEL,NULLNODE,return_label);
	return_label = old_return_label;
    }

    block_end();
    emit(ic_end,NULLNODE,NOLAB);
}


				/* 3. Procs - Processes and Procedures */
static void
proc()
{
    Symptr proc_sym, block_sym;
    int old_return_label;

    mustbe(TK_IDENTIFIER,"proc_identifier");

    if ((proc_sym = st_lookup(tk_str)) == NULLSYM) {
	ERROR(E_FATAL+1,tk_str);
	find_follow(fl_proc);
	return;
    }
    if (proc_sym->s_kind != K_OP || 
	(proc_sym->s_type != T_FUNC && proc_sym->s_type != T_VOID)) {
	    FATAL("bad proc identifier");
	    return;
    }
    if (proc_sym->s_ranges)
	FATAL("proc cannot service array op");

    switch (proc_sym->s_impl) {
	case IM_NOTYETIMPL:
	    proc_sym->s_impl = IM_PROC;
	    break;
	case IM_PROC:
	    FATAL("op already implemented by proc");
	    break;
	case IM_INPUT:
	    FATAL("op already implemented by input statement(s)");
	    break;
	default:
	    boom("bad impl in proc");
	    /*NOTREACHED*/
    }

    emit(I_PROC,idnode(proc_sym),NOLAB);

    block_sym = block_begin(T_PROC);
    block_sym->s_tdef = proc_sym;

    formal_ids(block_sym);

    old_return_label = return_label;
    return_label = NEWLAB;
    proc_guts(TK_IDENTIFIER,proc_sym->s_name);
    emit(I_LABEL,NULLNODE,return_label);
    return_label = old_return_label;
    block_end();
    emit(I_PROC_END,NULLNODE,NOLAB);
}


/* process is an abbreviation for a special kind of proc.
 * so, do some stuff like op declaration and then stuff like proc.
 */
static void
process()
{
    Symptr process_sym,block_sym;
    int old_return_label;

    mustbe(TK_IDENTIFIER,"process_identifier");

    /* simulate op_declr() (and op_spec() and op_restrict()).
     * note: no parameters, so s_tdef should be set to null,
     * which it was when symbol table entry was made.
     */
    if (!(process_sym = st_install(tk_str,K_OP))) {
	find_follow(fl_proc);
	return;
    }

    process_sym->s_type = T_VOID;
    process_sym->s_segment = S_NOTASEGMENT;
    process_sym->s_restrict = R_SEND;
    process_sym->s_impl = IM_PROCESS;

    /* do we need to do a new_class? probably not, but we will to be safe.  */
    new_class(process_sym);

	/* do implicit send(s) to create an instance(s) of the process.
	 */
		send_to_process(process_sym);

    emit(I_PROC, idnode(process_sym), NOLAB);

    block_sym = block_begin(T_PROC);
    block_sym->s_tdef = process_sym;

	/* seed symbol table with the process's params,
	 * much in the same way that formal_ids does.
	 */
	{
		Symptr param_sym, formal_sym;
		if ((param_sym = process_sym->s_tdef) != NULLSYM)
			param_sym = param_sym->s_next;
		while (param_sym != NULLSYM) {
			if ((formal_sym=st_install(param_sym->s_name,K_PARAM))
			   ==NULLSYM)
				boom("dup param for process");
			copy_from_decl(formal_sym,param_sym);
/* should we
			formal_sym->s_used = TRUE;
 * probably not, although error messages look a bit wierd
 * since the quantifier bound variables are used.
 */
			param_sym = param_sym->s_next;
		}
	}

    old_return_label = return_label; return_label = NEWLAB;
    proc_guts(TK_IDENTIFIER,process_sym->s_name);
    emit(I_LABEL,NULLNODE,return_label);
    return_label = old_return_label;
    block_end();
    emit(I_PROC_END,NULLNODE,NOLAB);
}

/* generate icode to simulate invocation:  send process_sym()
 * so as to create instances of the process when the resource is created.
 * handles quantifiers if present.
 */
static void
send_to_process(process_sym)
Symptr process_sym;
{
	Nodeptr n;
	Symptr quant_block;
	Instptr ic_quantifier_guts, ic_before;

	/* generated icode gets put on special ic list. */
		emit(I_PROCESS_SEND, NULLNODE, NOLAB);

	/* parse and generate code for the quantifier if present.
	 * also copy quantifier bound variables to make them parameters
	 * for this op (i.e., process).
	 */
		quant_block = NULLSYM;
		ic_quantifier_guts = NULLINST;
		if (maybe(TK_LEFTPAREN)) {
		    if (maybe(TK_RIGHTPAREN)) { /* common error */
			WARN("() in process heading is missing quantifier");
		    }
		    else {
			quant_block = block_begin(T_CMD);
			ic_quantifier_guts = quantifier(FALSE);
			mustbe(TK_RIGHTPAREN,") in process's quantifier");
			copy_quant(quant_block,process_sym);
		    }
		}

	/* generate code for a single invocation of process_sym. */
		ic_before = ic_mark();

		/* build up TK_SEND node, as denotation would do.
		 * (like call_send_stmt)
		 */
			n = bnode(TK_INVOKE, idnode(process_sym),
				  process_args(quant_block));
			n = bnode(TK_SEND,n,NULLNODE);
			call_send(n);

		/* if there was a (valid) quantifier,
		 * patch the invocation into the inner most loop
		 * of the quantifier.
		 */
			if (ic_quantifier_guts != NULLINST){
				ic_move(ic_before,ic_quantifier_guts);
			}

		/* done with the quantifier block if there is one. */
			if (quant_block != NULLSYM) {
				block_end();
			}

	/* end of generated code for process. */
		emit(I_PROCESS_SEND_END, NULLNODE, NOLAB);

	/* simulate side effect of invocation.
	 * note: operation hasn't been seen before,
	 * so needn't worry about promoting its usage.
	 */
		assert(process_sym->s_invoked == R_NOTARESTRICT);
		promote_invoked(process_sym,R_SEND);
}

/* copy quantifier bound variables from the quantifier block
 * into parameter block for process_sym.
 */
static void
copy_quant(quant_block_sym, process_sym)
Symptr quant_block_sym, process_sym;
{
	Symptr b, t;

	assert(quant_block_sym && process_sym);

	/* quit if not at least one bound variable.
	 * should be, but if error occurred above, maybe not.
	 */
		if (quant_block_sym->s_next == NULLSYM) return;

	/* create new name space for parameters (ala op declaration) */
		process_sym->s_tdef = new_symbol(K_NOTAKIND);

	/* install parameters. */			
		for (b = quant_block_sym->s_next; b; b = b->s_next) {
			t = at_install(process_sym->s_tdef,b->s_name,K_PARAM);
			copy_from_decl(t,b);
			t->s_restrict = R_VAL;
		}
}

/* build arguments -- i.e., the quantifier variables -- for the invocation
 * of the process just as denotation (and farg) would do.
 */
static Nodeptr
process_args(quant_block)
Symptr quant_block;
{
	Nodeptr top, end, q;
	Symptr quant_sym;
	top = NULLNODE;
	if ((quant_sym = quant_block) != NULLSYM)
		quant_sym = quant_sym->s_next;
	while (quant_sym != NULLSYM) {
		q = idnode(quant_sym);
		if (top == NULLNODE) {
			top = end = bnode(TK_LIST,q,NULLNODE);
		}
		else {
			end->e_r = bnode(TK_LIST,q,NULLNODE);
			end = end->e_r;
		}
		quant_sym = quant_sym->s_next;
	}
	return (top);
}

/* parsing and semantics for formal identifiers.
 * look up their types in symbol table,
 * make sure right number, including return 'parameter',
 * and return present iff T_FUNC.
 * and seed the symbol table for this block with this info.
 * block_sym points to the block entry in st for the proc or input statement.
 * we assume that nothing comes after block_sym.
 * note: this code should be able to be used
 * for both input statements and procs.
 */
void
formal_ids(block_sym)
Symptr block_sym;
{
    Symptr impl_sym;	/* symbol table entry of op being implemented. */
    register Symptr op_sym, formal_sym;

    if ((impl_sym = block_sym->s_tdef) == NULLSYM) {
	/* just in case.
	 * try to keep going by making up an op symbol
	 * with no parameters and no return value.
	 * might want to free this at end of routine.
	 * note: assume that code generator won't get called
	 * with this phony symbol.
	 * maybe better to skip some of below instead of this
	 * so get less inaccurate error messages.
	 */
	impl_sym = new_symbol(K_OP);
	impl_sym->s_type = T_VOID;
    }

    mustbe(TK_LEFTPAREN,"(");
    if (!maybe(TK_RIGHTPAREN)) {
	do {
	    mustbe(TK_IDENTIFIER, "formal id");
	    if (st_install(tk_str,K_PARAM) == NULLSYM)
		WARN("dup formal id");
	    /* common mistakes.
	     * try to give types or array in list.
	     * skip over stuff.
	     */
	    if (maybe(TK_COLON) || maybe(TK_LEFTBKET)) {
		WARN("op heading can only contain id's (no types or ranges)");
		find_follow(fl_par);
	    }
	} while (maybe(TK_COMMA));

	mustbe(TK_RIGHTPAREN,") in formal list");
    }

    if (maybe(TK_RETURNS)) {
	mustbe(TK_IDENTIFIER,"formal return id");
	if (st_install(tk_str,K_RESULT) == NULLSYM)
	    WARN("dup formal id");
	if (impl_sym->s_type != T_FUNC)
	    WARN("return for a void op");
    } else {
	if (impl_sym->s_type != T_VOID)
	    WARN("missing return for a func op");
    }

    /* Make sure number of parameters (including return value)
     * match the op declaration (which is in an auxiliary table),
     * and fill in symbol table with info from op declaration.
     */
    op_sym = impl_sym->s_tdef;
    if (op_sym != NULLSYM)
	op_sym = op_sym->s_next;
    formal_sym = block_sym->s_next;
    while (op_sym != NULLSYM && formal_sym != NULLSYM) {
	copy_from_decl(formal_sym,op_sym);
	op_sym = op_sym->s_next;
	formal_sym = formal_sym->s_next;
    }
    if (op_sym != NULLSYM) {
	FATAL("too few formals");
	return;
    } else if (formal_sym != NULLSYM) {
	FATAL("too many formals");
	return;
    }
}


static void
proc_guts(proc_tok,name)
Token proc_tok;
char *name;
{
    static Token follow[] = {TK_END,TK_NOTATOKEN};

    command_block(follow);
    assert (tok == TK_END);
    optional_end_follower(proc_tok,name);
}


/* new command block.
 * used by do and if because they need a new block.
 * proc and input don't because they start their
 * own block that contains the parameters.
 */
void
new_command_block(follow)
Token *follow;
{
    block_begin(T_CMD);
    command_block(follow);
    block_end();
}


/* parses command_block.
 * follow is what should follow the statements;
 * e.g., TK_FI or TK_SQUARE for the statements in an arm of an if statement.
 */
void
command_block(follow)
Token *follow;
{
    Bool warned = FALSE;	/* to suppress duplicate unreachable errors. */
    Bool old_unreachable;
    Symptr old_reply_block;

    old_unreachable = unreachable;
    unreachable = FALSE;

    /* save reply_block, except for T_CMD's.
     * note: might be better to save as part of block_begin
     * and restore as part of block_end,
     * but then would need explicit stack.
     */
    assert (st_cb->s_kind == K_BLOCK);
    switch (st_cb->s_type) {
	case T_INIT:
	case T_FINAL:
	case T_PROC:
	case T_INPUT:
	    old_reply_block = reply_block;
	    reply_block = st_cb;
	    break;
	case T_CMD:
	    /* do nothing. */
	    break;
	default:
	    boom("bad st_cb block type in command_block");
	    /*NOTREACHED*/
	    break;
    }

    while (!is_follow(follow)) {
	if (unreachable && !warned) {
	    WARN("unreachable code after exit, next, stop, or return");
	    warned = TRUE;
	}
	statement();
	/* skip over optional semicolon as terminator. */
	maybe(TK_SEMICOLON);
    }

    /* skip over the statement list terminator that is_follow put back. */
    get_token();

    /* check for unused identifiers. */
		check_used();

    /* restore reply_block, except for T_CMD's.  */
    assert (st_cb->s_kind == K_BLOCK);
    switch (st_cb->s_type) {
	case T_INIT:
	case T_FINAL:
	case T_PROC:
	case T_INPUT:
	    reply_block = old_reply_block;
	    break;
	case T_CMD:
	    /* do nothing. */
	    break;
	default:
	    boom("bad st_cb block type in command_block");
	    /*NOTREACHED*/
	    break;
    }

    unreachable = old_unreachable;
}


/* block_begin(type)
 * makes a new block in the symbol table,
 * tells code generator about it,
 * returns pointer to new block.
 * used for procs, initial/final, do, co, quantifiers, in,
 * but not for component (resource) level blocks.
 */
Symptr
block_begin(type)
Type type;
{
    Symptr block_sym;

    assert (type == T_PROC || type == T_INIT || type == T_FINAL
	    || type == T_CMD || type == T_INPUT);

    block_sym = st_install(NULLSTR, K_BLOCK);
    block_sym->s_type = type;

    /* tell the code generator that we're entering a block.
     * and give it a pointer to it. (Yup, pretty kludgy.)
     */
    assert (block_sym == st_cb && st_cb != NULLSYM && st_cb->s_kind == K_BLOCK);
    emit(I_BLOCK, snode(TK_BLOCK,st_cb,NULLNODE), NOLAB);
    return (block_sym);
}


/* block_end()
 * pop off the block and tell code generator did so.
 */
void
block_end()
{
    pop_block();
    emit(I_BLOCK_END,NULLNODE,NOLAB);
}



/*
 * parses optional following token after end.
 * for resource, proc, process, it is the id.
 * for initial, final, it is the keyword.
 * tok might not be TK_END when called after fake_res_body.
 */
static void
optional_end_follower(which, name)
Token which;
char *name;
{
    Token mytok;
    if (tok == TK_END) {
	mytok = get_eof_token();
	if (mytok == which) {
	    if (which == TK_IDENTIFIER && strcmp(name,tk_str))
		errmsg(E_WARN,
		    "identifier following end doesn't match: %s",name);
	} else
	    putback();
    }
}
