/*  class.c - routines that deal with equivalence classes of operations  */

#include <stdio.h>
#include "../util.h"
#include "sr.h"
#include "funcs.h"
#include "globals.h"

static void combine_class(), combiner(), class_release();
static Classptr class_create();


/*  the overall idea:
 *
 *  when see the declaration of an operation, put it in its own equivalence
 *  class.  later, when see operations in an input statement, combine their
 *  equivalence classes into one.
 *
 *  this is done on a per resource basis; the combining of classes is done by
 *  running through all input statements at the end of the resource body.
 *
 *  when an operation is created, the RTS is given the runtime class-id of the
 *  operation.  thus, the class must exist before an operation in the class is
 *  created.  the code generator takes care of this kind of stuff.
 */


static Classptr class_free = NULLCLASS;		/* free list of Classes */


/* class_check()
 * called at the end of a resource body to:
 * 	get rid of proc and process operation classes.
 *	make sure all operations are implemented.
 *	implement externals.
 */
void
class_check()
{
    Classptr cl, next_cl;

    class_print();

    for (cl = classes; cl != NULLCLASS; cl = next_cl) {
	next_cl = cl->cl_next;
	assert (cl->cl_first != NULLSYM);
	
	switch (cl->cl_first->s_impl) {
	case IM_NOTYETIMPL:
	    errmsg(E_FATAL,"declared op not implemented: %s",
		cl->cl_first->s_name);
	    break;
	case IM_PROC:
	case IM_PROCESS:
	case IM_EXTERNAL:
	    assert(! cl->cl_first->s_cl_next);
	    class_release(cl);
	    break;
	case IM_INPUT:
	    break;
	default:
	    boom("bad impl in class_check");
	    /*NOTREACHED*/
	}
    }
    class_print();
}



/* equiv()
 * called at end of a resource body.
 * for each input statement,
 * extends the equivalence relation by combining classes
 * of all operations that are present in the input operation.
 */
void
equiv()
{
    Inptr in;
    Symptr first;
    Legptr leg;

    for (in = inlist(); in != NULLIN; in = in->in_next) {
	leg = in->in_legs;
	first = leg->leg_sym;
	for (leg = leg->leg_next; leg != NULLLEG; leg = leg->leg_next) {
	    /* if two different lists, combine them into one. */
	    assert(first->s_class!=NULLCLASS &&
		   leg->leg_sym->s_class!=NULLCLASS);
	    if (first->s_class != leg->leg_sym->s_class)
		combine_class(first->s_class, leg->leg_sym->s_class);
	}
    }
}

/* combine_class(cl1, cl2)
 * combine classes cl1 and cl2 into a single class.
 * we keep the larger class and
 * get rid of the smaller class.
 * this is more efficient because we need to update class pointers
 * in the changed class.
 */
static void
combine_class(cl1,cl2)
Classptr cl1, cl2;
{
    assert (cl1 != cl2);
    assert (cl1->cl_first != NULLSYM && cl1->cl_last != NULLSYM);
    assert (cl2->cl_first != NULLSYM && cl2->cl_last != NULLSYM);

    if (cl1->cl_members > cl2->cl_members)
	combiner(cl1, cl2);
    else
	combiner(cl2, cl1);
}

/* combiner(large, small)
 * combines small into large.
 */
static void
combiner(large, small)
Classptr large, small;
{
    Symptr sym;

    /* change class pointers in small to point to large. */
    for (sym = small->cl_first;	sym != NULLSYM;	sym = sym->s_cl_next)
	sym->s_class = large;

    /* link small list into large. */
    large->cl_last->s_cl_next = small->cl_first;
    large->cl_last = small->cl_last;

    /* update number of members. */
    large->cl_members += small->cl_members;

    /* free up the small class. */
    class_release(small);

    assert (classes != NULLCLASS);
}

/* new_class(sym)
 * create a new class for operation sym upon seeing its declaration.
 */
void
new_class(sym)
Symptr sym;
{
    Classptr cl;

    assert(sym != NULLSYM);
    cl = class_create();
    cl->cl_members = 1;
    cl->cl_first = cl->cl_last = sym;
    cl->cl_refcnt = -1;
    sym->s_class = cl;
}

/*
  class_spec_ops()
 * called at the start of a separate body
 * to do new_class on each operation in the spec.
 */
void
class_spec_ops()
{
    Symptr sym;
    assert (st_cb->s_kind == K_BLOCK && st_cb->s_type == T_SPEC);
    assert (st_cb->s_tdef != NULLSYM);

    for (sym = st_cb->s_tdef->s_next; sym != NULLSYM; sym = sym->s_next) {
	if (sym->s_kind == K_OP)
	    new_class(sym);
    }
}

/* allocate a Class structure
 * and insert it on the class list.
 */
static Classptr
class_create()
{
    register Classptr t;

    /* allocate. */
    if (class_free) {
	t = class_free;
	class_free = class_free->cl_next;
    } else
	t = (Classptr)alloc(sizeof(Class));

    assert(t != NULLCLASS);

    /* link it into class list. */
    if (classes != NULLCLASS) 
	classes->cl_prev = t;
    t->cl_next = classes;
    t->cl_prev = NULLCLASS;
    classes = t;

    return (t);
}

/* free a class structure and link it into the free list.
 * note: of course, the caller should not use any of its fields
 * (e.g., link fields) after it has been released.
 */
static void
class_release(cl)
Classptr cl;
{
    /* if cl is the first class,
     * then update classes to point to the next.
     */
    if (classes == cl)
	classes = classes->cl_next;

    /* update links of neighbors. */
    if (cl->cl_prev != NULLCLASS)
	cl->cl_prev->cl_next = cl->cl_next;
    if (cl->cl_next != NULLCLASS)
	cl->cl_next->cl_prev = cl->cl_prev;

    cl->cl_next = class_free;
    class_free = cl;
}

/* initialize the class list and the input list.
 * called at the beginning of each resource (spec or body).
 * frees any classes left from last time.
 */
void
class_init()
{
    Classptr cl, next_cl;

    for (cl = classes; cl != NULLCLASS; cl = next_cl) {
	next_cl = cl->cl_next;
	class_release(cl);
    }
    assert (classes == NULLCLASS);
}

