/* callret.c */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"
#include "../srsys.h"
#include "../util.h"

static void local_call();
static Symptr copy_back_args(), get_par_list();

typedef struct inv_info {
	int pb_addr,
	    argsize,
	    quantsize;
	struct inv_info *next_inv;
	struct inv_info *next_arm;
	} Inv_info; 
	    
static int
	co_sp = 0,
	co_pb_addr[MAX_NEST];

static Inv_info
	info_stack[MAX_NEST],
	*this_arm[MAX_NEST] = { &info_stack[0] };


/* generate call to invoke an op.  */   
void
gen_invoke(exp, use_result, arm_number)
Nodeptr exp;
Result use_result;
int arm_number;
{
    register Nodeptr inv, func;
    register Inv_info *info;
    int tsize,blsize;
    Token inv_type;
    Symptr func_sym;
    
    inv = exp->e_l;
    assert(inv->e_op == TK_INVOKE);
    
    func = inv->e_l;
    inv_type = exp->e_op;

    /* check for predefined functions, semaphores, and local function calls */
    if (func->e_op == TK_IDENTIFIER) {
	int asize;
	Symptr fsym = func->e_s;
	if (fsym->s_kind == K_PREDEF) {
	    if (inv_type == TK_SEND)
		FATAL("shouldn't send to a predefined op");
	    gen_predef(fsym, inv->e_r);
	    return;
	} else if (fsym->s_kind == K_SEMAPHORE) {
	    assert(inv_type == TK_SEND);
	    cprintf("V(A(%A))",fsym);
	    return;
	} else if (inv_type == TK_CALL &&
		   fsym->s_kind == K_OP &&
		   (fsym->s_impl == IM_PROC || fsym->s_impl == IM_EXTERNAL) &&
		   ! fsym->s_reply &&
		   (asize = known_size_args(fsym->s_tdef)) != SIZE_UNK) {
	    local_call(exp, fsym, asize, inv->e_r, use_result);
	    return;
	}
    }
    
    
    switch (inv_type) {
    case TK_CO_CALL:
    case TK_CO_SEND:
	/* put a new arm on list */
	this_arm[co_sp]->next_arm=
		(Inv_info *)alloc(sizeof(Inv_info));
	this_arm[co_sp] = this_arm[co_sp]->next_arm;
	this_arm[co_sp]->next_arm = 
		this_arm[co_sp]->next_inv = (Inv_info *)0;
	
	/* push an info block on current arm */
	info = (Inv_info *)alloc(sizeof(Inv_info));
	info->next_inv = this_arm[co_sp]->next_inv;
	this_arm[co_sp]->next_inv = info;
	info->pb_addr = co_pb_addr[co_sp];
	break;
    case TK_CALL:
    case TK_SEND:
	/* push an info block on current arm */
	info = (Inv_info *)alloc(sizeof(Inv_info));
	info->next_inv = this_arm[co_sp]->next_inv;
	this_arm[co_sp]->next_inv = info;
	info->pb_addr = temp_alloc(INTSIZE);
	break;
    default:
	boom("bad tree in gen_invoke");
	break;
    }

    func_sym = get_par_list(func);
    
    info->quantsize = align(static_size(exp->e_r));
    blsize = temp_alloc(INTSIZE);
    info->argsize = 
	    pb_alloc(func_sym, 
		     inv->e_r,
		     exp,
		     INVOCATION_HEADER_SIZE + info->quantsize,
		     info->pb_addr,
		     PROG_OWN,blsize);
    tsize = INVOCATION_HEADER_SIZE + info->argsize + info->quantsize;
    
	
    /* fill in quantifiers */
	{   int off, qsz;
	    Nodeptr q;
	    off = INVOCATION_HEADER_SIZE + info->argsize;
	    for (q=exp->e_r; q; q=q->e_r) {
		qsz = get_size(q->e_l);
		assert(qsz==INTSIZE);
		cprintf(",\nI(LA(%d)+%d)=%e",info->pb_addr,off,q->e_l);
		off += qsz;
	    }
	}
    
    /* fill in header */
	inv_header(info->pb_addr, func, inv_type, arm_number,blsize);
	
    /* fill in args */
	copy_list(func_sym, inv->e_r, exp, info->pb_addr, tsize);
	
    /* call invoke */
	cprintf(",\n");
	if (inv_type == TK_CALL)
	    cprintf("LA(%d)=",info->pb_addr);
	cprintf("sr_invoke(LA(%d))",info->pb_addr);
    
    /* copy out res and var parameters and quantifiers */
	switch (inv_type) {
	    case TK_CALL:
		gen_copy_back(exp, use_result, TRUE);
		break;
	    case TK_SEND:
		this_arm[co_sp]->next_inv = info->next_inv;
		break;
	    case TK_CO_CALL:
	    case TK_CO_SEND:
		break;
	}
}


/*  Generate code to
 *	1) copy back quantifier vars
 *	2) copy back args
 *	3) leave address of function result on the stack
 *	4) deallocate block
 *  for CO_SEND, only do 1 and 4.
 */
void
gen_copy_back(exp, use_result, free_block)
Nodeptr exp;
Result use_result;
Bool free_block;
{
    int sz;
    int off;
    Nodeptr a, actual, inv;
    Symptr f;
    Inv_info *info, *temp;
    
    switch (exp->e_op) {
    case TK_SEND:
	return;
    case TK_CALL:
	info = this_arm[co_sp]->next_inv;
	this_arm[co_sp]->next_inv = info->next_inv;
	break;
    case TK_CO_CALL_COPY_BACK:
    case TK_CO_SEND_COPY_BACK:
	temp = info_stack[co_sp].next_arm;
	info = temp->next_inv;
	info_stack[co_sp].next_arm = temp->next_arm;
	break;
    default:
	boom("bad tree in gen_copy_back");
    }
    
    inv = exp->e_l;
    assert(inv->e_op == TK_INVOKE);
    
    /* copy back quantifiers.  note that quantifiers must be "nice"
       types like int, enum, char, bool
    */
    off = INVOCATION_HEADER_SIZE + info->argsize;
    for (a = exp->e_r; a; a=a->e_r) {
	actual = a->e_l;
	assert(assignable(actual->e_sig));
	sz = get_size(actual);
	cprintf(",\n%e=I(LA(%d)+%d)",actual,info->pb_addr,off);
	off += align(sz);
    }
    
    if (exp->e_op != TK_CO_SEND_COPY_BACK) {
	f = get_par_list(inv->e_l);
	f = copy_back_args(inv->e_r, f, info->pb_addr);
	gen_result(f, info->pb_addr, use_result);
    }    

    if (free_block)
	free_later(info->pb_addr);
}

/* generate comma, then expr list to copy back "var" and "result" args.
 * return the formal for the function result. */
static Symptr
copy_back_args(a, f, pb_addr)
Nodeptr a;			/* actual param list */
Symptr f;			/* formal param list */
int pb_addr;			/* offset of block address */
{
    struct assign_descriptor desc;
    Nodeptr actual;
    
    for (;  a && f;  a=a->e_r, f=f->s_next) {
	actual = a->e_l;
	if (f->s_restrict != R_VALRES && f->s_restrict != R_RES)
	    continue;
	if (! contiguous(&desc, actual)) {
	    ERROR(E_FATAL+4, "non-contiguous var/res parameter");
	} else if (! is_lvalue(actual) || is_a_const(actual)) {
	    WARN("lvalue required for var or res parameter");
	} else {
	    cprintf(",");
	    gen_assign(a->e_l, make_formal(f,pb_addr));
	}
    }
    return f;
}

/* generate result of function if wanted */
void
gen_result(f, pb_addr, use_result)
Symptr f;
int pb_addr;
Result use_result;
{
    if (use_result == NO_RESULT)
	return;
    assert(f);
    cprintf(",");
    if (use_result == VAL_RESULT) {
        if (f->s_type == T_CHAR || f->s_type == T_BOOL)
            cprintf("*");
        else
            cprintf("I");
    }
    if (f->s_offset != OFF_UNK)
	cprintf("(LA(%d)+%d)",pb_addr,f->s_offset);
    else
	cprintf("(LA(%d)+I(LA(%d)+%d))",pb_addr,pb_addr,f->s_desoff+AD_ADDR);
}

	      
void
gen_co_start()
{
    cprintf("sr_co_start();\n");
    
    ++ co_sp;
    if (co_sp == MAX_NEST) {
	boom("co statement nested too deeply");
    }
    this_arm[co_sp] = &info_stack[co_sp];
    co_pb_addr[co_sp] = temp_alloc(INTSIZE);
}

/* generate the code at the end of a ppc arm.  i.e., branch to the 
   co_wait label
*/
void
gen_ppc_end(lab)
int lab;
{
    cprintf("goto %g;\n",lab);
}




/* generate code to 
	1) call co-wait
	2) check for termination
	3) do a case-jump on the basis of the arm number 
*/

void
gen_co_wait(lab)
int lab;
{
    /* call co-wait, which returns the address of the invocation block */
    /* check for termination -- co_wait returns 0) */
	wlab(lab);
	cprintf("if(LA(%d)=sr_co_wait())",co_pb_addr[co_sp]);
    /* extract the arm number */
    /* do the case-jump */
	cprintf("switch(((invb)LA(%d))->co.arm_num){\n",co_pb_addr[co_sp]);
}

void
gen_co_end(lab)
int lab;
{
    wlab(lab);
    cprintf("sr_co_end();\n");
    --co_sp;
}


/* generate a jump table (now a case statement) */

static int jcase;


void
gen_jt()
{
    jcase = 0;
}


void
add_jt(lab)
int lab;
{
    if (lab)
	cprintf("case %d:goto %g;\n",jcase++, lab);
    else
	cprintf("  /* %d:  ;  */\n",jcase++);
}


void
end_jt()
{
    cprintf("};\n");
}





static Symptr
get_par_list(e)
Nodeptr e;
{
    Symptr s;
    
    switch (e->e_op) {
    case TK_IDENTIFIER:
	s = e->e_s;
	switch (s->s_kind) {
	case K_OP:
	    if (!s->s_tdef)
		return (Symptr)0;
	    return s->s_tdef->s_next;
	case K_VAR:
	case K_FIELD:
	case K_RESULT:
	case K_PARAM:
	    assert(s->s_tdef);
	    if (!s->s_tdef->s_tdef || !s->s_tdef->s_tdef->s_tdef)
		return (Symptr)0;
	    return s->s_tdef->s_tdef->s_tdef->s_next;
	default:
	    boom("bad kind in get_par_list");
	    /* NOTREACHED */
	}
    case TK_INDEX:
	return get_par_list(e->e_l);
    case TK_PERIOD:
	return get_par_list(e->e_r);
    default:
	boom("bad tree in get_par_list");
	/* NOTREACHED */
    }
}


/* if sizes of all parameters are known, return sum; otherwise SIZE_UNK */
int
known_size_args(alist)
Symptr alist;
{
    int sum = 0;
    Symptr s;
    for (s=alist; s; s=s->s_next) {
	if (s->s_kind != K_PARAM && s->s_kind != K_RESULT)
	    continue;
	if (s->s_size > 0) {
	    sum += align(s->s_size) + s->s_dessize;
	} else
	    return SIZE_UNK;
    }
    return sum;
}


/* generate code to do a "local" function call without the rts */
static void
local_call(exp, func, asize, args, use_result)
Nodeptr exp;
Symptr func;
int asize;
Nodeptr args;
Result use_result;
{
    int bl, pb_add;
    Symptr f;

    if (asize > 0) {
	/* store location of parameter block at pb_add.  Note that this param
	   block is a little bogus --- the header isn't really there */
	bl = temp_alloc(asize);
	pb_add = temp_alloc(PTRSIZE);
	cprintf("LA(%d)=lv+%d",pb_add,bl-INVOCATION_HEADER_SIZE);
	copy_list(func->s_tdef->s_next, args, exp, pb_add, 0 /*not used*/);
	cprintf(",\n");
    }

    /* generate the call */
    cprintf("P%s(rp,rv,",func->s_name);
    if (asize > 0)
	cprintf("lv+%d",bl-INVOCATION_HEADER_SIZE);
    else
	cprintf("0");
    cprintf(",%d)",PROG_OWN);

    /* copy back args and gen result if wanted */
    f = func->s_tdef ? func->s_tdef->s_next : 0;
    if (args) f = copy_back_args(args, f, pb_add);
    gen_result(f, pb_add, use_result);
}



/* produce code to call a C function */
void
gen_ext(op)
Symptr op;
{
    Symptr b, r, s;
    char t[20];

    /* find symbol and type of result, if any */
    r = NULLSYM;
    if (s = op->s_tdef)
	while (s = s->s_next)  
	    if (s->s_kind == K_RESULT)  {
		b = r = s;			/* save result symbol */
		if (b->s_type == T_PTR)
		    b = b->s_tdef;
		switch (b->s_type) {
		    case T_INT:    strcpy(t,"int ");    break;
		    case T_ENUM:   strcpy(t,"int ");    break;
		    case T_CHAR:   strcpy(t,"char ");   break;
		    case T_BOOL:   strcpy(t,"char ");   break;
		    case T_PTR:    strcpy(t,"char *");  break;
		    case T_FILE:   strcpy(t,"FILE *");  break;
		    case T_STRING: strcpy(t,"char *");  break;
		    default:       strcpy(t,"char ");   break;	/* for ptr xx */
		}
		if (r->s_type == T_PTR)
		    strcat(t," *");
		if (b->s_ranges)
		    strcat(t," *");
		break;
	    }

    /* generate prologue */
    gen_entry_code('P',op->s_name);
    if (r)
	cprintf("%sresult;\n%s%s();\n",t,t,op->s_name);
    cprintf("rp=pb;\n");	/* kludge to correct incorrect segment */

    /* insert '\0' character in strings to terminate them for C */
    if (s = op->s_tdef)
	while (s = s->s_next)
	    if (s->s_kind == K_PARAM && s->s_restrict != R_RES
		    && s->s_type == T_STRING && !s->s_ranges)
		cprintf("*(%A+%d+I(%A))='\\0';\n",s,INTSIZE,s);

    /* call the C function */
    if (r)
	cprintf("result=");
    cprintf("%s(",op->s_name);
    if (s = op->s_tdef)
	while (s = s->s_next)
	    if (s->s_kind == K_PARAM) {
		if (!s->s_ranges) switch (s->s_type) {
		    case T_INT:
		    case T_ENUM:
		    case T_CHAR:
		    case T_BOOL:
		    case T_PTR:
			cprintf(",%r(%A)",s,s);
			break;
		    case T_FILE:
			cprintf(",(FILE*)A(%A)",s);
			break;
		    case T_STRING:
			cprintf(",%A+%d",s,INTSIZE);
			break;
		    default:
			cprintf(",%A",s);
			break;
		} else
		    cprintf(",%A",s);
	    }
    cprintf(");\n");

    /* set length for res string parameters */
    if (s = op->s_tdef)
	while (s = s->s_next)
	    if (s->s_kind == K_PARAM && s->s_restrict != R_VAL
		    && s->s_type == T_STRING && !s->s_ranges)
		cprintf("I(%A)=strlen(%A+%d);\n",s,s,INTSIZE);

    /* return result */
    if (r)
	if (r->s_type == T_STRING && !r->s_ranges)
	    cprintf("sr_str_result(result,%A,%#);\n",r,r);
	else
	    cprintf("%r(%A)=result;\n",r,r);

    gen_free_mem();
    cprintf("if(!wc)sr_finished_proc();\n");
    gen_exit_code(0);
}
