/*
 *  SR Run-Time Support.  Resource Management.
 */

#include "rts.h"

static void destroy_one ();


static rint rid_table;		/* table of resource instance descriptors */
static rint rid_free;		/* header of rint free list */
static sem rid_mutex;

static bool suiciding = FALSE;	/* nonzero if destroy(myvm()) in progress */

/*
 *  Create new instance of a resource.
 *
 *  crbp is assumed to point to an alloc'd block; the block header is altered.
 */
void
sr_create (crbp)				/* RTS Primitive */
crb crbp;
{
    rint res;
    proc pr;
    sem wait;
    memh mp;

    sr_check_stk();

    if (suiciding)
	sr_abort ("attempting to create while VM is being destroyed");

    if (crbp->vm == NULL_VM)
	sr_abort ("attempting to create resource on null machine");

    if (crbp->vm != sr_my_vm && crbp->vm != NOOP_VM) {
	pach ph;
	daddr src, dest;

	/* Send request to remote machine.  */
	ph = sr_remote (crbp->vm, REQ_CREATE, &crbp->ph, crbp->ph.size);

	/* Copy back new resource capability.  */
	if ((dest = (daddr) crbp->rcp) != NULL) {
	    src = (daddr) &((struct crep_st *) ph)->rcap;
	    while (crbp->rc_size--)
		*dest++ = *src++;
	}

	sr_free ((daddr) ph);
	return;
    }

    if (sr_rpat [crbp->rpatid].init == NULL)
	sr_abort ("pattern not loaded on machine.");

    /*
     *	Allocate a new resource instance descriptor.
     */
    if (sr_rid_count->value == 0)
	rts_warn ("low on resource instance descriptors");

    P (sr_rid_count);
    P (rid_mutex);
    res = rid_free;
    rid_free = rid_free->next;
    res->status = 0;
    V (rid_mutex);

    /*
     *	Initialize descriptor.
     */
    res->mutex = sr_make_sem (0);
    res->rpatid = crbp->rpatid;
    res->crb_addr = crbp;
    res->rc_size = crbp->rc_size;
    res->noper = 0;
    res->oper_list = NULL;
    res->procs = NULL;
    res->meml = NULL;
    res->rcp = crbp->rcp;
    /*
     *	Make newly created resource owner of CRB.
     *  We're assuming here that crbp is the address of an alloc'd block.
     */
    mp = (memh) ((daddr) crbp - MEMH_SZ);
    mp->res = res;
    insert (mp, res->meml, rnext, rlast);

    /*
     *	Invoke UC resource entry code as a separate process.
     */
    pr = sr_spawn(sr_rpat[crbp->rpatid].init, res, crbp, 0, crbp->rcp, RTS_OWN);
    pr->wait = wait = sr_make_sem (0);
    pr->ptype = INITIAL;

    /*
     *  Start new process;  returns when initialization finishes.
     */
    V (res->mutex);
    sr_activate (pr);
    P (wait);
    sr_kill_sem (wait);
}



/*
 *  Allocate memory for resource variables.
 *  Initialize resource ID part of myresource capability.
 *  Remember the main resource so we can destroy it at termination time.
 *  Called by initial code after it determines how much memory is needed.
 */
daddr
sr_alloc_rv (size)				/* RTS Primitive */
int size;
{
    rescap *rcp;

    sr_check_stk();

    sr_cur_res->rv_base = sr_own_alloc (size, sr_cur_res);
    rcp = (rescap *) sr_cur_res->rv_base;
    rcp->vm = sr_my_vm;
    rcp->res  = sr_cur_res;
    rcp->seqn = sr_cur_res->seqn;
    if (sr_main_res.res == 0)
	memcpy ((char * ) &sr_main_res, (char *) rcp, sizeof (sr_main_res));
    return (sr_cur_res->rv_base);
}



/*
 *  Called when resource's initial code has been executed.
 *  Copy myresource of new resource to capability.
 *  Remove initial process from list of processes for resource.
 *  Notify creator.
 */
void
sr_finished_init ()				/* RTS Primitive */
{
    daddr src, dest;

    sr_check_stk();

    if (! (sr_cur_res->status & INIT_REPLY)) {
	if ((dest = (daddr) sr_cur_res->rcp) != NULL) {
	    src = sr_cur_res->rv_base;
	    while (sr_cur_res->rc_size--) *dest++ = *src++;
	}

	V (sr_cur_proc->wait);
    }

    sr_kill (sr_cur_proc, TRUE);
}



/*
 *  Create a null or noop resource capability.
 */
void
sr_build_rcap (rcp, size, ocp)			/* RTS Primitive */
rescap *rcp;
int size;
opcap *ocp;
{
    opcap *o;

    *rcp = sr_nu_rcap;			/* base skeleton */
    rcp->seqn = ocp->seqn;		/* set null or noop */
    o = rcp->opcap_list;
    while ((char *)(o + 1) <= (((char *) rcp) + size))
	*o++ = *ocp;			/* fill in opcaps with null or noop */
}



/*
 *  Destroy an instance of a resource.
 */
void
sr_destroy (rcp)				/* RTS Primitive */
rescap *rcp;
{
    rint res;
    proc pr;
    bool kill_me = FALSE;
    sem wait;
    sem wake_creator = NULL;

    sr_check_stk();

    /*
     *	Check for null or noop resource capability.
     */
    if (rcp->res == NULL) {
	if (rcp->seqn == NOOP_SEQN)  return;
	sr_abort ("attempting to destroy null resource");
    }

    if (rcp->vm != sr_my_vm) {
	/* Send destroy request to the remote machine. */
	struct dest_st dr;
	pach ph;

	dr.rc.vm = rcp->vm;
	dr.rc.seqn = rcp->seqn;
	dr.rc.res  = rcp->res;

	ph = sr_remote (rcp->vm, REQ_DESTROY, (pach) &dr, sizeof (dr));
	sr_free ((daddr) ph);
	return;
    }

    DEBUG (8, "r%06X p%06X destroy resource", sr_cur_res, sr_cur_proc, 0);
    res = rcp->res;

    /*
     *	Check the sequence number.  Also invalidate it
     *	so subsequent destroys of same resource fail.
     */
    if (rcp->seqn != res->seqn++) {
	/* raise exception */
	rts_error 
	    ("attempting to destroy resource that no longer exists");
	return;
    }

    P (res->mutex);

    /*
     *	Execute finalization code if there is any.
     */
    if (sr_rpat [res->rpatid].final != NULL) {
	DEBUG (8, "r%06X p%06X exec final", sr_cur_res, sr_cur_proc, 0);
	pr = sr_spawn (sr_rpat [res->rpatid].final, res,
		res->crb_addr, res->rv_base, 0, RTS_OWN);
	
	pr->wait = wait = sr_make_sem (0);
	pr->ptype = FINAL;

	V (res->mutex);
	sr_activate (pr);
	P (wait);
	sr_kill_sem (wait);
	P (res->mutex);
	DEBUG (8, "r%06X p%06X final done", sr_cur_res, sr_cur_proc, 0);
    }

    sr_kill_resops (res);

    /*
     *	Kill all processes belonging to the resource,
     *	but defer killing myself and notifying creator
     *	if the resource has an initial process.
     */
    while (pr = res->procs) {
	res->procs = res->procs->procs;
	/* if (pr->ptype == FINAL)  continue; */

	if (pr->ptype == INITIAL && !(res->status & INIT_REPLY)) {
	    rts_warn ("initial process destroyed");
	    wake_creator = pr->wait;
	}

	if (pr == sr_cur_proc)
	    kill_me = TRUE;
	else
	    sr_kill (pr, FALSE);
    }

    /*
     *	Free memory associated with resource and
     *	return resource instance descriptor to free list.
     */
    sr_res_free (res);

    P (rid_mutex);
    sr_kill_sem (res->mutex);
    res->next = rid_free;
    rid_free = res;
    V (rid_mutex);
    V (sr_rid_count);

    if (wake_creator)  V (wake_creator);

    /*
     *	Commit suicide if the destroyer is part of the
     *	dead resource.
     */
    if (kill_me) {
	DEBUG (8, "r%06X p%06X killing self", sr_cur_res, sr_cur_proc, 0);
	sr_kill (sr_cur_proc, FALSE);
    }
}




/*
 *  Destroy all the resource instances on this machine.
 */
void
sr_dest_all()
{
    tindex i;

    suiciding = TRUE;
    for (i = 1; i < sr_max_resources; i++)
	if (!(rid_table[i].status & FREE_SLOT))
	    sr_activate (sr_spawn (destroy_one,RTS_OWN,&rid_table[i],0,0,0));
}



/*
 *  Destroy one resource.
 */
static void
destroy_one(r)
rint r;
{
    rescap rc;
    rc.vm = sr_my_vm;
    rc.res = r;
    rc.seqn = r->seqn;
    sr_destroy(&rc);
    sr_kill (sr_cur_proc, FALSE);
}



/*
 *  A resources' final code has completed.  Notify destroyer.
 */
void
sr_finished_final ()				/* RTS Primitive */
{
    sr_check_stk();

    if (! (sr_cur_res->status & FINAL_REPLY))
	V (sr_cur_proc->wait);
    
    sr_kill (sr_cur_proc, TRUE);
}



/*
 *  Initialize resource management part of SR RTS.
 */
void
sr_init_res ()
{
    tindex i;

    sr_nu_rcap.vm = 0;		/* create null resource capability */
    sr_nu_rcap.seqn = NULL_SEQN;	/* indicate NULL */
    sr_nu_rcap.res  = 0;		/* rint[0] is never allocated */

    sr_no_rcap.vm = 0;		/* create noop resource capability */
    sr_no_rcap.seqn = NOOP_SEQN;	/* indicate NOOP */
    sr_no_rcap.res  = 0;		/* rint[0] is never allocated */

    sr_init_mem ();

    /*
     *	Initialize free list of resource instance descriptors.
     */
    rid_table = (rint) sr_alloc (sr_max_resources * sizeof (struct rint_st));
    for (i = 1 ; i < sr_max_resources-1 ; i++) {
	(rid_table + i)->next = rid_table + i + 1;
	(rid_table + i)->seqn = INIT_SEQ_RES;
	(rid_table + i)->status = FREE_SLOT;
    }
    
    (rid_table + i)->next = NULL;
    (rid_table + i)->seqn = INIT_SEQ_RES;
    (rid_table + i)->status = FREE_SLOT;

    rid_free  = rid_table + 1;		/* res 0 never used */
    sr_rid_count = sr_make_sem (sr_max_resources-1);
    rid_mutex = sr_make_sem (1);
}
