/*
 *  SR Run-Time Support.  Remote Request Processing Routines.
 */

#include "rts.h"

static void contact();

static remd Remd;		/* remote request mesg descriptors */
static remd rem_free;		/* header of remd free list */
static sem rem_mutex;

/*
 *  Initialize free list of pending remote request message descriptors.
 */
void
sr_init_rem ()
{
    tindex i;

    Remd = (remd) sr_alloc (sr_max_rmt_reqs * sizeof (struct remd_st));
    for (i = 0 ; i < sr_max_rmt_reqs-1 ; i++)
	Remd [i].next = &Remd [i+1];
	
    Remd [i].next = NULL;
    
    rem_free  = &Remd [0];
    rem_mutex = sr_make_sem (1);
    sr_rem_count = sr_make_sem (sr_max_rmt_reqs);
}



/*
 *  Send a request to a remote machine and wait for the reply message.
 *  Return pointer to reply packet.
 */
pach
sr_remote (dest, type, ph, size)
tindex dest;
enum ms_type type;
pach ph;
short size;
{
    remd rem;

    if (!(sr_net_known((int) dest)))
	contact((int) dest);	/* establish contact if none yet made */

    if (sr_rem_count->value == 0)
	rts_warn ("low on remote request blocks");

    P (sr_rem_count);
    P (rem_mutex);
    rem = rem_free;
    rem_free = rem_free->next;
    V (rem_mutex);

    ph->origin = sr_my_vm;
    ph->rem = rem;
    rem->ph = ph;
    rem->wait = sr_make_sem (0);

    sr_net_send (dest, type, ph, size);
    P (rem->wait);		/* wait for reply */
    sr_kill_sem (rem->wait);
    ph = rem->reply;
    
    P (rem_mutex);
    rem->next = rem_free;
    rem_free = rem;
    V (rem_mutex);
    V (sr_rem_count);
    
    return (ph);
}



/*
 *  Service a request to establish a connection.
 *  Executes as a separate process.
 */
void
sr_rmt_callme (ph)
pach ph;
{
    int n = ((struct num_st *) ph) ->num;
    contact (n);
    sr_net_send (n, ACK_CALLME, ph, PACH_SZ);
    sr_free ((daddr) ph);
    sr_kill (sr_cur_proc, FALSE);
}



/*
 *  Service a request to create a resource from a remote machine.
 *  Executes as separate process.
 */
void
sr_rmt_create (ph)
pach ph;
{
    short size;
    struct crb_st *crbp;
    struct crep_st *reply;

    crbp = (struct crb_st *) ph;
    size = sizeof (struct crep_st) - sizeof (rescap) + crbp->rc_size;
    reply = (struct crep_st *) sr_own_alloc (size, (rint) NULL);
    
    crbp->rcp = & reply->rcap;
    sr_create (crbp);
    
    reply->ph.rem    = ph->rem;
    sr_net_send (ph->origin, ACK_CREATE, &reply->ph, size);

    sr_free ((daddr) reply);
    /* ph freed when resource destroyed */
    
    sr_kill (sr_cur_proc, FALSE);
}



/*
 *  Service a request to destroy a resource from a remote machine.
 *  Executes as separate process.
 */
void
sr_rmt_destroy (ph)
pach ph;
{
    sr_destroy (& ((struct dest_st *) ph)->rc);
    sr_net_send (ph->origin, ACK_DESTROY, ph, ph->size);
    sr_free ((daddr) ph);
    sr_kill (sr_cur_proc, FALSE);
}



/*
 *  Service a request from srx to destroy this virtual machine.
 *  Executes as a separate process.
 */
void
sr_rmt_destvm (ph)
pach ph;
{
    sr_dest_all ();
    sr_net_send (ph->origin, ACK_DESTVM, ph, ph->size);
    sr_free ((daddr) ph);
    sr_kill (sr_cur_proc, FALSE);
}



/*
 *  Service a request to invoke an operation from a remote machine.
 *  Executes as separate process.
 */
void
sr_rmt_invk (ph)
pach ph;
{
    sr_invoke ((invb) ph);	/* invocation block won't move when local */
    sr_net_send (ph->origin, ACK_INVOKE, ph, ph->size);
    if (((invb) ph)->type != SEND_IN)
	sr_free ((daddr) ph);
    sr_kill (sr_cur_proc, FALSE);
}



/*
 *  Establish contact with machine n, ensuring that only one connection is
 *  made.  A higher numbered machine never calls sr_connect to a lower machine,
 *  but instead asks the lower machine to connect to it, by passing a message
 *  through srx.
 */
static void
contact (n)
{
    static bool started[MAX_VM];	/* connection requested for each vm? */
    static proc_queue waiting[MAX_VM];	/* processes waiting for each vm */
    struct num_st npk;			/* actual message packet */
    pach ph;				/* packet header pointer */

    if (n == sr_my_vm)
	sr_abort ("trying to connect to self?!");

    /* connecting to a lower machine is simple; ask it to do the work */
    /* (no problem if this happens more than once) */
    if (n < sr_my_vm)  {
	npk.num = n;
	ph = sr_remote (SRX_VM, REQ_CALLME, (pach) &npk, sizeof(npk));
	return;
    }

    /* get interlock and check again that we're not connected */
    P (rem_mutex);
    if (sr_net_known(n)) {
	V (rem_mutex);			/* ah, we now have contact */
	return;
    }

    /* if we've already requested a connection, just wait until it's complete */
    if (started[n]) {
	block(&waiting[n]);		/* put self on wait list */
	V (rem_mutex);			/* release interlock */
	sr_cswitch();			/* block */
	return;
    }

    /* must be the first time. set flag, then release interlock */
    started[n] = TRUE;
    V (rem_mutex);

    /* establish the connection */
    npk.num = n;
    ph = sr_remote (SRX_VM, REQ_FINDVM, (pach) &npk, sizeof(npk));
    sr_net_connect (n, ((struct saddr_st *) ph) -> addr);

    /* release everybody else who may now be waiting */
    P (rem_mutex);			/* interlock wait list */
    while (waiting[n].head != NULL)
	awaken(waiting[n]);		/* unblock everybody */
    V (rem_mutex);			/* release interlock */
}
