/*
 *  SR Run-Time Support.  Routines to Set Up Operations.
 */

#include <varargs.h>
#include "rts.h"

static sem oper_mutex;		/* mutual exclusion */
static sem oper_avail;		/* for waiting on entries */
static short oper_need;		/* # of procs needing entry */

static class Class;		/* operation class table */
static class class_free;	/* free list */
static sem class_mutex;		/* mutual exclusion */

static void free_class(), free_oper(), purge();
static oper get_oper();

/* make_op is used directly for making a local op
 * and indirectly for making a resource op.
 * someone thought make_op should be a macro for efficiency purposes.
 * probably a good idea, especially since now used for creating local
 * operations too.
 */
#define make_op(t,f,v,opcp) { \
    oper op; \
    \
    opcp->vm = sr_my_vm; \
    opcp->oper_index = (op = get_oper ()) - sr_optab; \
    opcp->seqn = op->seqn; \
    \
    op->type = (t); \
    op->f = (v); \
    op->pending = 0; \
    \
    op->res  = sr_cur_res; \
    op->next = sr_cur_res->oper_list; \
    sr_cur_res->oper_list = op; \
}

/* interface to make_op for making resource ops;
 * it needs to do a little more.
 * use local opcp so calculation done just once.
 */
#define make_resop(t,f,v) { \
    opcap *opcp; \
    \
    opcp = &((rescap *) sr_cur_res->rv_base)->opcap_list[sr_cur_res->noper++];\
    make_op(t,f,v,opcp); \
}

/*
 *  Add new resource input or proc operations.
 *  Store operation capability in resource capability.
 *  Called during resource initialization.
 */
void
sr_make_resops (va_alist)			/* RTS Primitive */
    va_dcl		{
    va_list vap;
    enum op_type type;
    class clap;
    int i, count;

    sr_check_stk();

    va_start (vap);
    while ((type = va_arg (vap, enum op_type)) != END_OP) {
	switch (type) {
	    case PROC_OP:
	    case PROC_REP_OP:
		make_resop (type, u.code, va_arg (vap, paddr))
		break;
		
	    case INPUT_OP:
		clap  = va_arg (vap, class);
		count = va_arg (vap, int);
		clap->numops += count;

		for (i = 0 ; i < count ; i++)
		    make_resop (type, u.clap, clap)
		break;
		
	    default:
		sr_abort ("bad oper type in sr_make_resops");
	}
    }

    va_end (vap);
}



/*
 *  Kill all resource operations for the named resource.
 *  Purge any pending input invocations.
 */
void
sr_kill_resops (res)
rint res;
{
    oper op;
    class clap;
    
    for (op = res->oper_list ; op ; op = op->next) {
	op->seqn++;
	if (op->type == INPUT_OP) {
	    clap = op->u.clap;
	    if (clap->old_in.head != NULL)
		purge (op - sr_optab, &clap->old_in);
	    if (clap->new_in.head != NULL)
		purge (op - sr_optab, &clap->new_in);

	    if (--clap->numops == 0)
		free_class (clap);
	}
	
	else if (op->type == SEMA_OP)
	    sr_kill_sem (op->u.sema);
    }

    if (res->oper_list != NULL)
	free_oper (res->oper_list);
}



/*
 *  Make a set of new local input operations.
 */
void
sr_make_liop (clap, opcp, count)		/* RTS Primitive */
    class clap;
    opcap *opcp;
    int count;
{
    int i;

    sr_check_stk();
    
    for (i = 0 ; i < count ; i++, opcp++) {
	make_op(INPUT_OP, u.clap, clap, opcp);
    }

    clap->numops += count;
}


    
/*
 *  Remove local operations from the operation table.  If told to do so, purge
 *  any pending invocations from the queues.  If the killed operation was the
 *  last of its class, free the class as well.
 */
void
sr_kill_liop (opcp, count)			/* RTS Primitive */
    opcap *opcp;
    int count;
{
    oper op, rop;
    class clap;
    int i;

    sr_check_stk();

    for (i = 0 ; i < count ; i++, opcp++) {
	op = sr_optab + opcp->oper_index;
	op->seqn++;

	if ((rop = op->res->oper_list) == op)
	    op->res->oper_list = op->next;
	else {
	    while (rop->next != op)  rop = rop->next;
	    rop->next = op->next;
	}

	clap = op->u.clap;
	if (clap->old_in.head != NULL)
	    purge (opcp->oper_index, &clap->old_in);
	if (clap->new_in.head != NULL)
	    purge (opcp->oper_index, &clap->new_in);

	if (--clap->numops == 0)  free_class (clap);
	op->next = NULL;
	free_oper (op);
    }
}



/*
 *  Remove all invocations of the specified operation from an invocation list.
 *  Operations are represented only by index numbers since the machine and
 *  sequence numbers have been checked when the invocation was done.
 */
static void
purge (op_index, ilist)
short op_index;
inv_queue *ilist;
{
    invb ibp, ribp, last;
    
    last = NULL;
    ibp = (*ilist).head;
    while (ibp) {
	ribp = ibp;
	ibp = ibp->next;
	if (ribp->opc.oper_index == op_index) {
	    if (last == NULL)
		(*ilist).head = ibp;
	    else
		last->next = ibp;
	    if (ibp != NULL) 
		ibp->last = ribp->last;
	    else
		(*ilist).tail = last;
	    sr_rej_inv (ribp);
	}
	else
	    last = ribp;
    }
}



/*
 *  Returns pointer to the next eligible invocation block for the GC to check in
 *  processing an input statement.  Process must have access to the operation
 *  class.  If no invocations are available, wait until more arrive.
 */
invb
sr_get_anyinv ()				/* RTS Primitive */
{
    invb ibp;

    sr_check_stk();
    
    if (sr_cur_proc->next_inv == NULL)  {
	if (sr_cur_proc->else_leg  ) {
	    return(NULL); }
	else{	
	    sr_reaccess (); }
    }
    ibp = sr_cur_proc->next_inv;
    sr_cur_proc->next_inv = ibp->next;
    return (ibp);
}


    
/*
 *  Returns pointer to next eligible invocation of the specified operation.
 *  If none are available, wait until more arrive.
 */
invb
sr_get_myinv (opc)				/* RTS Primitive */
    opcap opc;
{
    invb ibp;

    sr_check_stk();
    
    while (TRUE) {
    	if (sr_cur_proc->next_inv == NULL)  {
	    if (sr_cur_proc->else_leg  ) {
	    	return(NULL); }
	    else{	
	    	sr_reaccess (); }
    	}
	
	while (ibp = sr_cur_proc->next_inv) {
	    sr_cur_proc->next_inv = ibp->next;
	    if (opc.oper_index == ibp->opc.oper_index)
		return (ibp);
	}
    }
}



/*
 *  Returns pointer to next eligible invocation of the specified operation.
 *  If none are available, return NULL.
 */
invb
sr_chk_myinv (opc)				/* RTS Primitive */
    opcap opc;
{
    invb ibp;

    sr_check_stk();
    
    while (ibp = sr_cur_proc->next_inv) {
	sr_cur_proc->next_inv = ibp->next;
	if (opc.oper_index == ibp->opc.oper_index)  break;
    }
    
    return (ibp);
}



/*
 *  Get next invocation for operations appearing in a single
 *  class and with no synchronization or scheduling expressions.
 *  This is an optimization; it should probably go in the nugget.
 */
invb
sr_receive (clap)				/* RTS Primitive */
    class clap;
{
    invb ibp;

    sr_check_stk();
    
    sr_iaccess (clap, FALSE);
    ibp = sr_get_anyinv ();
    sr_rm_iop (ibp);
    return (ibp);
}



/*
 *  Create an operation to act as a semaphore (i.e., a non-exported,
 *  parameterless, operation in its own class.  This is an optimization.
 */
sem
sr_make_semop()					/* RTS Primitive */
{
    oper op;
    sem sp;

    sr_check_stk();
    
    op = get_oper ();
    op->type = SEMA_OP;
    op->u.sema = sp = sr_make_sem (0);
    
    op->next = sr_cur_res->oper_list;
    sr_cur_res->oper_list = op;
    return (sp);
}



/*
 *  Initialize RTS operation table.
 */
void
sr_init_oper ()
{
    short i;

    sr_optab = (oper) sr_alloc (sr_max_operations * sizeof (struct oper_st));
    for (i = 1 ; i < sr_max_operations ; i++) {
	(sr_optab + i)->next = sr_optab + i + 1;
	(sr_optab + i)->seqn = INIT_SEQ_OP;
	(sr_optab + i)->type = END_OP;
    }

    (sr_optab + i)->next = NULL;
    sr_oper_free = sr_optab + 1;

    oper_mutex = sr_make_sem (1);
    oper_avail = sr_make_sem (0);
    oper_need  = 0;

    sr_no_ocap.seqn = NOOP_SEQN;		/* other fields zero */
    sr_nu_ocap.seqn = NULL_SEQN;		/* other fields zero */
}



/*
 *  Get a free operation table entry.
 */
static oper
get_oper ()
{
    oper op;
    
    P (oper_mutex);
    if ((op = sr_oper_free) != NULL) {
	sr_oper_free = sr_oper_free->next;
	V (oper_mutex);
    }
    
    else {
	/*
	 *	Wait for an entry to become available.
	 */
	rts_warn ("low on operation table entries");
	oper_need++;
	V (oper_mutex);
	P (oper_avail);
	op = sr_oper_free;
	sr_oper_free = sr_oper_free->next;
	/*
	 *	Wake up any others waiting for table entries.
	 */
	if (--oper_need > 0 && sr_oper_free != NULL)
	    V (oper_avail);
	else
	    V (oper_mutex);
    }
    
    return (op);
}



/*
 *  Return a non-empty list of operation table entries to the free list.
 */
static void
free_oper (op)
oper op;
{
    oper opp;

    P (oper_mutex);
    for (opp = op ; opp->next != NULL ; opp = opp->next);
    opp->next = sr_oper_free;
    sr_oper_free = op;

/*
 *  If there are any processes waiting for an entry, pass mutual exclusion to
 *  one of them.
 */
    if (oper_need > 0)
	V (oper_avail);
    else
	V (oper_mutex);
}



/*
 *  Initialize the operation class table.
 */
void
sr_init_class ()
{
    short i;
    
    Class = (class) sr_alloc (sr_max_classes * sizeof (struct class_st));
    for (i = 0 ; i < sr_max_classes ; i++) {
	(Class + i)->next = Class + i + 1;
	(Class + i)->numops = 0;
    }
	
    Class [sr_max_classes-1].next = NULL;
    class_free = Class;
    class_mutex = sr_make_sem (1);
    sr_class_count = sr_make_sem (sr_max_classes);
}



/*
 *  Give the GC a new operation class.
 */
class
sr_make_class ()				/* RTS Primitive */
{
    class clap;

    sr_check_stk();
    
    if (sr_class_count->value == 0)
	rts_warn ("low on class descriptors");

    P (sr_class_count);
    P (class_mutex);
    clap = class_free;
    class_free = clap->next;
    
    clap->pending = 0;
    clap->inuse   = FALSE;
    clap->old_in.head  = clap->new_in.head = NULL;
    clap->old_in.tail  = clap->new_in.tail = NULL;
    clap->old_pr.head  = clap->new_pr.head = NULL;
    clap->old_pr.tail  = clap->new_pr.tail = NULL;
    clap->else_pr = NULL;
    clap->else_tailpr = NULL;
    
    V (class_mutex);
    return (clap);
}



/*
 *  Return an operator class to the free list.
 */
static void
free_class (clap)
class clap;
{
     
    P (class_mutex);
    clap->next = class_free;
    class_free = clap;
    V (class_mutex);
    V (sr_class_count);
}



/*
 *  Return number of pending invocations for an input operation.
 */
int
sr_query_iop (opc)				/* RTS Primitive */
    opcap *opc;
{
    oper op;

    sr_check_stk();
    op = sr_optab + opc->oper_index;
    return (op->pending);
}



/*
 *  Return number of pending invocations for a semaphore operation.
 *  Depends on our representation of semaphore;  i.e., value >= 0.
 */
int
sr_query_sem (sp)				/* RTS Primitive */
    sem sp;
{
    
    sr_check_stk();
    return (sp->value);
}
