/*
 *  SR Run-Time Support.  Network Interface Routines.
 */

#include "rts.h"

static daddr start_srx();


/*
 *  Initialize network, if this hasn't already been done before.
 *  srx_addr is socket address of srx if it's already running.
 */
void
sr_init_net (srx_addr)
daddr srx_addr;
{
    daddr my_addr;
    struct saddr_st as;

    if (sr_exec_up)
	return;				/* network is already inited */

    if (!srx_addr)			/* is srx running? */
	srx_addr = start_srx();		/* no, start it */

    my_addr = sr_net_start ();		/* start socket routines */
    sr_net_connect (SRX_VM, srx_addr);	/* connect to srx */
    sr_exec_up = TRUE;			/* set flag that the network's up */

    /* add detail to err messages because now there's more than one of us */
    sprintf(sr_my_label,"[vm %d] ",sr_my_vm);

    /* tell srx we are here */
    strcpy(as.addr,my_addr);
    sr_net_send(SRX_VM,MSG_HELLO,&as.ph,sizeof(as));

    /* start network interface process */
    sr_activate (sr_spawn (sr_net_interface, RTS_OWN, 0, 0, 0, 0));
}



/*
 *  Fork and start srx.  Return the address of the srx socket.
 */

static daddr
start_srx ()
{
    char *path;
    int pid, fd[2];
    static char buf[100];	/* must be static because we return it */

    if (!(path = getenv("SRXPATH")))
	path = sr_exec_path;	/* use path set by srl if no env variable */
    DEBUG(2,"srx path: %s",path,0,0);
    fflush(stdout);
    fflush(stderr);
    if (pipe(fd) != 0)		/* make pipe for initial message from srx */
	sr_abort ("can't open pipe for srx");
    if ((pid = vfork()) < 0)
	sr_abort ("can't vfork to start srx");
    if (pid == 0) {
	/* we're the child - execute srx */
	dup2(fd[1],0);	/* make pipe output fd 0 for srx, replacing stdin */
	execl(path,RUNTIME_EXEC,VM_MAGIC,PROTO_VER,sr_argv[0],NULL);
	perror(path);
	sr_abort("can't execute srx");
    }
    /*
     *	The parent continues execution of the SR program.
     *	Read back the srx socket address over the pipe we just made.
     *	(The pipe is for synchronization as well as communication!)
     */
    close(fd[1]);
    if (read(fd[0],buf,sizeof(buf)) == 0)	/* read srx socket address */
	sr_abort("no reply from srx startup");
    close(fd[0]);				/* no longer need the pipe */
    return (buf);				/* return socket address */
}



/*
 *  Network interface process.
 *  Read messages from other machines when they come in.
 *  In order to ensure fairness, this process should be
 *  run at the same priority as user processes.
 */
void
sr_net_interface ()
{
    daddr pbuf;
    pach ph, nph;
    enum ms_type t;

    ph = 0;
    for (;;) {
	/* allocate space for packet header if we need a new one */
	if (!ph)  {
	    pbuf = sr_own_alloc (PBUF_SZ, (rint) NULL);
	    ph = (pach) pbuf;
	}

	/* get the header of the next available message */
	t = sr_net_recv (ph);

	/* if the whole packet is too big for the buffer,
	 * allocate a new one and copy the header. */
	if (ph->size > PBUF_SZ) {
	    nph = (pach) sr_own_alloc (ph->size, (rint) NULL);
	    *nph = *ph;
	    sr_free ((daddr) ph);
	    ph = nph;
	}

	/* read the rest of the packet */
	sr_net_more (ph);

	/* process the message according to its type */

	switch (t) {

	    case MSG_EOF:
		/* ignore deaths of others, except for srx */
		if (ph->origin == SRX_VM) {
		    sr_exec_up = FALSE;
		    sr_abort("srx died!");
		}
		break;

	    case MSG_EXIT:
		sr_exec_up = FALSE;
		sr_stop (((struct num_st *) ph) -> num);
		break;

	    case REQ_CALLME:
		sr_activate (sr_spawn (sr_rmt_callme, RTS_OWN, ph, 0, 0, 0));
		ph = 0;
		break;

	    case REQ_CREATE:
		sr_activate (sr_spawn (sr_rmt_create, RTS_OWN, ph, 0, 0, 0));
		ph = 0;
		break;
		
	    case REQ_DESTROY:
		sr_activate (sr_spawn (sr_rmt_destroy, RTS_OWN, ph, 0, 0, 0));
		ph = 0;
		break;

	    case REQ_DESTVM:
		sr_activate (sr_spawn (sr_rmt_destvm, RTS_OWN, ph, 0, 0, 0));
		ph = 0;
		break;
		
	    case REQ_INVOKE:
		sr_activate (sr_spawn (sr_rmt_invk, RTS_OWN, ph, 0, 0, 0));
		ph = 0;
		break;
		
	    case ACK_LOCVM:
	    case ACK_CREVM:
	    case ACK_FINDVM:
	    case ACK_CREATE:
	    case ACK_INVOKE:
	    case ACK_DESTROY:
	    case ACK_DESTVM:
	    case ACK_CALLME:
		ph->rem->reply = ph;
		V (ph->rem->wait);
		ph = 0;
		break;

	    default:
		sr_abort ("unknown incoming message type");
	}
    }
}
