/*
 *  SR Run-time Support.
 *  Main and miscellaneous routines for running SR programs.
 *
 *  This module is the one that actually declares the globals.
 */

#define global
#define initval(v) = v
#include "rts.h"

static void startup();


/*
 *  Start up an SR program or one of its virtual machines.
 *
 *  For the original startup, argument interpretation is up to the programmer.
 *  For a new virtual machine, args are:
 *	argv[1]	    a magic string indicating that this is a VM startup
 *	argv[2]	    physical machine number for the new VM
 *	argv[3]	    virtual machine number for the new VM
 *	argv[4]	    network address of srx's connection port
 *	argv[5]	    debugging flags
 */
main (argc, argv)
int argc;
char **argv;
{
    sr_argv = argv;
    if (argc != 6 || strcmp(argv[1],VM_MAGIC) != 0)  {
	/* this is the initial startup */
	sr_argc = argc;
	sr_my_vm = MAIN_VM;
	sr_init_debug ((char *) NULL);
    } else {
	/* this is a new virtual machine */
	sr_argc = 0;	/* hide args from SR program */
	sr_my_machine = atoi (argv[2]);
	sr_my_vm = atoi (argv[3]);
	sprintf (sr_my_label, "[vm %d] ", sr_my_vm);
	sr_init_debug (argv[5]);
    }

    if (sr_trace_flag)
	fprintf (stderr, "%sSR starting\n", sr_my_label);

    sr_init_proc(startup);/* initialize processes (no return; calls startup) */
    /*NOTREACHED*/
}

/*
 *  initialization continues here in an SR process context.
 */
static void
startup()
{
    crb crbp;

    /* initialize runtime system */
    sr_init_sem ();
    sr_init_event ();
    sr_init_res ();
    sr_init_oper ();
    sr_init_class ();
    sr_init_co ();
    sr_init_rem ();
    sr_init_io ();

    if (sr_my_vm == MAIN_VM) {
	/* create an instance of main resource on main machine */
	crbp = (crb) sr_own_alloc (sizeof (struct crb_st), (rint) NULL);
	crbp->rpatid = 0;
	crbp->vm = sr_my_vm;
	sr_create (crbp);
    }
    else {
	/* otherwise, just start up network and wait for instructions */
	sr_argc = 0;		/* hide args from users */
	sr_init_net (sr_argv[4]);
    }

    sr_kill (sr_cur_proc, FALSE);	/* kill this initialization process */
}



/*
 *  Print run-time support error or warning message.
 */
void
sr_message (s, t)
char *s, *t;
{

    fflush (stdout);
    fprintf (stderr, "%sRTS %s: %s\n", sr_my_label, s, t);
    fflush (stderr);
}



/*
 *  Fatal run time support error.
 *  Print error message and tell srx to abort other nodes.
 */
void
sr_abort (s)
char *s;
{
    sr_message ("abort", s);
    sr_stop (1);
}



/*  Fatal error detected by network routines.  */

void
sr_net_abort (s)
char *s;
{

    sr_abort(s);
}



/*  Terminate execution of the SR program.  */

void
sr_stop (exitcode)				/* RTS Primitive */
int exitcode;
{
    struct num_st pkt;

    if (sr_exec_up) {
	sr_exec_up = FALSE;		/* prevent looping if send aborts */
	pkt.num = exitcode;
	sr_net_send (SRX_VM, MSG_EXIT, &pkt.ph, sizeof(pkt));
    }
    fflush(stdout);
    if (sr_trace_flag) {
	fprintf (stderr, "%sSR terminating, code %d\n",
	    sr_my_label, exitcode);
	fflush (stderr);
    }
    exit (exitcode);
}



/*  Terminate program due to stack overflow.  */

void
sr_stk_overflow ()
{
    sr_abort ("stack overflow");
}



/*  Terminate program due to stack underflow.  */

void
sr_stk_underflow ()
{
    sr_abort ("stack underflow");
}



/*  Terminate program due to stack corruption.  */

void
sr_stk_corrupted ()
{
    sr_abort ("stack corrupted");
}
