/*
 *  SR Run-Time Support.  Event waiting and handling.
 */

#include "rts.h"

enum fdset_op { FDSET_AND, FDSET_OR };



static int fdmerge ();
static void update_clock ();
static int wakeup_sleepers ();



struct napper {		/* napping job information */
    struct napper *next;	/* next list entry */
    sem sp;			/* semaphore to tickle */
    int wakeup;			/* wakeup time */
};

struct iowait {		/* i/o blocked job information */
    struct iowait *next;	/* next list entry */
    sem sp;			/* semaphore to tickle */
    fd_set *query;		/* where to find set of wanted files */
    fd_set *results;		/* where to put the results */
    enum io_type inout;		/* INPUT or OUTPUT wait */
};



struct napper *nap_list = NULL;	/* napping jobs (unordered list) */
static sem nap_mutex;		/* semaphore for interlocking the list */

struct iowait *io_list = NULL;	/* I/O blocked jobs (in order blocked) */
struct iowait *io_tail;		/* end of the list */
static sem iow_mutex;		/* semaphore for interlocking the list */

static struct timeval start;	/* system time when we started executing */
static int msclock;		/* elapsed real time, in milliseconds */




/*
 *  Initialize the clock so we can measure a program's age.
 *  Initialize the semaphores interlocking the lists.
 */
void
sr_init_event ()
{
    struct timezone tz;

    gettimeofday(&start,&tz);
    nap_mutex = sr_make_sem(1);
    iow_mutex = sr_make_sem(1);
}



/*
 *  Return the elapsed time in milliseconds.
 */
int
sr_age ()					/* RTS Primitive */
{
    update_clock();			/* get the clock */
    wakeup_sleepers();			/* having done so, check wakeups */
    return msclock;			/* return the age */
}



/*
 *  Delay a process a specified number of milliseconds.
 */
void
sr_nap (msec)					/* RTS Primitive */
{
    sem sp;
    struct napper *np;

    if (msec <= 0)			/* return if no delay */
	return;
    update_clock();			/* get clock reading */
    wakeup_sleepers();			/* awaken all whose clocks have rung */

    sp = sr_make_sem (0);
    np = (struct napper *) sr_own_alloc (sizeof (struct napper), (rint) NULL);
    np->sp = sp;
    np->wakeup = msclock + msec;

    P (nap_mutex);			/* add to list, under interlock */
    np->next = nap_list;
    nap_list = np;
    V (nap_mutex);

    P (sp);				/* wait for wakeup call */
    sr_kill_sem (sp);
}



/*
 *  Wait for input or activity on a set of files.
 *  "query" is not dereferenced here yet, but later when blocking on I/O.
 */
void
sr_iowait (query, results, inout)
fd_set *query, *results;
enum io_type inout;
{
    sem sp;
    struct iowait *ip;

    sp = sr_make_sem (0);
    ip = (struct iowait *) sr_own_alloc (sizeof (struct iowait), (rint) NULL);
    ip->next = NULL;
    ip->sp = sp;
    ip->query = query;
    ip->results = results;
    ip->inout = inout;

    P (iow_mutex);			/* add to end of list, with interlock */
    if (io_list == NULL)
	io_list = ip;
    else
	io_tail->next = ip;
    io_tail = ip;
    V (iow_mutex);

    P (sp);				/* wait for file ready signal */
    sr_kill_sem (sp);
}



/*
 * Check for I/O or timer event; return nonzero if any processes were awakened.
 *
 * If "wait" is true, block until a new process comes ready.
 *
 * Called only from process.c, so we don't need to interlock the lists.
 */
int
sr_evcheck (wait)
{
    struct napper *np;
    struct iowait *ip, **ipp;
    struct timeval tv, *timeout;
    fd_set inset, outset, readyset;
    static struct timeval zerotime;
    static fd_set zeroset;
    int n, nfds, nfound, waketime;

    if (nap_list == NULL && io_list == NULL)
	return 0;			/* return 0 if lists are empty */

    if (!wait) {
	tv = zerotime;			/* set immediate timeout */
	timeout = &tv;
	update_clock ();		/* update millisecond clock reading */
	waketime = msclock;		/* this will be clock after the poll */

    } else if (nap_list != NULL) {

	/* check the time and calculate a timeout */
	update_clock ();
	if (wakeup_sleepers ())
	    return 1;			/* accomplished something, so return */

	/* we know at this point there's somebody still napping */
	waketime = nap_list->wakeup;
	for (np = nap_list->next; np != NULL; np = np->next)
	    if (np->wakeup < waketime)
		waketime = np->wakeup;

	tv.tv_sec = (waketime - msclock) / 1000;
	tv.tv_usec = ((waketime - msclock) % 1000) * 1000;
	timeout = &tv;

    } else {

	timeout = NULL;			/* set no timeout for I/O wait */
    }

    /* now we know we have to select(), with or without a timeout */
    nfds = 0;
    inset = outset = zeroset;
    for (ip = io_list; ip != NULL; ip = ip->next) {
	if (ip->inout == INPUT)
	    n = fdmerge (&inset, &inset, ip->query, FDSET_OR);
	else
	    n = fdmerge (&outset, &outset, ip->query, FDSET_OR);
	if (n > nfds)
	    nfds = n;
    }

    DEBUG (0x80, "select: %08X %08X t=%d",
	inset.fds_bits[0], outset.fds_bits[0],
	timeout ? tv.tv_sec * 1000 + tv.tv_usec / 1000 : -1);

    nfound = select (nfds, &inset, &outset, (fd_set *) 0, timeout);

    DEBUG (0x80, "selectd %08X %08X n=%d",
	inset.fds_bits[0], outset.fds_bits[0], nfound);

    if (nfound < 0)  {
	perror ("select failure");
	sr_abort ("I/O error");
    }

    if (nfound == 0) {			/* if we got a timeout */
	msclock = waketime;		/* we know what clock is (at least) */
	return wakeup_sleepers ();	/* awaken jobs */
    }

    /* one or more files are ready for I/O */
    for (ipp = &io_list; (ip = *ipp) != NULL; ) {
	if (ip->inout == INPUT)
	    n = fdmerge (&readyset, &inset, ip->query, FDSET_AND);
	else
	    n = fdmerge (&readyset, &outset, ip->query, FDSET_AND);
	if (n > 0) {
	    *ip->results = readyset;
	    V (ip->sp);
	    *ipp = ip->next;
	    sr_free ((daddr) ip);
	} else {
	    ipp = &ip->next;
	    io_tail = ip;
	}
    }

    return 1;				/* indicate we accomplished something */
}



/*
 *  Merge two fd_set structures to create a third.
 *
 *  fdmerge (result, set1, set2, op)
 *	first three parameters are pointers to fd_set structs.
 *	op is FDSET_AND or FDSET_OR.
 *  return value is 1 + highest fd set in result, or 0 if none.
 */
static int
fdmerge (result, set1, set2, op)
fd_set *result, *set1, *set2;
enum fdset_op op;
{
    int i, n;
    fd_set tempset;
    static fd_set zeroset;

    tempset = zeroset;
    for (i = n = 0; i < FD_SETSIZE; i++) {
	if ((op == FDSET_AND) 
	    ? (FD_ISSET (i, set1) && FD_ISSET (i, set2))
	    : (FD_ISSET (i, set1) || FD_ISSET (i, set2))) {
		FD_SET (i, &tempset);
		n = i + 1;
	}
    }
    *result = tempset;
    return n;
}




/*
 *  Put the current elapsed time in msclock.
 */
static void
update_clock ()
{
    struct timeval tv;
    struct timezone tz;

    gettimeofday(&tv,&tz);
    msclock = 1000 * (tv.tv_sec - start.tv_sec)
		    + tv.tv_usec/1000 - start.tv_usec/1000;
    DEBUG (4, "clock %d", msclock, 0, 0);
}



/*
 *  Awaken napping jobs whose wakeup times have arrived.  Returns the number
 *  awakened.  Called every time we've read the system clock, or inferred its
 *  value after a timeout.
 */
static int
wakeup_sleepers ()
{
    struct napper **ptr, *np;
    int n = 0;

    P (nap_mutex);			/* obtain interlock */
    for (ptr = &nap_list; (np = *ptr) != NULL; ) {
	if (np-> wakeup <= msclock) {	/* if wakeup time */
	    V (np->sp);				/* awaken process */
	    *ptr = np->next;			/* remove from list */
	    sr_free ((daddr) np);		/* free the entry */
	    n++;
	} else {
	    ptr = &np->next;
	}
    }
    V (nap_mutex);			/* release interlock */
    return n;
}
