/*
 *  SR Run-Time Support.  Routines to Handle the co Statement.
 */

#include "rts.h"

static cob Cotab;			/* co block table */
static cob cob_free;			/* free list */
static sem cob_mutex;			/* mutual exclusion */

static cob get_cob ();



/*
 *  Get a new co block for the start of a co statement
 *  and link it to the current process.
 */
void
sr_co_start ()					/* RTS Primitive */
{
    cob cobp;

    sr_check_stk();

    cobp = get_cob ();
    cobp->next = sr_cur_proc->cob_list;
    sr_cur_proc->cob_list = cobp;

    cobp->pending = 0;
    cobp->done = sr_make_sem (0);
}



/*
 *  Set things up for a call within a co statement.
 */
void
sr_co_call (ibp)
invb ibp;
{
    cob cobp;
    
    cobp = ibp->co.cobp = sr_cur_proc->cob_list;
    P (cobp->mutex);
    ibp->co.seqn = cobp->seqn;
    cobp->pending++;
    V (cobp->mutex);
}



/*
 *  A call invocation from a co statement has returned.
 *  If the invoker is still interested in this event
 *  notify him.
 */
void
sr_co_call_done (ibp)
invb ibp;
{
    cob cobp;

    cobp = ibp->co.cobp;
    P (cobp->mutex);

    if (cobp->seqn == ibp->co.seqn) {
	ibp->co.next = cobp->done_list;
	cobp->done_list = ibp;
	cobp->pending--;
	V (cobp->done);
    }
    else
	sr_free ((daddr) ibp);

    V (cobp->mutex);
}


/*
 *  Handle a send within a co statement.
 *    Make a copy of the invocation block to return to invoker.
 *    Then, simulate a sr_co_call and sr_co_call_done.
 */
void
sr_co_send (ibp)
invb ibp;
{
    invb new_ibp;
    
    new_ibp = sr_dup_invb (ibp);
    sr_co_call (new_ibp);
    sr_co_call_done (new_ibp);
}



/*
 *  Wait for a co invocation to terminate.  Return a pointer to the original
 *  invocation block so that the GC can copy result parameters and
 *  find out which arm terminated.
 */
invb
sr_co_wait ()					/* RTS Primitive */
{
    cob cobp;
    invb ibp;

    sr_check_stk();

    cobp = sr_cur_proc->cob_list;
    P (cobp->mutex);

    if (cobp->pending == 0 && cobp->done_list == NULL)
	ibp = NULL;

    else {
	V (cobp->mutex);
	P (cobp->done);
	P (cobp->mutex);

	ibp = cobp->done_list;
	cobp->done_list = ibp->co.next;
    }

    V (cobp->mutex);
    return (ibp);
}



/*
 *  A co statement has terminated.  Release the co block.
 */
void
sr_co_end ()					/* RTS Primitive */
{
    cob cobp;
    invb ibp;

    sr_check_stk();

    cobp = sr_cur_proc->cob_list;
    sr_cur_proc->cob_list = cobp->next;

    P (cobp->mutex);
    cobp->seqn++;
    sr_kill_sem (cobp->done);

/*
 *  Free any invocation blocks that were returned after
 *  the last sr_co_wait() was done.
 */
    while (ibp = cobp->done_list) {
	cobp->done_list = ibp->co.next;
	sr_free ((daddr) ibp);
    }

    P (cob_mutex);
    cobp->next = cob_free;
    cob_free = cobp;
    V (cobp->mutex);
    V (cob_mutex);
    V (sr_cob_avail);
}



/*
 *  Initialize co statement part of SR RTS.
 */
void
sr_init_co ()
{
    tindex i;

    Cotab = (cob) sr_alloc (sr_max_co_stmts * sizeof (struct cob_st));
    for (i = 0 ; i < sr_max_co_stmts ; i++) {
	Cotab [i].next = & Cotab [i+1];
	Cotab [i].seqn = INIT_SEQ_CO;
	Cotab [i].mutex = sr_make_sem (1);
	Cotab [i].done_list = NULL;
    }

    Cotab [sr_max_co_stmts-1].next = NULL;
    cob_free = Cotab;

    cob_mutex = sr_make_sem (1);
    sr_cob_avail = sr_make_sem (sr_max_co_stmts);
}



/*
 *  Get a free co block table entry.
 */
static cob
get_cob ()
{
    cob cobp;

    if (sr_cob_avail->value == 0)
	rts_warn ("low on co block table entries");

    P (sr_cob_avail);
    P (cob_mutex);
    cobp = cob_free;
    cob_free = cob_free->next;

    V (cob_mutex);
    return (cobp);
}
