/*
 *  SR Run-Time Support.  Memory Management.
 *
 *  This file deals with memory allocated by the runtime system or *implicitly* 
 *  by SR programs.  Explicit SR new() and free() calls are supported in misc.c.
 */

#include "rts.h"


static sem mem_mutex;
static memh all_mem;



/*
 *  Initialize memory management.
 */

void
sr_init_mem ()
{
    mem_mutex = sr_make_sem (1);
    all_mem = NULL;
}



/*
 *  Called by the generated code to allocate memory for
 *  the current resource or on behalf of the RTS.
 */
daddr
sr_gen_alloc (size, owner)			/* RTS Primitive */
int size;
int owner;
{
    sr_check_stk();
    if (owner == RTS_OWN)
	return (sr_own_alloc (size, (rint) NULL));
    else
	return (sr_own_alloc (size, sr_cur_res));
}



/*
 *  Called by the generated code to free memory for
 *  the current resource or on behalf of the RTS.
 */
void
sr_gen_free (addr)				/* RTS Primitive */
daddr addr;
{
    sr_check_stk();
    sr_free (addr);
}



/*
 *  Called by the generated code to allocate memory for an object whose size is
 *  calculated at runtime.  This memory is later freed by code generated inline.
 *
 *  talloc(size,plist) allocates "size" bytes, updates the linked list at
 *  *plist, and returns the address of the allocated memory.
 */

union memhdr {		/* block header; the union forces good alignment */
    daddr next;		/* link to next block */
    long l;
    double d;
};

daddr
sr_talloc (size, plist)				/* RTS Primitive */
int size;
daddr *plist;
{
    daddr p;

    sr_check_stk();			/* check stack */
    if (size < 0)			/* check call */
	sr_abort("bad sr_talloc size");
    DEBUG (0x20000, "talloc(%d)", size, 0, 0);

    p = sr_own_alloc (sizeof (union memhdr) + size, sr_cur_res);
    *(daddr *)p = *plist;		/* link to current head */
    *plist = p;				/* update list head */
    return (p + sizeof (union memhdr));	/* return pointer */
}



/*
 *  Allocate a chunk of contiguous memory.
 *  Keep track of allocated memory and which resource owns it (NULL means RTS).
 */
daddr
sr_own_alloc (size, res)
int size;
rint res;
{
    memh mp;

    P (mem_mutex);
    mp = (memh) sr_alloc (size + MEMH_SZ);
    mp->res = res;

    insert (mp, all_mem, mnext, mlast);
    if (res != NULL)
	insert (mp, res->meml, rnext, rlast);

    V (mem_mutex);
    return ((daddr) mp + MEMH_SZ);
}



/*
 *  malloc a chunk of memory and check for success.
 */
daddr
sr_alloc (size)
int size;
{
    daddr mp;

    mp = malloc ((unsigned) size);
    DEBUG(0x40000,"sr_alloc(%d) -> %06X",size,mp,0);
    if (mp == NULL)
	sr_abort ("out of memory");
    return (mp);
}



/*
 *  Free a previously allocated chunk of memory.
 *  Remove record from global and resource memory lists.
 */
void
sr_free (addr)
daddr addr;
{
    memh mp;

    P (mem_mutex);
    mp = (memh) (addr - MEMH_SZ);
    
    delete (mp, all_mem, mnext, mlast);
    if (mp->res != NULL)
	delete (mp, mp->res->meml, rnext, rlast);

    DEBUG(0x40000,"sr_free(%06X)",mp,0,0);
    free ((daddr) mp);		/* UNIX free for now */
    V (mem_mutex);
}



/*
 *  Free all memory belonging to the specified resource.
 */
void
sr_res_free (res)
rint res;
{
    memh mp;
    
    P (mem_mutex);
    for (mp = res->meml ; mp ; mp = mp->rnext) {
	delete (mp, all_mem, mnext, mlast);
	DEBUG(0x40000,"sr_res_free(%06X)",mp,0,0);
	free ((daddr) mp);	/* UNIX free for now */
    }
	
    V (mem_mutex);
}
