/*  symtab.c -- various routines that mess with symbol tables.
 *
 *  A symbol table is a set of Symbol structs linked together via "next" and
 *  "prev" fields.  The first entry is typically a dummy entry serving only
 *  as an anchor point.
 *  
 *  The primary symbol table table begins at "st_top".  It contains predefined
 *  functions, enums, and resources;  global and resource components;  and 
 *  currently accessible resource and block variables.
 *  
 *  "comp_sym" points to the current component within the primary table.
 *  Symbols defined in its spec are in an auxiliary table hung off the s_tdef
 *  field of the resource entry.
 *  
 *  "main_table" is set when parsing a global or resource *specification* (but
 *  not the body) and points to this auxiliary table.  While "main_table" is
 *  set, types, enums, ops, and optypes are installed there instead of in the
 *  primary table or in "aux_table".  The "main_table" pointer is reset to NULL
 *  at the end of the spec, but the auxiliary table is still accessable as 
 *  "comp_sym->s_tdef".
 *  
 *  If "aux_table" is set, other new symbols are added to it instead of to the
 *  primary table.  For example, "aux_table" is set while parsing a record
 *  definiton;  the record elements form an auxiliary table located through the
 *  s_tdef field of the main entry.
 *  
 *  Symbol structs are also used for signatures; these generally are not part
 *  of any symbol table.
 *  
 *  [still need to discuss local_block stuff]
 */


#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"
#include "../util.h"


static Symptr st_top;		/* top of main symbol table */
static Symptr st_end; 		/* end of main symbol table */
static Symptr st_end_predefs;	/* last predef in main symbol table */

static int nest_level = 0;	/* current nesting: 0 = global, 1 = proc, etc */


static Bool check_dup();
static Symptr st_import_lookup();
static void at_append(), st_append(), lb_append();



/*  main_install(name,kind) - install in spec, if any, else in primary table  */

Symptr
main_install(name, kind)
char *name;
Kind kind;
{
    if (main_table)
	return at_install(main_table, name, kind);
    else
	return st_install(name, kind);
}



/*  st_install(name,kind) - install in primary symbol table.
 *
 *  returns pointer to Symbol, or NULL if name is a duplicate.
 */
Symptr
st_install(name,kind)
char *name;
Kind kind;
{
    Symptr s;

    if (name) {
	name = nt_lookup(name,TRUE);
	if (st_dup(name)) {
	    ERROR(E_WARN+7,name);
	    return (NULLSYM);
	}
	s = new_symbol(kind);
	s->s_name = name;
    } else
	s = new_symbol(kind);

    st_append(s);
    if (s->s_kind == K_BLOCK)
	lb_append(s);
    return (s);
}



/*  import_append(s) - append an imported entry to the appropriate table.
 *  note: already did an st_lookup on s->s_name to ensure it is unique.
 */
void
import_append(s)
Symptr s;
{
    assert (s && s->s_kind == K_IMPORT);
    if (main_table) 
	at_append(main_table,s);		
    else
	st_append(s);
}



/*
 * st_append(s) - add s to the symbol table.
 * update st_end and, if necessary, st_cb.
 */
static void
st_append(s)
Symptr s;
{
    assert(s);
    if (st_end != NULLSYM) {
	st_end->s_next = s;
	s->s_next = NULLSYM;
	s->s_prev = st_end;
	st_end = s;
    } else {
	s->s_prev = s->s_next = NULLSYM;
	st_top = st_end = s;
    }
    if (s->s_kind == K_BLOCK)
	st_cb = s;
}



/* lb_append(s) - attach block s to local block table, but not for objects
 * being imported;  those will get moved to appropriate part of symbol table.
 */
static void
lb_append(s)
Symptr s;
{
    Local_blockptr lb;

    if (s->s_kind != K_BLOCK)
	return;		/* not a block */
    if (inter_depth == 0) {
	lb = (Local_blockptr)alloc(sizeof (Local_block));
	if (lb_end)
	    lb_end->l_next = lb;
	else
	    lb_top = lb;
	lb_end = lb;
	lb->l_nest = nest_level;
	lb->l_st = s;
	lb->l_next = 0;
    }
    nest_level++; /* keep pop_block happy in all cases. */
}




/* at_install(table,name,kind) - install new symbol in aux table, returning
 * a pointer to the installed symbol, or NULLSYM if duplicating another entry.
 */
Symptr
at_install(table,name,kind)
Symptr table;
char *name;
Kind kind;
{
	Symptr s;

	assert(table);
	assert(name);
	name = nt_lookup(name,TRUE);
	if (check_dup(table,name)) {
		ERROR(E_WARN+7,name);
		return (NULLSYM);
	}
	s = new_symbol(kind);
	s->s_name = name;
	at_append(table,s);
	return (s);
}



/*
 * append s to the aux table at.
 */
static void
at_append(at,s)
Symptr at, s;
{
	assert(at);
	while (at->s_next != NULLSYM)		/* set at to end of table */
		at = at->s_next;
	at->s_next = s;				/* link s into table */
	s->s_next = NULLSYM;
	s->s_prev = at;
}




/* returns true iff name exists in the current symbol table block.
 * name is assumed to be in the name table so that check_dup can be used.
 */
Bool
st_dup(name)
char *name;
{
	if (check_dup(st_cb,name)) {
		return (TRUE);
	}
	/* check the at for the component; see comment on st_lookup. */
	if (st_cb &&
	   (st_cb->s_type == T_SPEC || st_cb->s_type == T_GLOBAL)) {
		assert(st_cb->s_tdef != NULLSYM);
		if (check_dup(st_cb->s_tdef,name)) {
			return (TRUE);
		}
	}
	return FALSE;
}



/* return TRUE iff name is already defined in table.
 * name is assumed to be in name table so that == can be used
 * instead of strcmp.
 */
static Bool
check_dup(table,name)
Symptr table;
char *name;
{
	assert(name != NULL);
	while (table != NULL) {
		if (name == table->s_name)
			return (TRUE);
		table = table->s_next;
	}
	return (FALSE);
}



/* look up entry name in symbol table.
 * return pointer to entry if found; otherwise return NULLSYM.
 * succeed if name is found in primary symbol table
 * or if name is only imported object with that name.
 */
Symptr
st_unqual_lookup(name,err)
char *name;
int err;
{
	Symptr ret;
	char *nm;

	ret = st_lookup(name);
	if (ret != NULLSYM)
		return(ret);
	if ((nm = nt_lookup(name,FALSE)) != NULL)
		ret = st_import_lookup(nm);
	if (ret == NULLSYM)
		ERROR(E_FATAL+err,name);
	return (ret);
}



/* check for name in all imports and make sure it is unique.
 * check s_tdef and s_next of current component.
 * note: no imports in predefs area.
 */
static Symptr
st_import_lookup(name)
char *name;
{
	Symptr ret,s,t;
	int i;
	assert(comp_sym);

	ret = NULLSYM;
	/* do the below first for
	 * comp_sym->s_next and then comp_sym->s_tdef
	 * pretty ugly due to possible early return.
	 */
	s = comp_sym->s_next;
	for (i = 1; i <= 2; i++) {
		for (;s; s = s->s_next) {
			if (s->s_kind == K_IMPORT) {
				assert (s->s_type == T_SPEC
				       || s->s_type == T_GLOBAL);
				assert(s->s_tdef != NULLSYM);
				if (((t=at_lookup(s->s_tdef,name))!=NULLSYM)
				    && (t->s_kind == K_CONST
					|| t->s_kind == K_LITERAL
					|| t->s_kind == K_TYPE
					|| t->s_kind == K_OPTYPE)) {
					if (ret != NULL) {
						errmsg(E_FATAL,
				"ambiguous unqualified imported name: %s",name);
						return (NULLSYM);
					}
					ret = t;
				}
			}
		}
		s = comp_sym->s_tdef; /* since loop back only once. */
	}
	return ret;
}



/* look up entry name in primary symbol table.
 * return pointer to entry if found;
 * otherwise return NULLSYM.
 * The aux table of the component being parsed contains
 * the declarations from the spec.
 * So, the search starts looks at each entry in the primary st
 * until the current component is found, its aux table is then
 * searched, then the search resumes in the primary st
 * (in what should be the predef area).
 */
Symptr
st_lookup(name)
char *name;
{
	register Symptr s,t;

	assert(name!=NULL);
	name = nt_lookup(name,FALSE);
	if (name == NULL)
		return (NULLSYM);
	s = st_end;
	while (s != NULLSYM) {
		if (name == s->s_name)
			return (s);
		if (s == comp_sym) {
			assert(s->s_type == T_SPEC || s->s_type == T_GLOBAL);
		    	assert(s->s_kind == K_BLOCK);
			assert(s->s_tdef != NULLSYM);
			if ((t=at_lookup(s->s_tdef,name))!=NULLSYM)
				return (t);
		}
		s = s->s_prev;
	}
	return (NULLSYM);
}




/* look up entry name in auxiliary table.  return pointer to entry if found;
 * otherwise return NULLSYM
 */
Symptr
at_lookup(table,name)
Symptr table;
char *name;
{
	register Symptr s;

	assert(name!=NULL);
	name = nt_lookup(name,FALSE);
	if (name == NULL)
		return (NULLSYM);
	s = table;
	while (s != NULLSYM) {
		if (name == s->s_name)
			return (s);
		s = s->s_next;
	}
	return (NULLSYM);
}



/* pop_block()
 * pop current block off primary symbol table.
 * note: the popped block is made auxiliary until freed
 * after the component is compiled.
 */
void
pop_block()
{
	assert(nest_level > 0 && st_cb != 0);
	--nest_level;

	st_cb = st_end = st_cb->s_prev;		/* reset st_end */
	if (st_end) {
	    st_end->s_next = NULLSYM;
	    while (st_cb->s_kind != K_BLOCK) {
		    st_cb = st_cb->s_prev;
	    }
	}
}



/* push_block()
 * push a block onto the primary symbol table.
 * used to push block for separate body back onto st.
 * note: block not already on lb (since inter_depth == 1)
 * so need to put it there.
 * also, it's s_next is NULLSYM, so don't need to change st_end
 * more than st_append does.
 */
void
push_block(b)
Symptr b;
{
    assert (b && b->s_kind == K_BLOCK && b->s_next == NULLSYM);
    assert (nest_level == 1);

    st_append(b);
    lb_append(b);
}



/* pop_component()
 * pop all blocks in current component off primary symbol table.
 * this is used so imports get read in in predef environment.
 */
Symptr
pop_component()
{
    Symptr ret;

    assert (nest_level > 0);

    nest_level = 1;

    ret = st_end_predefs->s_next;
    st_end = st_end_predefs;
    st_cb = st_top;
    assert (ret);
    return ret;
}



/* push_component()
 * push a component back onto the primary symbol table.
 * used to restore state after import.
 * if block belongs on lb, it is already there.
 */
void
push_component(b)
Symptr b;
{
	assert (b && b->s_kind == K_BLOCK);
	assert (nest_level == 1);
	assert (st_end_predefs->s_next == NULLSYM);
	assert (b->s_prev == st_end_predefs);
	st_end_predefs->s_next = b;
	for (; b != NULLSYM; b = b->s_next) {
		/* alternatively, save/restore nest_level in read_inter.
		 * though still need to set st_end.
		 * (could save/restore that too, or pass back from pop.)
		 */
		if (b->s_kind == K_BLOCK) {
			nest_level++;
			st_cb = b;
		}
		st_end = b;
	}

}



/* returns storage allocated during pass over each component
 * note: the predef block is on the lb only for the first component being
 * compiled, which is ok since assign_offset doesn't use that anyway.
 * note: imports (which are recursive) don't get put on lb.
 */

void
tidy_sym()
{
	if (inter_depth != 0)
		return;

	/* free up the st structs.
	 * obviously, more than this code is required to reclaim all structures.
	 * note that the predefined block gets removed from lb first time this
	 * procedure is executed; this is ok for now, but when structures are
	 * actually reclaimed, be careful not to reclaim stuff in that block.
	 */

	lb_end = (Local_blockptr) NULL;
}



/* make note of the end of the predefs */

void
note_end_predefs()
{
    st_end_predefs = st_end;
}
