/* sizes.c -- get info about sizes */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"
#include "tokmacs.h"
#include "../srsys.h"



/*  get_size(e) - get size of e; return SIZE_UNK if not known at compile time */
int
get_size(e)
Nodeptr e;
{
    register Symptr esig = e->e_sig;
    int n;
    
    /* a few things get recognized by their TK_ values */
    switch (e->e_op) {
    case TK_CLONE:
	if (e->e_l->e_op != TK_NUMBER)
	    return SIZE_UNK;
	n = get_size(e->e_r);
	if (n == SIZE_UNK)
	    return n;
	else
	    return e->e_l->e_i * n;
    case TK_FILE_NULL:
    case TK_FILE_NOOP:
	return FILESIZE;
    case TK_PTR_NULL:
	return PTRSIZE;
    case TK_NULL:
    case TK_NOOP:
    case TK_OPCAP_NULL:
    case TK_OPCAP_NOOP:
	return OP_CAP_SIZE; 
    case TK_RESCAP_NULL:
    case TK_RESCAP_NOOP:
	return PTRSIZE;
    case TK_VMCAP_NULL:
    case TK_VMCAP_NOOP:
	return VMSIZE;
    case TK_CONST:
    case TK_ASSIGN:
    case TK_SWAP:
	return get_size(e->e_l);
    case TK_STRLIT:
	return align(e->e_str->str_len + STR_OVH);
    }

    /* the rest by their signatures */
    if (esig->s_size == SIZE_UNK || esig->s_size == SIZE_ARB)
	return SIZE_UNK;
    if (esig->s_kind == K_OP)
	return esig->s_size * OP_CAP_SIZE;

    switch (esig->s_type) {
    case T_BOOL:
	return esig->s_size * BOOLSIZE;
    case T_CHAR:
	return esig->s_size * CHARSIZE;
    case T_INT:
    case T_FILE:
	return esig->s_size * INTSIZE;
    case T_ENUM:
	return esig->s_size * ENUMSIZE;
    case T_CAP:
	if (esig->s_tdef->s_type == T_SPEC)	/* resource cap */
	    return esig->s_size * esig->s_tdef->s_size;
	else					/* op capability */ 
	    return esig->s_size * OP_CAP_SIZE;
    case T_REC:
    case T_STRING:
	if (!esig->s_tdef)
	    return SIZE_UNK;
	n = esig->s_tdef->s_size;
	if (n == SIZE_UNK || n == 0)	/* shouldn't be 0, but just in case...*/
	    return SIZE_UNK;
	return esig->s_size * n;
    case T_PTR:
	return esig->s_size * PTRSIZE;
    case T_VOID:
	return SIZE_UNK;
    default:
	boom("bad type in get_size");
	/*NOTREACHED*/
    }
}



/*  get_ob_size(s) - get size of an individual object
    (which may be shaped into an array..) */
int
get_ob_size(s)
Symptr s;
{
    if (s->s_kind == K_OP)
	return OP_CAP_SIZE;
    else if (s->s_kind == K_SEMAPHORE)
	return OP_CAP_SIZE;
	
    switch (s->s_type) {
	case T_BOOL:
	    return BOOLSIZE;
	case T_CHAR:
	    return CHARSIZE;
	case T_INT:
	case T_FILE:
	    return INTSIZE;
	case T_ENUM:
	    return ENUMSIZE;
	case T_CAP:
	    if (s->s_tdef->s_type == T_SPEC) {
	    	return s->s_tdef->s_size;
	    } else if (s->s_tdef->s_tdef && 
	    		s->s_tdef->s_tdef->s_type == T_SPEC){
		return s->s_tdef->s_tdef->s_size;
	    } else {		/* op capability */ 
		return OP_CAP_SIZE;
	    }
	case T_REC:
	case T_STRING:
	    return s->s_tdef->s_size;
	case T_PTR:
	    return PTRSIZE;
	case T_VOID:
	    return 0;
	case T_FUNC:
	    boom("can't get size of this object");
	    /*NOTREACHED*/
	case T_NULL:
	case T_NOOP:
	    return sizeof(opcap);
	default:
	    ERROR(E_FATAL+4, typetos(s->s_type));
	    boom("get_ob_size: don't recognize this object");
	    /*NOTREACHED*/
    }
}
