/* signature.c -- operator typechecking and signature synthesis */

#include <stdio.h>

#include "sr.h"
#include "funcs.h"
#include "globals.h"

#define MAX_RECURSION 10	/* stop infinite recursion in sigcmps */



static Bool rec_compare(), resource_compare(), func_compare();
static Bool fix_const(), default_sig(), restrict_compatible();


Bool
sigcmp(type1, type2, tree1, tree2, depth)
Symptr type1, type2;
Nodeptr tree1, tree2;
int depth;
{
    Symptr tdef1, tdef2;

    /* check first for built-in types */

    if (type1 == NULLSYM && type2 == NULLSYM)
	return TRUE;	/* i guess */
    else if (type1 == NULLSYM || type2 == NULLSYM)
	return FALSE;

    /* check recursion depth exceeded and exit if so.
       this gets us out of recursive definitions */
    if (++depth > MAX_RECURSION)
	return TRUE;

    switch (type1->s_type) {
    case T_NULL:
    case T_NOOP:
	if (type2->s_type == T_NULL || type2->s_type == T_NOOP)
	    return TRUE;
	else
	    return sigcmp (type2, type1, tree2, tree1, depth);
    case T_STAR:
	return TRUE;
    case T_INT:
    case T_BOOL:
	return (type1->s_type == type2->s_type);
    case T_CHAR:
    case T_STRING:
	return (type1->s_type == type2->s_type) 	/* both same, or */
	    || (is_string(type1) && is_string(type2));	/* char[] and string */
    case T_FILE:
    	return (type2->s_type == T_FILE)
	    || (fix_const(type2,tree2,TK_FILE_NULL,TK_FILE_NOOP,NULLSYM));
    case T_FUNC:
    case T_VOID:
	if (type2->s_type == T_NULL || type2->s_type == T_NOOP)
	    return TRUE;
	else if (type1->s_type != type2->s_type)
	    return FALSE;
	else
	    return func_compare(type1->s_tdef, type2->s_tdef, TRUE, depth);
    case T_ANON:
	if (type2->s_type != T_ANON)
	    return FALSE;
	else
	    return sigcmp(type1->s_tdef,type2->s_tdef,NULLNODE,NULLNODE,depth);
    case T_ENUM:
	if (type2->s_type != T_ENUM)
	    return FALSE;
	tdef1 = type1->s_tdef;
	tdef2 = type2->s_tdef;
	while (tdef1 || tdef2)  {
	    if (tdef1 == tdef2)
		return TRUE;
	    if (strcmp(tdef1->s_name,tdef2->s_name) != 0)
		return FALSE;
	    tdef1 = tdef1->s_next;
	    tdef2 = tdef2->s_next;
	}
	return (!tdef1 && !tdef2);
    case T_REC:
    case T_UNION:
	if (type1->s_type != type2->s_type)
	    return FALSE;
	else
	    return rec_compare
		(type1->s_tdef->s_tdef, type2->s_tdef->s_tdef, depth);
    case T_CAP:  {
	if (!(tdef1 = type1->s_tdef))
	    return TRUE;	/* err in definition caught earlier */
	if (tdef1->s_kind == K_BLOCK || tdef1->s_kind == K_IMPORT) {
	    /* resource capability */
	    if (fix_const(type2,tree2,TK_RESCAP_NULL,TK_RESCAP_NOOP,type1))
		return TRUE;
	    else if (type2->s_type != T_CAP)
		return FALSE;
	    tdef2 = type2->s_tdef;
	    if (tdef1->s_type != T_SPEC || tdef2->s_type != T_SPEC)
		return FALSE;
	    else
		return resource_compare(tdef1,tdef2,depth);
	} else if (tdef1->s_kind == K_VM)	 {
	    /* VM capability */
	    return (fix_const(type2,tree2,TK_VMCAP_NULL,TK_VMCAP_NOOP,type1))
		|| (type2->s_type == T_CAP && type2->s_tdef->s_kind == K_VM);
	} else if (tdef1->s_kind == K_OP) {   /* operation equivalence */
	    if (fix_const(type2,tree2,TK_OPCAP_NULL,TK_OPCAP_NOOP,type1))
		return TRUE;
	    else if (type2->s_type != T_CAP && type2->s_kind != K_OP)
		return FALSE;
	    else if (type2->s_type == T_CAP
		    && tdef1->s_type != type2->s_tdef->s_type)
		return FALSE;
	    else if (type2->s_kind == K_OP && tdef1->s_type != type2->s_type)
		return FALSE;
	    else
		return func_compare(tdef1->s_tdef,
		    (type2->s_type == T_CAP)?
			type2->s_tdef->s_tdef: type2->s_tdef,
		    (type1->s_tdef->s_type == T_FUNC),
		    depth);
	} else if (tdef1->s_kind == K_ANON) {
	    return sigcmp(tdef1,type2->s_tdef,NULLNODE,NULLNODE,depth);
	} else
	    boom("bad kind in sigcmp for T_CAP");
	    /*NOTREACHED*/
    }
    case T_PTR:
	if (type2->s_type == T_NULL) {
	    replace(tree2, TK_NULL, TK_PTR_NULL, NULLSYM);
	    return TRUE;
	}
	if (type2->s_type != T_PTR)
	    return FALSE;
	tdef1 = type1->s_tdef;
	tdef2 = type2->s_tdef;
	if (tdef2 == NULLSYM)
	    return TRUE;	/* e.g. alloc() */

	if (tdef1->s_kind == K_ANON && tdef1->s_type == T_PTR)
	    tdef1 = tdef1->s_tdef;
	if (tdef2->s_kind == K_ANON && tdef2->s_type == T_PTR)
	    tdef2 = tdef2->s_tdef;

	if (tdef1->s_type == T_ANY || tdef2->s_type == T_ANY)
	    return TRUE;
	if (tdef1->s_type != tdef2->s_type
	    && (tdef1->s_type == T_CHAR || tdef1->s_type == T_STRING))
	    return FALSE;	/* ptr char incompatible with ptr string */
	return sigcmp(tdef1,tdef2,NULLNODE,NULLNODE,depth);
    default:
	boom("bad type in signature compare");
	/*NOTREACHED*/
    }
    /*NOTREACHED*/
}

static Bool
fix_const(type2, tree2, nulltok, nooptok, sig)
Symptr type2;
Nodeptr tree2;
Token nulltok, nooptok;
Symptr sig;
{
    if (type2->s_type == T_NULL)  {
	replace(tree2,TK_NULL,nulltok,sig);
	return TRUE;
    } else if (type2->s_type == T_NOOP)  {
	replace(tree2,TK_NOOP,nooptok,sig);
	return TRUE;
    } else
	return FALSE;
}

static Bool
resource_compare(name1, name2, depth)
Symptr name1, name2;
int depth;
{           /* name1, name2 point to auxiliary table that contains
		the stuff in the spec of each of the resources */
    if (name1 == name2)
	return TRUE;        /* name equivalence is easy */
    else if (!name1 || !name2)
	return FALSE;
    else {
	Symptr field_name1, field_name2;
	assert (name1->s_tdef && name2->s_tdef);
	field_name1 = name1->s_tdef->s_next;
	field_name2 = name2->s_tdef->s_next;
	for (;;)  {
	    while (field_name1 && field_name1->s_kind != K_OP)
		field_name1 = field_name1->s_next;
	    while (field_name2 && field_name2->s_kind != K_OP)
		field_name2 = field_name2->s_next;
	    if (!field_name1 || !field_name2)
		break;
	    /* compare two ops. */
	    if (!func_compare(field_name1->s_tdef, field_name2->s_tdef,
		    (field_name1->s_type == T_FUNC), depth))
		return FALSE;
	    field_name1 = field_name1->s_next;
	    field_name2 = field_name2->s_next;
	}
	return !field_name1 && !field_name2;
    }
}

static Bool
rec_compare(name1, name2, depth)
Symptr name1, name2;
int depth;
{           /* name1, name2 point to auxiliary table that contains
		names of each of the record fields */
    if (name1 == name2)
	return TRUE;        /* name equivalence is easy */
    else if (!name1 || !name2)
	return FALSE;
    else {
	Symptr field_name1, field_name2;

	for (field_name1 = name1->s_next, field_name2 = name2->s_next;
	    field_name1 != NULLSYM && field_name2 != NULLSYM;
	    field_name1 = field_name1->s_next,
	    field_name2 = field_name2->s_next) {
	    if (!sigcmp(field_name1, field_name2, NULLNODE, NULLNODE, depth))
		return FALSE;
	}
	return !field_name1 && !field_name2;
    }
}



static Bool
func_compare(name1, name2, chk_result, depth)
Symptr name1, name2;
Bool chk_result;
int depth;
{
    if (name1 == name2)
	return TRUE;        /* name equivalence is easy */
    else if (name1 == NULLSYM || name2 == NULLSYM)
	return FALSE;
    else {
	Symptr field_name1, field_name2;

	for (field_name1 = name1->s_next, field_name2 = name2->s_next;
	    field_name1 != NULLSYM && field_name1->s_kind == K_PARAM
	    && field_name2 != NULLSYM && field_name2->s_kind == K_PARAM;
	    field_name1 = field_name1->s_next,
	    field_name2 = field_name2->s_next) {
	    if (!sigcmp(field_name1, field_name2, NULLNODE, NULLNODE, depth))
		return FALSE;
	}
	if (chk_result) {
	    if (field_name1 == NULLSYM || field_name2 == NULLSYM)
		return FALSE;
	    else if (field_name1->s_next != NULLSYM ||
		     field_name2->s_next != NULLSYM)
		return FALSE;
	    else
		return sigcmp(field_name1,field_name2,NULLNODE,NULLNODE,depth);
	} else if (field_name1 != NULLSYM || field_name2 != NULLSYM)
	    return FALSE;
	else
	    return TRUE;
    }
}



Bool
check_assign(left, right)
Nodeptr left, right;
{
    /* check_assign should generate bounds-checking requests
	wherever sizes are not known at compile time */

    if (!left || !right)
	return FALSE;
    if (left->e_sig->s_type == T_NOTYETATYPE)
	return default_sig(left,right);
    else if (sigcmp(left->e_sig, right->e_sig, left, right, 0) == FALSE)
	return FALSE;
    if (left->e_op != TK_FORMAL && right->e_op != TK_FORMAL
	    && !check_size(left, right))
	return FALSE;
    if (right->e_sig->s_type != T_NOOP
	     && right->e_sig->s_type != T_NULL
	     && ! restrict_compatible(left->e_sig->s_restrict,
				      right->e_sig->s_restrict))
	return FALSE;
    return TRUE;
}



/* default_sig(left,right) - set type of left identifier if possible */
static Bool
default_sig(left,right)
Nodeptr left, right;
{
    Symptr sy, ls, rs;

    assert (left->e_op == TK_IDENTIFIER);
    sy = left->e_s;
    ls = left->e_sig;
    rs = right->e_sig;
    if (rs->s_type == T_NULL || rs->s_type == T_NOOP) {
	FATAL("type required for initialization to null or noop");
	sy->s_type = ls->s_type = T_FILE;	/* error recovery */
	sy->s_tdef = ls->s_tdef = NULLSYM;
    } else {
	sy->s_type = ls->s_type = rs->s_type;
	sy->s_tdef = ls->s_tdef = rs->s_tdef;
    }
    if (rs->s_size == 1)
	ls->s_ranges = NULL;
    else
	add_range(ls,bnode(TK_RANGE,numnode(1),numnode(rs->s_size)),NULLNODE);
    sy->s_ranges = ls->s_ranges;
    if (dbflags['I'])  {
	printf("implicitly typed:\n");
	pnode(left);
	pnode(right);
	printf("\n");
    }
    return TRUE;
}



/* can an object with restriction right be assigned to one with restriction
   left??
   */

static Bool
restrict_compatible(left, right)
Restrict left, right;
{
    switch (left) {
    case R_CALL:
	return (Bool)(right == R_CALL || right == R_CALLSEND);
    case R_SEND:
	return (Bool)(right == R_SEND || right == R_CALLSEND);
    case R_CALLSEND:
	return (Bool)(right == R_CALLSEND);

    case R_PUBLIC:
	return TRUE;
    case R_PRIVATE:
	return FALSE;
    default:
	return (Bool)(left == right);
    }
}
