/* seqctrl.c -- Sequential control statements: if, do, fa */

/* note: error recovery here is minimal. */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"

static Instptr quant(), quant_ic();



void
if_stmt()
{
	Bool did_first;		/* true after start of first arm */
	Bool else_present;	/* true if else command body present */
	int next_guard;	/* label of next guard (or end of loop if all false) */
	int end_if;	/* label to goto after executing an arm of the if */
	static Token follow[] = {TK_SQUARE,TK_FI,TK_NOTATOKEN};

	did_first = FALSE;
	else_present = FALSE;
	next_guard = 0;
	end_if = NEWLAB;

	do {
		if (did_first) {
			emit(I_BRANCH,NULLNODE,end_if);
			emit(I_LABEL,NULLNODE,next_guard);
		}

		if (maybe(TK_ELSE)) {
			else_present = TRUE;
			if (!did_first)
				FATAL("else can't be first arm of if");
		}
		else {
			emit(I_BRANCH_FALSE,bool_expr(),(next_guard=NEWLAB));
		}

		mustbe(TK_ARROW,"->");
		new_command_block(follow);
		assert (tok == TK_SQUARE || tok == TK_FI);

		did_first = TRUE;

	} while (tok != TK_FI && !else_present);

	if (else_present) {
		if (tok != TK_FI)
/* not really a warning; need general philosophy for errors. */
/* want to skip to FI? */
			FATAL("else must be last arm of if");
	}
	else
		emit(I_LABEL,NULLNODE,next_guard);

	emit(I_LABEL,NULLNODE,end_if);
}


void
do_stmt()
{
	Bool did_first;		/* true after start of first arm. */
	Bool else_present;	/* true if else command body present. */
	int next_guard;		/* label of next guard. */
	int old_next, old_exit; /* old next_label and exit_label. */
	static Token follow[] = {TK_SQUARE,TK_OD,TK_NOTATOKEN};

	old_next = next_label;
	old_exit = exit_label;
	next_label = NEWLAB;
	exit_label = NEWLAB;
	did_first = FALSE;
	else_present = FALSE;
	next_guard = 0;
	emit(I_LABEL,NULLNODE,next_label);
	emit(I_LOOPTOP,NULLNODE,0);

	do {
		if (did_first)
			emit(I_LABEL,NULLNODE,next_guard);

		if (maybe(TK_ELSE)) {
			else_present = TRUE;
			if (!did_first)
				FATAL("else can't be first arm of do");
		}
		else {
			emit(I_BRANCH_FALSE,bool_expr(),(next_guard=NEWLAB));
		}

		mustbe(TK_ARROW,"->");
		new_command_block(follow);
		assert (tok == TK_SQUARE || tok == TK_OD);

		emit(I_BRANCH,NULLNODE,next_label);

		did_first = TRUE;

	} while (tok != TK_OD && !else_present);

	if (else_present) {
		if (tok != TK_OD)
/* not really a warning; need general philosophy for errors. */
/* want to skip to OD? */
			FATAL("else must be last arm of do");
	}
	else
		emit(I_LABEL,NULLNODE,next_guard);

	/* emit bottom label. used only by exit statements. */
		emit(I_LABEL,NULLNODE,exit_label);

	next_label = old_next; exit_label = old_exit;
}


/* note: if exiting or nexting an iterative statement
 * from within an input statement, must terminate that input statement too.
 * this is taken care of in input statements.
 */
void
exit_stmt()
{
	if (exit_label == NOLAB)
		WARN("exit outside iterative or co statement ignored");
	else{
		emit(I_BRANCH,NULLNODE,exit_label);
		unreachable = TRUE;
	}
}

void
next_stmt()
{
	if (next_label == NOLAB)
		WARN("next outside iterative or co statement ignored");
	else{
		emit(I_BRANCH,NULLNODE,next_label);
		unreachable = TRUE;
	}
}

/* reminder what exit and next mean in fa stmt.
 * e.g., fa i := 1 to 3, j := 5 to 6 -> ... af
 * exit jumps out of both implicit loops;
 * next causes next combination of bound variables.
 */

void
stop_stmt()
{
    Nodeptr e = NULLNODE;
    if (maybe(TK_LEFTPAREN)) {
	e = expr();
	mustbe(TK_RIGHTPAREN,")");
    };
    emit(I_STOP, e ? e : numnode(0), NOLAB);
    unreachable = TRUE;
}
    
void
for_all_stmt()
{
	static Token follow[] = {TK_AF,TK_NOTATOKEN};
	Instptr ic_quantifier_guts, ic_before;
	int old_exit, old_next;

	old_next = next_label; old_exit = exit_label;
	next_label = NEWLAB; exit_label = NEWLAB;

	/* make a block for the quantifiers. */
	block_begin(T_CMD);

	ic_quantifier_guts = quantifier(TRUE);

	mustbe(TK_ARROW,"->");

	ic_before = ic_mark();
	new_command_block(follow);
	assert (tok == TK_AF);

	/* patch the command body into the inner most quantifier loop.
	 */
	if (ic_quantifier_guts != NULLINST) 
		ic_move(ic_before,ic_quantifier_guts);

	/* emit bottom label. used only by exit statements. */
	emit(I_LABEL,NULLNODE,exit_label);

	next_label = old_next; exit_label = old_exit;

	/* done with the quantifier block. */
	block_end();

}

/* this quantifier routine is used by fa statement,
 * concurrent invocation, and input statement.
 * caller is responsible for making new block
 * in which to place these identifiers
 * and popping that block when appropriate.
 * this routine generates only looping code;
 * returns pointer to where the guts of the command body
 * should be patched in by caller.
 * The for_all parameter is TRUE iff quantifier is part of for_all
 * statement;
 * special code needs to be generated to accommodate exit and next.
 */
Instptr
quantifier(for_all)
Bool for_all;
{
	Instptr ic_guts, last_guts, this_start;
	Bool good; /* so far, so good. */

	last_guts = NULLINST;
	good = TRUE;
	do {
		this_start = ic_mark();
		ic_guts = quant();
		if (ic_guts == NULLINST)
			good = FALSE;
		if (good) {
			/* patch this quantifier into previous one, if any. */
				if (last_guts != NULLINST) {
					assert(this_start->i_next != NULLINST);
					ic_move(this_start,last_guts);
				}
			last_guts = ic_guts;
		}
	} while (maybe(TK_COMMA));

	/* for a fa statement, generate the next label.
	 * it goes on the increment/decrement of the inner most loop,
	 * which is right after where the body will be put (after ic_guts).
	 * so, keep ic_guts where it is, but add label thereafter.
	 * also generate the looptop check.
	 */
		if (for_all && good) {
			Instptr old_tail;
			old_tail = ic_mark();
			assert (ic_guts != NULLINST);
			emit(I_LABEL,NULLNODE,next_label);
			emit(I_LOOPTOP,NULLNODE,0);
			ic_move(old_tail,ic_guts);
		}


	assert ((good && (ic_guts!=NULLINST)) || !good);
	return (ic_guts);
}

/* quant()
 * parses a single quantifier.
 * returns pointer to where inner quantifier or loop body should go.
 * note: enforcing that expressions have ordered types
 * is done after get expressions, not when we get them like
 * in other places.
 */
static Instptr
quant()
{
	Symptr bound;
	Nodeptr initial_expr, final_expr, such_that;
	Token direction;
	Bool bad; /* true if encountered a fatal error; false otherwise.
		   * just so to save us from generating intermediate code.
		   */

	bad = FALSE;

	/* bound_variable. */
	if (get_token() != TK_IDENTIFIER
	    || (bound=st_install(tk_str,K_VAR)) == NULLSYM) {
			FATAL("bad/missing id in quantifier");
			return (NULLINST);
		}
		bound->s_type = T_NOTYETATYPE;

	if (get_token() != TK_ASSIGN)
		WARN("missing := in quantifier");

	/* initial_expression. */
		if ((initial_expr=expr()) == NULLNODE) {
			FATAL("bad initial expr in quantifier");
			initial_expr = numnode(1);	/* error recovery */
			bad = TRUE;
		}

	get_token();
	if (tok != TK_TO && tok != TK_DOWNTO) {
		WARN("missing to/downto in quantifier");
		/* error recovery. */
			direction = TK_TO;
		putback();
	}
	else
		direction = tok;


	/* final_expression. */
		if ((final_expr=expr()) == NULLNODE) {
			FATAL("bad final expr in quantifier");
			final_expr = numnode(1);	/* error recovery */
			bad = TRUE;
		}

	/* compare signatures of initial and final expressions.
	 * use signature to assign type to identifier.
	 * assume error recovery has made initial and final expressions.
	 */
		assert (initial_expr != NULLNODE);
		assert (final_expr != NULLNODE);
	{
		Symptr iv_sig, fv_sig;
		assert(bound!=NULLSYM && bound->s_type == T_NOTYETATYPE);
		iv_sig = initial_expr->e_sig;
		fv_sig = final_expr->e_sig;
		assert (iv_sig != NULLSYM);
		assert (fv_sig != NULLSYM);
		if (!sigcmp(iv_sig,fv_sig,initial_expr,final_expr,0)){
			FATAL("initial/final signature conflict");
			/* use the initial one so can keep going. */
				release_node(final_expr);
				final_expr = initial_expr;
				bad = TRUE;
		}
		if (!IS_ORDERED(iv_sig->s_type) || !IS_SCALAR(iv_sig)){
		    FATAL("initial/final expression not (scalar) ordered type");
			/* fabricate a reasonable type for the id
			 * so that can
			 * (1) continue with such that below
			 * (2) better handle uses of bound variable.
			 *     e.g., so i has a type in a[i] in fa stmt
			 *     otherwise sigcmp chokes.
			 */
				bound->s_type = T_INT;
				bound->s_tdef = NULL;
				bound->s_ranges = NULL;
		    
			bad = TRUE;
		}
		else {
			/* fill in the id's type. */
				assert (iv_sig->s_size == 1);
				assert (iv_sig->s_ranges == NULL);
				bound->s_type = iv_sig->s_type;
				bound->s_tdef = iv_sig->s_tdef;
				bound->s_ranges = NULL;
		}

		bound->s_used = TRUE;
	}

	/* this must be after set type of bound variable
	 * because it might be used in the such_that expression.
	 */
		if (maybe(TK_SUCHTHAT))
			such_that = bool_expr();
		else {
			such_that = NULLNODE;
		}

	/* finally, generate the intermediate code
	 * unless already FATAL due to bad id type.
	 */
		return (bad ? NULLINST :
			quant_ic(bound,initial_expr,final_expr,
				direction,such_that));
}

/* this routine actually generates the intermediate code for a quantifier.
 * note: this routine creates an id node for each occurrence of bound
 * in the intermediate code.
 * Kelvin told me to do it this way because
 * we don't want several pointers to same node.
 */
static Instptr
quant_ic(bound, initial_expr, final_expr, direction, such_that)
Symptr bound;
Nodeptr initial_expr, final_expr;
Token direction;
Nodeptr such_that;
{
	/* labels of loop test, loop increment, and loop exit. */
		int loop_test, loop_inc_dec, loop_exit;
	Nodeptr n;
	Instptr ic_guts; /* where body of loop should go. */

	assert (bound != NULLSYM);
	assert (initial_expr != NULLNODE && final_expr != NULLNODE);

	loop_test = NEWLAB;
	loop_inc_dec = NEWLAB;
	loop_exit = NEWLAB;

	/* assign initial value. */
		n = idnode(bound);
		emit(I_EXPR,bnode(TK_ASSIGN,n,initial_expr),NOLAB);

	/* compare against final. */
		emit(I_LABEL,NULLNODE,loop_test);
		n = idnode(bound);
		n = bnode((direction==TK_TO)?TK_GT:TK_LT, n, final_expr);
		emit(I_BRANCH_TRUE,n,loop_exit);

	/* evaluate such that. */
		if (such_that != NULLNODE)
			emit(I_BRANCH_FALSE,such_that,loop_inc_dec);

	/* remember where the next quantifier or loop body should go. */
		ic_guts = ic_mark();

	/* increment/decrement loop variable.*/	
		emit(I_LABEL,NULLNODE,loop_inc_dec);
		n = idnode(bound);
		n = bnode((direction==TK_TO)?TK_INCREMENT:TK_DECREMENT,
				     n, NULLNODE);
		emit(I_EXPR,n,NOLAB);

	/* goto top of loop. */
		emit(I_BRANCH, NULLNODE, loop_test);

	/* end of loop. */
		emit(I_LABEL,NULLNODE,loop_exit);

	return (ic_guts);
}
