/*  print.c -- print structures on stdout for debugging  */


#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "tokmacs.h"
#include "names.h"
#include "globals.h"

static char *numtos();

#define DENT 2
#define INDENT spacing -= DENT
#define UNDENT spacing += DENT
#define SPACEOUT printf(spacing);

static char spaces[] =
    "                                                                        ";
static char *spacing = spaces + sizeof(spaces) - 1;



/*  pnode(e) - print a node and its descendents */

void
pnode (e)
Nodeptr e;
{
    Token t;

    SPACEOUT;
    if (!e) {
	printf("null Nodeptr\n");
	return;
    }
    t = e->e_op;
    printf ("%s (", tokentos(t));

    switch (LEFT_CHILD(t)) {
	case N_SYM:
	    printf("\"%s\"", (e->e_s && e->e_s->s_name) ? e->e_s->s_name : "");
	    break;
	case N_NODE:
	    printf("%s", e->e_l ? tokentos(e->e_l->e_op) : "NULL");
	    break;
	case N_CON:
	    printf("%s",numtos(e));
	    break;
	case N_NIL:
	    break;
    }

    if (RIGHT_CHILD(t) == N_NODE)
	printf(", %s", e->e_r ? tokentos(e->e_r->e_op) : "NULL");

    printf(") [%s] %X\n", e->e_sig ? typetos(e->e_sig->s_type) : "NOSIG", e);

    INDENT;
    switch (LEFT_CHILD(t)) {
	case N_SYM:	psym(e->e_s);	break;
	case N_NODE:	pnode(e->e_l);	break;
	case N_CON:			break;
	case N_NIL:			break;
    }
    if (RIGHT_CHILD(t) == N_NODE)
	pnode(e->e_r);
    if (e->e_sig)
	psym(e->e_sig);
    UNDENT;
}



/*  psym(s) - print a symbol and its descendents */

void
psym (s)
Symptr s;
{
    SPACEOUT;
    if (!s) {
	printf("null Symptr");
	return;
    }
    printf("SYM \"%s%s\": %s %s size=%d, ", s->s_name ? s->s_name : "",
	s->s_ranges ? "[]" : "",
	kindtos(s->s_kind), typetos(s->s_type), s->s_size);
    if (s->s_offset == OFF_UNK)
	printf("OFF_UNK");
    else
	printf("offset=%d",s->s_offset);
    printf(" %X\n",s);
    INDENT;
    if (s->s_tdef)
	psym(s->s_tdef);
    UNDENT;
}




/*
 * class_print() - print list of classes and members.
 *
 * Unfortunately, doesn't give level where operation is declared, just name.
 */
void
class_print()
{
    Classptr cl;
    Symptr sym;

    if (!dbflags['C'])
	return;
    if (classes == NULLCLASS)
	printf("no classes.\n");
    else {
	printf("classes:\n");
	for (cl = classes; cl != NULLCLASS; cl = cl->cl_next) {
	    printf("  equiv class with %d members:", cl->cl_members);
	    for (sym = cl->cl_first; sym != NULLSYM; sym = sym->s_cl_next)
		printf(" %s", sym->s_name);
	    printf("\n");
	}
    }
}



/*  numtos(node) - return a string representation of a constant node  */

static char *
numtos(node)
Nodeptr node;
{
    static char buf[50];

    if (!node)
	return "(NODELESS CONST!)";
    if (!node->e_sig)
	return "(CONST w/ no SIG!)";
    switch (node->e_sig->s_type) {
	case T_BOOL:
	    if (node->e_sig->s_size == 1)
		return node->e_i ? "true": "false";
	    else
		sprintf(buf,"%08Xx",node->e_i);
	    break;
	case T_STRING:
	    sprintf(buf,"'%.40s'", node->e_str->str_data);
	    break;
	case T_INT:
	    sprintf(buf, "%d", node->e_i);
	    break;
	default:
	    sprintf(buf,"const(%s,%d)",typetos(node->e_sig->s_type),node->e_i);
	    break;
    }
    return buf;
}



/*  xxxtos -- convert enum xxx to a string, with error checking  */

char *
tokentos(mytok)
Token mytok;
{
    if (((int) mytok) >= sizeof(token_names)/sizeof(char *)
	    || ((int) mytok) < 0)
	return "TK_?????";
    else
	return token_names[(int) mytok];
}

char *
kindtos(kind)
Kind kind;
{
    if (((int) kind) >= sizeof(kind_names)/sizeof(char *)
	    || ((int) kind) < 0)
	return "K_?????";
    else
	return kind_names[(int) kind];
}

char *
typetos(type)
Type type;
{
    if (((int) type) >= sizeof(type_names)/sizeof(char *)
	    || ((int) type) < 0)
	return "T_?????";
    else
	return type_names[(int) type];
}

char *
segtos(seg)
Segment seg;
{
    if (((int) seg) >= sizeof(segment_names)/sizeof(char*) || (int)seg < 0)
	return "S_?????";
    else
	return segment_names[(int) seg];
}
