/* predef.c -- signature checking and code generation for predefined funcs */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"
#include "../srsys.h"

static Nodeptr	check_file();
static void gen_lbub(), gen_minmax(), gen_getarg(), gen_read(), gen_write();




/*  predef_args(tree) - check args to predefined function  */
Bool
predef_args(op)
Nodeptr op;
{
    Predef pre;
    Nodeptr e;
    int nargs;			/* number of args. */

#define ARG1	(op->e_r)
#define ARG2	(op->e_r->e_r)
#define ARG3	(op->e_r->e_r->e_r)

#define SIGNAT(type,tdef)\
	(op->e_sig->s_type = type,\
	 op->e_sig->s_size = 1,\
	 op->e_sig->s_tdef = tdef)

    assert(op != NULLNODE);
    pre = op->e_l->e_s->s_predef;

    /* count the arguments */
    nargs = 0;
    for (e = op->e_r; e; e = e->e_r)
	nargs++;

    switch (pre) {
	/* standard predefined operations. */
    case PRE_myresource:
	if (nargs != 0 || !parsing_body)
	    return FALSE;
	assert(comp_sym->s_kind==K_BLOCK && comp_sym->s_type==T_SPEC);
	SIGNAT(T_CAP,comp_sym);
	break;
    case PRE_myvm: {
	Symptr res = st_lookup("vm");
	assert(res);
	assert(res->s_kind == K_VM);
	assert(res->s_type == T_SPEC);
	assert(res->s_tdef == NULLSYM);
	op->e_sig = new_symbol(K_VM);
	op->e_sig->s_name = res->s_name;
	op->e_sig->s_type = T_CAP;
	op->e_sig->s_tdef = res;
	op->e_sig->s_restrict = R_NOTARESTRICT;
	op->e_sig->s_size = 1;
	if (nargs != 0)
	    return FALSE;
	break;
	}
    case PRE_mymachine:
	if (nargs != 0)
	    return FALSE;
	SIGNAT(T_INT,NULLSYM);
	break;
    case PRE_lb:
    case PRE_ub:
    case PRE_lb1:
    case PRE_ub1:
    case PRE_lb2:
    case PRE_ub2:
	{
	    Nodeptr top;
	    Bool two;	/* true iff function is for second dimension. */

	    if (nargs != 1)
		return FALSE;
	    top = (ARG1->e_l->e_op == TK_PERIOD)? ARG1->e_l->e_r: ARG1->e_l;
	    if (top->e_op != TK_IDENTIFIER || !top->e_s->s_ranges)
		return FALSE;
	    two = (Bool) (pre == PRE_lb2 || pre == PRE_ub2);
	    if (two && !top->e_s->s_ranges->r_dim2) {
		ERROR(E_FATAL+3,"no second dimension");
		return FALSE;
	    }
	    if (!two) {
		SIGNAT(top->e_s->s_ranges->r_dim1->e_l->e_sig->s_type,
		    top->e_s->s_ranges->r_dim1->e_l->e_sig->s_tdef);
	    } else {
		SIGNAT(top->e_s->s_ranges->r_dim2->e_l->e_sig->s_type,
		    top->e_s->s_ranges->r_dim2->e_l->e_sig->s_tdef);
	    }
	}
	break;
    case PRE_length:
    case PRE_maxlength:
	if (nargs != 1)
	    return FALSE;
	if (!is_string(ARG1->e_sig)) {
	    ERROR(E_FATAL+3,"length/maxlength requires string arg");
	    return FALSE;
	}
	SIGNAT(T_INT,NULLSYM);
	break;
    case PRE_abs:
    case PRE_nap:
	if (nargs != 1)
	    return FALSE;
	if (!IS_SIMPLE(ARG1->e_sig,T_INT))
	    return FALSE;
	SIGNAT(T_INT,NULLSYM);
	break;
    case PRE_max:
    case PRE_min:
	{
	    Nodeptr an_arg;
	    Type arg_type;

	    if (!(an_arg = ARG1))
		return FALSE;
	    else if (!IS_ORDERED(arg_type = an_arg->e_sig->s_type)
		     || !IS_SCALAR(an_arg->e_sig))
		return FALSE;
	    else {
		while (an_arg = an_arg->e_r)
		    if (!IS_SIMPLE(an_arg->e_sig,arg_type))
			return FALSE;
		SIGNAT(arg_type,NULLSYM);
	    }
	}
	break;
    case PRE_low:
    case PRE_high:
	{
	    /* allow for user-defined and predefined enums
	     * and builtin ordered types.
	     */
	    Symptr type_id;	/* points to type definition. */
	    if (nargs!=1)
		return FALSE;
	    if (ARG1->e_l->e_op != TK_IDENTIFIER)    
		return FALSE;
	    type_id = ARG1->e_l->e_s;
	    if (!IS_ORDERED(type_id->s_type))
		return FALSE;
	    /* a bit ugly? but want to set tdef only for enum */
	    SIGNAT(type_id->s_type, (type_id->s_type==T_ENUM)?type_id:NULLSYM);
	}
	break;
    case PRE_pred:
    case PRE_succ:
	{	
	    /* allow for user-defined and predefined enums
	     * and builtin ordered types.
	     */
	    Symptr type_id;	/* points to type definition. */
	    if (nargs != 1)
		return FALSE;
	    type_id = ARG1->e_sig;
	    if (!IS_ORDERED(type_id->s_type) || type_id->s_size != 1)
		return FALSE;
	    /* a bit ugly? but want to set tdef only for enum */
	    SIGNAT(type_id->s_type,
		 (type_id->s_type==T_ENUM)?type_id->s_tdef:NULLSYM);
	}
	break;
    case PRE_free:
	if (nargs != 1)
	    return FALSE;
	if (ARG1->e_sig->s_type == T_NULL)
	    return TRUE;
	if (!IS_SIMPLE(ARG1->e_sig,T_PTR))
	    return FALSE;
	SIGNAT(T_VOID,NULLSYM);
	break;
    case PRE_locate:
	if (nargs != 2 && nargs != 3)
	    return FALSE;
	if (!is_string(ARG2->e_sig))
	    return FALSE;
	if (nargs == 3 && !is_string(ARG3->e_sig))
	    return FALSE;
	SIGNAT(T_VOID,NULLSYM);
	break;

	/* file operations. */
    case PRE_open:
	if (nargs != 2)
	    return FALSE;
	if (!is_string(ARG1->e_sig))
	    return FALSE;
	if (ARG2->e_sig->s_type != T_ENUM 
	    || ARG2->e_sig->s_kind != K_LITERAL
	    || ARG2->e_sig->s_tdef != accessmode_sym)
	    return FALSE;
	SIGNAT(T_FILE,NULLSYM);
	break;
    case PRE_flush:
    case PRE_close:
	if (nargs != 1)
	    return FALSE;
	if (!IS_SIMPLE(ARG1->e_sig,T_FILE))
	    return FALSE;
	SIGNAT(T_VOID,NULLSYM);
	break;
    case PRE_remove:
	if (nargs != 1)
	    return FALSE;
	if (!is_string(ARG1->e_sig))
	    return FALSE;
	SIGNAT(T_BOOL,NULLSYM);
	break;
    case PRE_read:
    case PRE_write:
    case PRE_writes:
	{
	    Nodeptr arg;
	    Symptr s;

	    arg = op->e_r;
	    if (pre == PRE_read && !arg) {
		return FALSE;
	    }
	    if (arg && IS_SIMPLE(arg->e_sig,T_FILE))
		arg = arg->e_r;

	    for (; arg; arg = arg->e_r) {
		s = arg->e_sig;
		if (!((s->s_type == T_CHAR && s->s_size != SIZE_ARB)
	              || (IS_SCALAR(s)
			      && (s->s_type == T_INT
				  || s->s_type == T_STRING
				  || s->s_type == T_BOOL
				  || (s->s_type == T_PTR && pre!=PRE_read)))))
			return FALSE;
	    }
	    if (pre == PRE_read)
		SIGNAT(T_INT,NULLSYM);
	    else
		SIGNAT(T_VOID,NULLSYM);
	}
	break;
    case PRE_get:
    case PRE_put:
	{
	    Nodeptr last;	/* pointer to last argument. */

	    if (nargs != 1 && nargs != 2)
		return FALSE;
	    last = ARG1;
	    if (nargs == 2) {
		/* optional first argument. */
		if (!IS_SIMPLE(ARG1->e_sig,T_FILE))
		    return FALSE;
		last = last->e_r;
	    }
	    if (!is_string(last->e_sig) || last->e_sig->s_size == SIZE_ARB)
		return FALSE;
	    if (pre == PRE_get)
		SIGNAT(T_INT,NULLSYM);
	    else
		SIGNAT(T_VOID,NULLSYM);
	}
	break;
    case PRE_seek:
	if (nargs != 3)
	    return FALSE;
	if (!IS_SIMPLE(ARG1->e_sig,T_FILE))
	    return FALSE;
	else if (ARG2->e_sig->s_type != T_ENUM
		 || ARG2->e_sig->s_kind != K_LITERAL
		 || ARG2->e_sig->s_tdef != seektype_sym)
	    return FALSE;
	else if (!IS_SIMPLE(ARG3->e_sig,T_INT))
	    return FALSE;
	SIGNAT(T_INT,NULLSYM);
	break;
    case PRE_where:
	if (nargs != 1)
	    return FALSE;
	if (!IS_SIMPLE(ARG1->e_sig,T_FILE))
	    return FALSE;
	SIGNAT(T_INT,NULLSYM);
	break;
	/* argument processing operations. */
    case PRE_numargs:
    case PRE_age:
	if (nargs != 0)
	    return FALSE;
	SIGNAT(T_INT,NULLSYM);
	break;
    case PRE_getarg:
	if (nargs != 2)
	    return FALSE;
	if (!IS_SIMPLE(ARG1->e_sig,T_INT))
	    return FALSE;
	if (!((ARG2->e_sig->s_type == T_CHAR && ARG2->e_sig->s_size != SIZE_ARB)
	      || (IS_SCALAR(ARG2->e_sig)
		  && (ARG2->e_sig->s_type == T_INT
		      || ARG2->e_sig->s_type == T_STRING
		      || ARG2->e_sig->s_type == T_BOOL))))
	    return FALSE;
	SIGNAT(T_INT,NULLSYM);
	break;

    default:
	boom("bogus predef in pen");
	/*NOTREACHED*/
    }

    return TRUE;
}



	
/* generate code to call predefined functions */
void
gen_predef(func, arglist)
Symptr func;
Nodeptr arglist;
{
    Nodeptr fd;
    
    assert (func != NULLSYM);
    switch (func->s_predef)  {
	case PRE_myresource:
	    cprintf("rv");
	    return;
	case PRE_myvm:
	    cprintf("&sr_my_vm");
	    return;
	case PRE_mymachine:
	    cprintf("sr_my_machine");
	    return;
	case PRE_abs:
	    cprintf("abs(%e)",arglist->e_l);
	    return;
	case PRE_lb:
	case PRE_lb1:
	    gen_lbub(arglist->e_l, "%[");
	    return;
	case PRE_ub:
	case PRE_ub1:
	    gen_lbub(arglist->e_l, "%]");
	    return;
	case PRE_lb2:
	    gen_lbub(arglist->e_l, "%{");
	    return;
	case PRE_ub2:
	    gen_lbub(arglist->e_l, "%}");
	    return;
	case PRE_length:
	    if (arglist->e_l->e_sig->s_type == T_STRING) {
		cprintf("I(%a)",arglist->e_l);
	    } else {
		assert(arglist->e_l->e_sig->s_type == T_CHAR);
		cprintf("%z",arglist->e_l);
	    }
	    return;
	case PRE_maxlength:
	    if (arglist->e_l->e_sig->s_type == T_STRING) {
		cprintf("%L",arglist->e_l);
	    } else {
		/* same as length */
		assert(arglist->e_l->e_sig->s_type == T_CHAR);
		cprintf("%z",arglist->e_l);
	    }
	    return;
	case PRE_min:
	case PRE_max:
	    gen_minmax(func->s_predef, arglist);
	    return;
	case PRE_free:
	    if (arglist->e_l->e_op != TK_NULL)
		cprintf("sr_newfree(%e)",arglist->e_l);
	    /* nothing needed for free(null) */
	    return;
	case PRE_locate:
	    cprintf("sr_locate(%e,%S,",arglist->e_l,arglist->e_r->e_l);
	    if (arglist->e_r->e_r)
		cprintf("%S",arglist->e_r->e_r->e_l);
	    else
		cprintf("\"\",0");
	    cprintf(")");
	    return;
	case PRE_open:
	    assert(arglist && arglist->e_r);
	    cprintf("sr_open(%S,%e)", arglist->e_l, arglist->e_r->e_l);
	    return;
	case PRE_flush:
	    assert(arglist && !arglist->e_r);
	    cprintf("sr_flush(%a)",arglist->e_l);
	    return;
	case PRE_close:
	    assert(arglist && !arglist->e_r);
	    cprintf("sr_close(%a)",arglist->e_l);
	    return;
	case PRE_get:
	    assert(arglist);
	    fd = check_file(&arglist,F_STDIN);
	    if (arglist->e_l->e_sig->s_type == T_STRING)
		cprintf("sr_get(%e,%d+%a,%L,%a)", fd, INTSIZE,
		    arglist->e_l, arglist->e_l, arglist->e_l);
	    else
		cprintf("sr_get(%e,%S,(int*)0)",fd,arglist->e_l);
	    return;
	case PRE_remove:
	    assert(arglist && ! arglist->e_r);
	    cprintf("sr_remove(%S)",arglist->e_l);
	    return;
	case PRE_seek:
	    assert(arglist && arglist->e_r && arglist->e_r->e_r);
	    cprintf("sr_seek(%e,%e,%e)",
		arglist->e_l,
		arglist->e_r->e_l,
		arglist->e_r->e_r->e_l);
	    return;
	case PRE_where:
	    assert(arglist && !arglist->e_r);
	    cprintf("sr_where(%e)",arglist->e_l);
	    return;
	case PRE_numargs:
	    assert(!arglist);
	    cprintf("sr_numargs()");
	    return;
	case PRE_read:
	    gen_read(arglist);
	    return;
	case PRE_write:
	case PRE_writes:
	case PRE_put:
	    gen_write(arglist, func->s_predef);
	    return;
	case PRE_getarg:
	    gen_getarg(arglist);
	    return;
	case PRE_low:
	    switch (arglist->e_l->e_s->s_type)  {
		case T_INT:
		    cprintf(" -2147483648");	/* extra space avoids "=-" */
		    break;
		case T_ENUM:
		case T_CHAR:
		case T_BOOL:
		    cprintf("0");
		    break;
		default:
		    FATAL("low() only applies to ordered types");
	    }
	    return;
	case PRE_high: {
	    Symptr typ = arglist->e_l->e_s;
	    switch (typ->s_type) {
		case T_INT:
		    cprintf("2147483647");
		    break;
		case T_CHAR:
		    cprintf("255");
		    break;
		case T_ENUM:
		    while    (typ->s_next 
			   && typ->s_next->s_kind == K_LITERAL 
			   && typ->s_next->s_type == T_ENUM)
			typ=typ->s_next;
		    assert(typ);	   
		    cprintf("%d",typ->s_offset);
		    break;
		case T_BOOL:
		    cprintf("1");
		    break;
		default:
		    FATAL("high() only applies to ordered types");
	    }
	    return;
	    }
	case PRE_pred:
	    cprintf("(%e-1)",arglist->e_l);
	    return;
	case PRE_succ:
	    cprintf("(%e+1)",arglist->e_l);
	    return;
	case PRE_nap:
	    cprintf("sr_nap(%e)",arglist->e_l);
	    return;
	case PRE_age:
	    assert(!arglist);
	    cprintf("sr_age()");
	    return;
	default:
	    boom("unknown function");
    }
}


/* return file pointer and advance arglist if file given, else return default */
static Nodeptr 
check_file(argptr, dfault)
Nodeptr *argptr;
SRfile dfault;
{
    union e_lu f;
    register Nodeptr arg = *argptr;

    if (arg && IS_SIMPLE(arg->e_sig,T_FILE)) {
	*argptr = arg->e_r;
	return arg->e_l;
    } else {
	f.e_file = dfault;
	return make_node(TK_FILE_CONST,f,NULLNODE);
    }
}

/* generate code to push the upper or lower bound of an array */
static void
gen_lbub(e, fmt)
Nodeptr e;
char *fmt;
{
    if (e->e_op == TK_PERIOD)
	e = e->e_r;
    if (e->e_op == TK_IDENTIFIER)
	cprintf(fmt,e->e_s);
    else
	FATAL("lb/ub applies only to arrays");
}


static void
gen_minmax(minormax, arglist)
Predef minormax;
Nodeptr arglist;
{
    int nargs;
    Nodeptr a;

    nargs = 0;
    for (a = arglist;  a;  a = a->e_r)
	nargs++;
    cprintf("%s(%d", minormax == PRE_min ? "sr_min" : "sr_max", nargs);
    for (a = arglist;  a;  a = a->e_r)
	cprintf(",%e",a->e_l);
    cprintf(")");
}



Bool
is_predef_addressable(f)
Symptr f;
{
    assert (f != NULLSYM && f->s_kind == K_PREDEF);
    switch (f->s_predef) {
	case PRE_myresource:
	case PRE_myvm:
	    return TRUE;
	case PRE_notapredef:
	    boom("bad s_predef in is_predef_addressable");
	default:
	    return FALSE;
    }
}


static void
gen_write(args, func)
Nodeptr args;
Predef func;
{
    Nodeptr fd, a;
    char format[200], *f;
    
    fd = check_file(&args,F_STDOUT);
    /* generate output format depending on arguments */
    f = format;
    for (a = args;  a;  a = a->e_r)  {
	switch (a->e_l->e_sig->s_type)  {
	    case T_INT:     *f++ = 'd';  break;
	    case T_PTR:     *f++ = 'p';  break;
	    case T_BOOL:    *f++ = 'b';  break;
	    case T_STRING:  *f++ = 's';  break;
	    case T_CHAR:    *f++ = 's';  break;
	    default: boom("illegal type for write");
	}
	if (func == PRE_write)
	    *f++ = ' ';		/* add space separator for write() only */
    }
    if (func == PRE_write)  {
	if (f > format)
	    f--;		/* remove trailing space */
	*f++ = '\\';
	*f++ = 'n';		/* insert newline */
    }
    *f++ = '\0';
    cprintf("sr_write(%e,\"%s\"", fd, format);
    /* generate arguments */
    for (a = args;  a;  a = a->e_r)
	switch (a->e_l->e_sig->s_type)  {
	    case T_INT:
	    case T_BOOL:
	    case T_PTR:
		cprintf(",%e",a->e_l);
		break;
	    case T_STRING:
	    case T_CHAR:
		cprintf(",%S",a->e_l);
		break;
	}
    cprintf(")");
}


static void
gen_read(args)
Nodeptr args;
{
    Nodeptr fd, a;
    char format[100], *f;
    
    fd = check_file(&args,F_STDIN);
    f = format;
    for (a = args;  a;  a = a->e_r)  {
	if (!is_lvalue(a->e_l) || is_a_const(a->e_l))
	    FATAL("arguments of read must be lvalues");
	switch (a->e_l->e_sig->s_type)  {
	    case T_INT:     *f++ = 'd';  break;
	    case T_BOOL:    *f++ = 'b';  break;
	    case T_STRING:  *f++ = 's';  break;
	    case T_CHAR:    *f++ = 'c';  break;
	    default: boom("illegal type for read");
	}
    }
    *f++ = '\0';
    cprintf("sr_read(%e,\"%s\"", fd, format);
    for (a = args;  a;  a = a->e_r)
	switch (a->e_l->e_sig->s_type)  {
	    case T_INT:
	    case T_BOOL:
		cprintf(",%a",a->e_l);
		break;
	    case T_STRING:
	    case T_CHAR:
		cprintf(",%a,%L",a->e_l,a->e_l);
		break;
	}
    cprintf(")");
}


static void
gen_getarg(args)
Nodeptr args;
{
    register Nodeptr arg2 = args->e_r->e_l;
    if (! is_lvalue(arg2)) {
	FATAL("second argument of getarg() must be an lvalue");
	return;
    }
    switch (arg2->e_sig->s_type) {
	case T_INT:
	    cprintf("sr_arg_int(%e,%a)",args->e_l,arg2);
	    return;
	case T_BOOL:
	    cprintf("sr_arg_bool(%e,%a)",args->e_l,arg2);
	    return;
	case T_CHAR:
	    cprintf("sr_arg_chars(%e,%S)",args->e_l,arg2);
	    return;
	case T_STRING:
	    cprintf("sr_arg_string(%e,%a,%L)",args->e_l,arg2,arg2);
	    return;
	default:
	    FATAL("bad getarg type");
    }
}
