/*  parse.c - parse one source file  */


#include <stdio.h>
#include "../util.h"
#include "sr.h"
#include "funcs.h"
#include "globals.h"
#include "tokmacs.h"

static int prev_number;		/* line number of the previous token */
static Bool put_back;		/* has a token been put back? */

/*  parse() - parse current file (i.e., yyin) and set error counts  */

void
parse()
{
    lineno = 1;
    put_back = FALSE;
    while (get_eof_token()) {
	put_back = TRUE;
	srparse();
	/* skip over optional semicolon as terminator. */
	if (get_eof_token() != TK_SEMICOLON)
	    put_back = TRUE;
	if (dbflags['T'])
	    printf("\nend of parser\n");
    }
}



/* get_token()
 * set tok to and return the next token. EOF is an error.
 */
Token
get_token()
{
    get_eof_token();
    if (tok)
	return (tok);
    FATAL("unexpected end of file");
    exit(1);
    /*NOTREACHED*/
}



/* get_eof_token()
 * set tok to and return the next token. EOF is ok.
 */
Token
get_eof_token()
{
    if (put_back) {
	put_back = FALSE;
	line_number = lineno;
    } else {
	prev_number = line_number;
	tok = (Token) yylex();
	line_number = lineno;
	if (wi_fd) write_to_inter();
    }
    return (tok);
}



/* unget the last token */
void
putback()
{
    put_back = TRUE;
    line_number = prev_number;
}


/* maybe(t) -- skip next token if it's t, return TRUE if so */
Bool
maybe(t)
Token t;
{
    if (get_token() == t)
	return TRUE;
    else {
	put_back = TRUE;
	return FALSE;
    }
}



/* mustbe(c,str)
 * checks that next token is c;
 * otherwise complains, using str to give specific error message,
 * and simulate effect that token would have had.
 * only complicated token is TK_IDENTIFIER, for which
 * we make up a name.
 */
void
mustbe(c,str)
Token  c;
char *str;
{
	static int id_count = 0;	/* count of invented identifiers */
	static char buffer[100];	/* buffer for name or message */

	if (get_token() != c) {
		if (c == TK_ASSIGN && tok == TK_EQ) {
			WARN("this isn't FORTRAN; use :=");	
		}
		else if (c == TK_IDENTIFIER) {	    
			/* handle this case separately to give a fatal message.
			 * must be fatal to avoid creating resource/globals
			 * with these made up names.  */
			if (IS_RESERVED(tok))
			    errmsg(E_FATAL,
				"expected identifier, got reserved word '%s'",
				yytext);
			else
			    errmsg(E_FATAL,
				"expected identifier, got '%s'",yytext);
			/* don't put back a reserved word used in a decl */
			if (!(in_variab_name && IS_RESERVED(tok)))
				put_back = TRUE;

			/* make an illegal name to avoid conflicts */
			sprintf(buffer , "_FAKE%08d", id_count++);
			tk_str = buffer;
		}
		else {
			sprintf(buffer,"missing syntactic unit: %s",str);
			WARN(buffer);
			put_back = TRUE;
		}
		tok = c;
	}
}

/* return the current state of the scanner */
Scanptr
get_scan_state()
{
    Scanptr state;
    
    state = (Scanptr)alloc(sizeof(Scan));
    state->put_back = put_back;
    state->yysptr = yysptr;
    state->yysbuf = yysbuf[0];
    state->yyin = yyin;
    state->lineno = lineno;
    state->line_number = line_number;
    state->prev_number = prev_number;
    state->tok = tok;
    return state;
}


/* set the state of the scanner to "state" */
void
set_scan_state(state)
Scanptr state;
{
    put_back = state->put_back;
    yysptr = state->yysptr;
    yysbuf[0] = state->yysbuf;
    yyin = state->yyin;
    lineno = state->lineno;
    line_number = state->line_number;
    prev_number = state->prev_number;
    tok = state->tok;
}


/* returns TRUE if token is a member of the list follow set.
 * note: has side effect of setting put_back to TRUE.
 */
Bool
is_follow(list)
Token *list;
{
	Token c, *p;

	c = (*list == TK_EOF) ? get_eof_token() : get_token();
	for (p = list; *p; p++)
		if (c == *p) {
			put_back = TRUE;
			return (TRUE);
		}
	put_back = TRUE;
	return (FALSE);
}



/* find_follow(list)
* skips tokens until a member of the list follow set is found.
* if the first element is TK_EOF, then eof is permitted.
* assumes that last element of a follow set is TK_NOTATOKEN.
*/
void
find_follow(list)
Token list[];
{
	Token c, *p;
	Token (*get)();

	get = (list[0] == TK_EOF) ? get_eof_token : get_token;

	/* make sure we don't get stuck on a token.
	 * this could be improved if we were smart.
	 */
		put_back = FALSE;

	while (c = (*get)()) {
		for (p = list; *p; p++) {
			if (c == *p) {
				put_back = TRUE;
				return;
			}
		}
	}
	/* got eof. it's ok if we expected it. */
		if (list[0] != TK_EOF) {
			boom("could not find in follow set\n");
			/*NOTREACHED*/
		}
}
