/*  main.c -- sr compiler main program, file handling, error processing  */

#include <stdio.h>
#include <ctype.h>
#include "sr.h"
#include "funcs.h"
#include "tokmacs.h"
#include "../config.h"
#include "../util.h"
#include "../paths.h"


/* SR version number */
static char version[] = VERSION;


/* define globals here (in the main program) */
#define GLOBAL
#define INIT(v) = v
#include "globals.h"



/* initialize global table of token attributes */
#define tkmac(name,rsv,left,right,prec) {rsv, left, right, prec},
struct tkatt token_atts [] = {
#include "tokens.h"
};



/* command line options. */

static Bool option_c;		/* compile only */
static Bool option_e;		/* experimental srl & runtime system */
static Bool option_q;		/* print file names as compiled */
static Bool option_g;		/* compile with -g for dbx, don't rm *.[ch] */
static Bool option_v;		/* verbose mode */
static Bool option_C;		/* don't call cc(1) */
static Bool option_O;		/* call cc(1) with -O */
static Bool option_S;		/* call cc(1) with -S */

static char *Iname;		/* Interfaces directory */
static char *oname;		/* output file name */



/* code generation file info */

static char cname[FILE_NAME_SIZE];	/* .c output file name */
static char hname[FILE_NAME_SIZE];	/* .h output file name */

static char *rlist[MAX_RES_DEF+1];	/* resource list */
static char **rlp = rlist;		/* resource list pointer */




/* main program */

main(argc,argv)
int  argc;
char *argv[];
{
    int c;
    char *p;
    int total_fatals = 0;	/* total number of fatal errors */
    extern int optind;
    extern char *optarg;

    xprefix = argv[0];		/* set exit message prefix */
#define USAGE "usage:  sr [-cegqvwCOS] [-I dir] [-o file] [-D flags] file.sr ..."
    while ((c = getopt(argc,argv,"cegqvwCOSI:o:D:")) != EOF)
	switch (c)  {
	    case 'c':	option_c = TRUE;  break;
	    case 'e':	option_e = TRUE;  break;
	    case 'g':	option_g = TRUE;  break;
	    case 'q':	option_q = TRUE;  break;
	    case 'v':	option_v = TRUE;  trcexec = 1;	break;
	    case 'w':	do_time_check = 0;break;
	    case 'C':	option_C = TRUE;  break;
	    case 'O':	option_O = TRUE;  break;
	    case 'S':	option_S = TRUE;  break;

	    case 'I':
		if (Iname)
		    mexit("duplicate -I option");
		Iname = optarg;
		break;

	    case 'o':
		if (oname)
		    mexit("duplicate -o option");
		oname = optarg;
		break;

	    case 'D':
		for (p = optarg; c = *p; p++)
		    if (c == 'A')
			memset(dbflags,1,sizeof(dbflags));
		    else
			dbflags[toascii(c)] ^= 1;
		break;

	    default:
	      mexit(USAGE);
	}

    if (oname && (option_c || option_C || option_S))	/* msg if -o useless */
	fprintf(stderr,"%s: -o ignored\n",argv[0]);

    if (option_S && option_c)		/* can't have both */
	fprintf(stderr,"%s: -c ignored\n",argv[0]);

    if (option_g && option_O)		/* can't have both */
	fprintf(stderr,"%s: -O ignored\n",argv[0]);

    if (optind >= argc)			/* if no files, quit now, quietly */
	exit(0);

    if (optind+1 == argc)		/* if just one, set -q option */
	option_q = TRUE;

    if (optind < argc) {		/* if any files at all, initialize */
	presym();			/* init predefined symbols */
	init_inter_dir(Iname);		/* init Interfaces directory */
    }		

    while (optind < argc)  {		/* process each file name */
	source_file = argv[optind++];
	if (!strtail(source_file,SOURCE_SUF))  {
	    fprintf(stderr,"%s:  unrecognized file extension:  %s\n",
		argv[0],source_file);
	    exit(1);
	}
	fatal_err_cnt = warn_err_cnt = 0;
	yyin = mustopen(source_file,"r");
	if (!option_q)
	    fprintf(stderr,"%s:\n",source_file);
	parse();
	fclose(yyin);
	total_fatals += fatal_err_cnt;
    }

    /* check that the predefs, but nothing else, are left */
    /* assertion failure means somebody did too many or too few pop_block()s */
    assert (st_cb && !st_cb->s_prev);

    if (dbflags['T'])
	printf("\ndone\n");
    if (total_fatals > 0 || option_c || option_S || option_C)
	exit(total_fatals);
    else {
	if (!option_q)
	    fprintf(stderr,"linking:\n");
	linker();	 /* exec srl to produce an executable file */
    }
}








/*  chopen(res) - open .c and .h files for output */

void
chopen(res)
char *res;
{
    char **p;

    if (rlp - rlist >= MAX_RES_DEF)
	boom("too many resources");

    for (p = rlist; p < rlp; p++)
	if (strcmp(res,*p) == 0)
	    errmsg(E_FATAL,"duplicate resource name: %s",res);
    *rlp++ = res;
    sprintf(cname,"%s.c",res);
    sprintf(hname,"%s.h",res);
    cfile = mustopen(cname,"w");
    hfile = mustopen(hname,"w");
    fprintf(cfile,"/* produced by %s */\n\n",version);
    fprintf(hfile,"/* produced by %s */\n\n",version);
    fprintf(cfile,"#include <stdio.h>\n");
    if (option_e || sizeof(SRLIB)<=1)  {
	fprintf(cfile,"#include \"%s/srsys.h\"\n",SRDIR);
	fprintf(cfile,"#include \"%s/sr/srgen.h\"\n",SRDIR);
    } else {
	fprintf(cfile,"#include \"%s/srsys.h\"\n",SRLIB);
	fprintf(cfile,"#include \"%s/srgen.h\"\n",SRLIB);
    }
    fprintf(cfile,"#include \"%s\"\n",hname);
}



/*  backend() - call C compiler to compile and assemble the intermediate code */

void
backend()
{
    int status;
    char *argv[20];
    char **argp = argv;

    if (!cfile)
	return;
    fclose(cfile);
    fclose(hfile);
    cfile = hfile = 0;
    if (option_C & !(option_S || option_c))
	return;
    if (fatal_err_cnt == 0)  {
	*argp++ = "cc";
	*argp++ = "-w";
	if (option_g)
	    *argp++ = "-g";
	else if (option_O)
	    *argp++ = "-O";
	*argp++ = option_S ? "-S" : "-c";

#ifdef hp9000s300
	*argp++ = "+Ne700";	/* HP Bobcat needs extra tree space */
#endif

	*argp++ = cname;
	*argp++ = 0;
	status = spawn(CCPATH, argv, NULLSTR, NULLSTR, NULLSTR);
	if (status != 0)
	    boom("bad .c file");
    }
    if (!(option_C || option_S || option_g))  {
	if (unlink(cname) != 0)
	    perror(cname);
	if (unlink(hname) != 0)
	    perror(hname);
    }
}



/*  linker() - invoke srl to link together that which we have compiled  */

void
linker()
{
    char path[FILE_NAME_SIZE];
    char *argv[MAX_RES_DEF+10];	/* argument vector */
    char **argp = argv;

    /* build argv with name and options */
    *argp++ = "srl";
    if (option_e)
	*argp++ = "-e";
    if (option_g)
	*argp++ = "-g";
    if (option_v)
	*argp++ = "-v";
    if (Iname)  {
	*argp++ = "-I";
	*argp++ = Iname;
    }
    if (oname)  {
	*argp++ = "-o";
	*argp++ = oname;
    }

    /* add resource list */
    rlp = rlist;
    while (*argp++ = *rlp++)	/* copy and terminate list */
	;

    /* exec srl */
    if (option_e || sizeof(SRCMD)<=1)
	sprintf(path,"%s/srl/srl",SRDIR);
    else
	sprintf(path,"%s/srl",SRCMD);
    doexec(path,argv);
}




/*  kboom - malfunction abort - called by "boom" macro  */

void
kboom(s,f,l)
char *s, *f;
int l;
{
    fprintf(stderr, "\"%s\", line %d: compiler malfunction: %s\n",
	source_file, line_number, s);
    fprintf(stderr, "\t(called from %s, line %d.)\n", f, l);
    exit(3);
}



/*  assfail - abort due to assertion failure - called by "assert" macro  */

void
assfail(f,l)
char *f;
int l;
{
    kboom("assertion failure",f,l);
}



/*  errmsg(code,text,param) - print error message.
 *
 *  code is E_WARN or E_FATAL, plus a small int to prepend a message from list.
 *  text is basic message text.
 *  param is string to insert at %s in text, or NULL if not needed.
 */

static char *errlist[] = {
    /*  0 */	"",
    /*  1 */	"identifier not defined: ",
    /*  2 */	"invalid type specified: ",
    /*  3 */	"signature mismatch: ",
    /*  4 */	"not yet implemented: ",
    /*  5 */	0,
    /*  6 */	"missing/invalid syntactic unit: ",
    /*  7 */	"duplicate symbol in this block: ",
    };

void
errmsg(code,text,param)
int code;
char *text, *param;
{
    int wf = code & (E_FATAL | E_WARN);
    int n = code & ~(E_FATAL | E_WARN);

    fprintf (stderr,"\"%s\", line %d: ", source_file, line_number);
    if (wf == E_FATAL)  {
	fprintf(stderr, "fatal: ");
	if (fatal_err_cnt++ == 0 && cfile != 0)
	    fprintf(cfile,"\n#error\t\t==== FATAL ERROR IN COMPILATION ====\n");
    } else {
	assert(wf == E_WARN);
	fprintf(stderr, "warning: ");
	warn_err_cnt++;
    }
    if (n != 0)
	if (n < 0 || n >= (sizeof(errlist) / sizeof(errlist[0])) || !errlist[n])
	    fprintf (stderr, "unknown error %d: ", n);
	else
	    fprintf (stderr, errlist[n]);
    if (text != 0 && *text != '\0')
	fprintf (stderr, text, param);
    fprintf (stderr, "\n");
    if (fatal_err_cnt >= MAX_FATALS)
	mexit("too many errors.  aborting.");
}
