/*  invoke.c -- call, send, and co statements  */

/*  note: error recovery here is minimal. */


#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"
#include "../util.h"


/* structure used to hold labels for each arm of a co statement. */
typedef struct arm_st {
    int arm_ppc_lab;	/* label of PPC code; or NOLAB if none. */
    Instlist arm_ppc_inst;	/* intermediate code for PPC. */
    struct arm_st *arm_next;	/* pointer to next arm_st. */
    } Arm, *Armptr;

#define NULLARM ((Armptr)0)


/* pointer to first on free list */
static Armptr arm_free = NULLARM;


static Armptr arm_alloc();
static Nodeptr co_id(), co_invoke();
static void co_copy_back(), arm_release();
static int start_ppc();



/* V(semaphore) statement
 * semaphore is an operation_indication
 * (i.e., opname or opname[subscripts], not a capability or an invocation.)
 */
void
sema_v_stmt()
{
	Nodeptr n;
	Symptr op_sym;

	mustbe(TK_LEFTPAREN,"( in V statement");

	mustbe(TK_IDENTIFIER,"op_identifier in V statement");

	if ((op_sym = st_lookup(tk_str)) == NULLSYM) {
		ERROR(E_FATAL+1,tk_str);
		return;
	}
	if (op_sym->s_kind != K_OP
	    || (op_sym->s_type != T_FUNC && op_sym->s_type != T_VOID)) {
		FATAL("bad op identifier");
		return;
	}
	/* make an id node for the operation. */
		n = idnode(op_sym);

	/* subscripts. */
		if (maybe(TK_LEFTBKET))
         		n = bnode(TK_INDEX,n,indices());

	/* simulate side effect of invocation. */
		promote_invoked(op_sym, R_SEND);

	mustbe(TK_RIGHTPAREN,") in V statement");
	n = bnode(TK_INVOKE,n,NULLNODE);
	if (n->e_sig->s_type != T_VOID)
		WARN("return value ignored");
	n = bnode(TK_SEND,n,NULLNODE);
	call_send(n);
}



/* for a call or send statement (but not an implicit call statement)
 * returns pointer to the call/send/co_call node so that it can be used by co.
 */
Nodeptr
call_send_stmt(mytok)
Token mytok;
{
	Nodeptr n;
	if ((n = denotation(mytok)) == NULLNODE){
		WARN("bad call/send stmt");
		return NULLNODE;
	}
	if (n->e_op != mytok) {
		WARN("bad denoted op in call/send");
		return NULLNODE;
	}
	/* would be nice to do this check in makesig,
	 * but such is difficult since makesig can't
	 * distinguish between implicit and explicit call.
	 */
	if (n->e_l->e_sig->s_type != T_VOID)
		WARN("return value ignored");
	call_send(n);
	return (n);
}



/* used by call_send_stmt and for an implicit call statement.
 * emits intermediate code for the invocation.
 */
void
call_send(n)
Nodeptr n;
{
	if (dbflags['S'])
		printf("call_send\n");
	assert    (n->e_op == TK_CALL || n->e_op == TK_SEND 
		|| n->e_op == TK_CO_CALL || n->e_op == TK_CO_SEND);
	emit(I_EXPR,n,NOLAB);
}



/* At runtime, the code needs to know when it has received all the replies
 * and can do the co_end; i.e., when it no longer needs to co_wait.
 * The RTS counts the number of co_calls and returns NULL from co_wait
 * we have done enough co_waits.
 * PPC stands for post-processing clause (or code).
 * Quantifiers cause the following problem:
 * how to associate the bound (index) variables with a particular invocation?
 * After much thought, we decided to put at invocation the index variables
 * in the parameter block and copy them back out when we get back the block.
 * We decided to handle sends in the same fashion, so we use a CO_SEND so that
 * the RTS gives us back the block.
 * This is optimized: for a send, we need to copyback quantifier variables
 * only when there is a quantifier and user PPC.
 * There is a PPC for all call invocations so that results can be copied
 * back.
 * For send invocations, there is a PPC iff there was one in the user's code.
 * The first part of PPC copies back parameters, if co_call, and, if present,
 * copies back quantifier index variables;
 * the rest is the user's PPC code, if any.
 * All the PPC's are put after the invocations
 * to avoid the branches around each PPC;
 * we do this by twiddling with the intermediate code.
 */
void
co_stmt()
{
	int
	   old_next, old_exit; /* old next_label and exit_label. */
	/* note: next_label is the label at start of co_wait loop. */

	int arm_cnt; /* number of arm of co that we are parsing; 0 based. */
	static Token follow[] = {TK_PARALLEL,TK_OC,TK_NOTATOKEN};
	/* pointer to intermediate code to be patched with invocation.
	 * ic_before is pointer of code to patch.
	 */
		Instptr ic_quantifier_guts, ic_before;
	/* block for quantifier, if present; else, null. */
		Symptr quant_block;
	int ppc; /* label at beginning of ppc. */
	Instptr ic_ppc; /* pointer to intermediate code for PPC. */
	Nodeptr inv; /* points at the node tree for an invocation. */
	Armptr arm_ppc, ppc_top, ppc_bot;

	old_next = next_label; old_exit = exit_label;
	next_label = NEWLAB; exit_label = NEWLAB;

	/* generate the co_start. */
		emit(I_CO_START,NULLNODE,NOLAB);

	arm_cnt = 0;
	ppc_top = ppc_bot = NULLARM;
	do {
		if (maybe(TK_LEFTPAREN)) {
			quant_block = block_begin(T_CMD);
			ic_quantifier_guts = quantifier(FALSE);
			mustbe(TK_RIGHTPAREN,") in co's quantifier");
		}
		else {
			quant_block = NULLSYM;
			ic_quantifier_guts = NULLINST;
		}

		ic_before = ic_mark();
		emit(I_CO_ARM,NULLNODE,arm_cnt);
		inv = co_invoke(quant_block);

		/* if there was a (valid) quantifier,
		 * patch the invocation into the inner most loop
		 * of the quantifier.
		 */
			if (ic_quantifier_guts != NULLINST){
				ic_move(ic_before,ic_quantifier_guts);
			}

		/* remember where the intermediate code for the PPC begins. */
			ic_ppc = ic_mark();

		get_token();
		if (tok == TK_ARROW) {
			ppc = start_ppc(inv,TRUE);
			new_command_block(follow);
			assert (tok == TK_PARALLEL || tok == TK_OC);
		}
		else {
			ppc = start_ppc(inv,FALSE);
			if (tok != TK_PARALLEL && tok != TK_OC) {
				WARN("missing // (or oc)");
			}
		}

		/* save the ppc label for the jump table. */
			arm_ppc = arm_alloc();
			arm_ppc->arm_ppc_lab = ppc;
			if (ppc_top == NULLARM)
				ppc_top = arm_ppc;
			else {
				ppc_bot->arm_next = arm_ppc;
			}
			ppc_bot = arm_ppc;

		if (ppc != NOLAB) {
			/* emit the "return" at the end of the ppc. */
				emit(I_CO_PPC_END,NULLNODE,next_label);
		}

		/* stow the PPC intermediate code, if any.
		 */
			ic_stow(ic_ppc,&arm_ppc->arm_ppc_inst);

		/* done with the quantifier block if there is one. */
			if (quant_block != NULLSYM) {
				block_end();
			}
		arm_cnt++;

	} while (tok != TK_OC);

	/* branch around all the PPC code to the co_wait loop. */
		emit(I_BRANCH,NULLNODE,next_label);

	/* patch the PPC's into the intermediate code. */
		for (arm_ppc = ppc_top;
		    arm_ppc != NULLARM;
		    arm_ppc = arm_ppc->arm_next){
		    ic_append(&arm_ppc->arm_ppc_inst);
		}

	/* give the generated code the info so it can build a jump table
	 * and generate the co_wait loop.
	 * this is a separate loop from the above because the
	 * jump table must not be nested;
	 * e.g., if a PPC contains a co, the code generator gets confused.
	 */
		emit(I_CO_WAIT,NULLNODE,next_label);
		emit(I_JT,NULLNODE,arm_cnt);
		for (arm_ppc = ppc_top;
		     arm_ppc != NULLARM;
		     arm_ppc = arm_ppc->arm_next) {
			emit(I_JT_LAB,NULLNODE,arm_ppc->arm_ppc_lab);
		}
		emit(I_JT_END,NULLNODE,NOLAB);

	arm_release(ppc_top,ppc_bot);

	/* finally, generate the co_end. */
		emit(I_CO_END,NULLNODE,exit_label);

	next_label = old_next; exit_label = old_exit;
}

/* co_invoke(quant_block)
 * concurrent invocation can be one of:
 *	call
 *	send
 *	implicit call
 *	simple assignment.
 *
 * builds a TK_CO_CALL or TK_CO_SEND node.
 * the right side of this node points to the list of quantifier variables,
 * which is given by quant_block.
 * returns pointer to node that was built.
 */
static Nodeptr
co_invoke(quant_block)
Symptr quant_block;
{
	Nodeptr ret, qv, node, last, n;
	Symptr s;

	get_token();
	switch (tok) {
		case TK_CALL:
			ret = call_send_stmt(TK_CO_CALL);
			break;
		case TK_SEND:
			ret = call_send_stmt(TK_CO_SEND);
			break;
		case TK_IDENTIFIER:
			ret = co_id();
			break;
		default:
			FATAL("bad start of co invocation");
			ret = NULLNODE;
			break;
	}

	/* build the list of quantifier variables, if any, into qv. */
		qv = last = NULLNODE;
		if (quant_block != NULLSYM) {

			for (s = quant_block->s_next;
			     s != NULLSYM;
			     s = s->s_next) {
				assert (s->s_kind == K_VAR);
				n = bnode(TK_LIST,idnode(s),NULLNODE);
				if (qv == NULLNODE)
					qv = n;
				else
					last->e_r = n;
				last = n;
			}
		}

	/* throw qv onto the invocation node. */
	if (ret != NULLNODE) {
		switch (ret->e_op) {
			case TK_CO_CALL:
			case TK_CO_SEND:
				node = ret;
				break;
			case TK_ASSIGN:
				assert(ret->e_r->e_op==TK_CO_CALL);
				node = ret->e_r;
				break;
			default:
				boom("bad ret node in qv throw");
				/*NOTREACHED*/
		}
		node->e_r = qv;
		node = node->e_l->e_l;
		if (node->e_op==TK_IDENTIFIER && node->e_s->s_kind==K_PREDEF)
		    FATAL("call to predefined function inside co statement");
	}

	return (ret);
}

/* concurrent invocation that starts with an id.
 * figures out if it is an implicit call invocation
 * or the start of an assignment statement by looking at denotation.
 * note: this code is similar to code in statement that
 * determines if statement is assignment, create, inc/dec, or implicit call.
 * returns a pointer to the invocation or assignment node.
 */
static Nodeptr
co_id()
{
	Nodeptr n, rhs;
	Symptr sig;

	putback();
	if ((n = denotation(TK_CO_CALL)) == NULLNODE) {
		FATAL("bad denotation at start of concurrent invocation");
		return (NULLNODE);
	}
	sig = n->e_sig;
	if (dbflags['S'])
		printf(
		"in co_id, sig->s_type: %s, sig->s_tdef: %x, ->s_type: %s\n",
		typetos(sig->s_type),sig->s_tdef,typetos(sig->s_tdef->s_type));

	if (n->e_op == TK_CO_CALL){
		if (n->e_l->e_sig->s_type != T_VOID)
			WARN("return value ignored");
		call_send(n);
		return (n);
	}

	/* we'll assume it's a simple assignment statement.
	 * signature will complain later if it's not.
	 * make sure that right hand side is simple, too.
	 */

		mustbe(TK_ASSIGN,":= in assignment in co");
		if ((rhs = denotation(TK_CO_CALL)) == NULLNODE) {
			FATAL("bad denotation in assignment in co");
			return (NULLNODE);
		}
		if (rhs->e_op != TK_CO_CALL) {
			FATAL(
			"right side of assignment in co must be an invocation");
			return (NULLNODE);
		}

		n = bnode(TK_ASSIGN,n,rhs);
		emit(I_EXPR,n,NOLAB);
		return (n);
}
						
/* start_ppc(inv,user_ppc_present)
 * called when we have determined whether user ppc is present;
 * inv is a pointer to the concurrent invocation.
 * we generate code here to do the copyback
 * and optimize TK_CO_SEND to TK_SEND if possible.
 * returns the label of the start of the ppc, or NOLAB if no ppc.
 * note:  each TK_CO_x needs a TK_CO_x_COPY_BACK
 * at least to deallocate the invocation block.
 * besides, the code generator would get confused without.
 */
static int
start_ppc(inv,user_ppc_present)
Nodeptr inv;
Bool user_ppc_present;
{
	int ppc;	/* label returned (most of the time) */

	if (inv == NULLNODE)
		return (NOLAB);

	switch (inv->e_op) {
		case TK_ASSIGN:
		case TK_CO_CALL:
			ppc = NEWLAB;
			emit(I_LABEL,NULLNODE,ppc);
			co_copy_back(inv);
			break;
		case TK_CO_SEND:
			if (!user_ppc_present) {
				inv->e_op = TK_SEND;
				ppc = NOLAB;
			}
			else {
				ppc = NEWLAB;
				emit(I_LABEL,NULLNODE,ppc);
				co_copy_back(inv);
			}
			break;
		default:
			boom("bad inv in start_ppc");
			/*NOTREACHED*/
	}
	return (ppc);


}

/* emits intermediate code to do the copy back for a co invocation.
 * first it makes a copy of the invocation tree, and changes
 * TK_CO_CALL to TK_CO_CALL_COPY_BACK or TK_CO_SEND to TK_CO_SEND_COPY_BACK.
 */
static void
co_copy_back(inv)
Nodeptr inv;
{
	Nodeptr new, change;

	assert (   inv->e_op == TK_CO_CALL
		|| inv->e_op == TK_ASSIGN
		|| inv->e_op == TK_CO_SEND);

	new = copy_nodes(inv);

	if (new->e_op == TK_CO_SEND)
		new->e_op = TK_CO_SEND_COPY_BACK;
	else if (new->e_op == TK_CO_CALL)
		new->e_op = TK_CO_CALL_COPY_BACK;
	else {
		change = new->e_r;
		assert (change->e_op == TK_CO_CALL);
		change->e_op = TK_CO_CALL_COPY_BACK;
	}

	emit(I_EXPR,new,NOLAB);
}

/* allocate and initialize a co arm structure.
 */
static Armptr
arm_alloc()
{
    register Armptr t;
    if (arm_free) {
	t = arm_free;
	arm_free = arm_free->arm_next;
    } else
	t = (Armptr)alloc(sizeof(Arm));

    t->arm_ppc_inst.il_head.i_next = NULLINST;
    t->arm_next = NULLARM;
    return (t);
}

/* free a list of arm structures and link them into the free list.
 */
static void
arm_release(top,bot)
Armptr top, bot;
{
	if (top != NULLARM) {
		assert (bot != NULLARM);
		bot->arm_next = arm_free;
		arm_free = top;
	}
}
