/*
 * Module that deals with interface between resources.
 * Handles imports, extends, separately compiled spec and bodies.
 *
 * .i files begin with a first line of "# path name."
 */

#include <stdio.h>
#include <sys/errno.h>
#include <sys/param.h>
#include <sys/timeb.h>
#include <sys/stat.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"
#include "../config.h"
#include "../util.h"

static Symptr read_inter();
static void extend_append(), get_sourcepath(), end_inter_cleanup();
static void impcmpinit(), impcmpfini(), impcmppush(), impcmppop();
static Bool impcmpcheck();

extern int errno;

static char interpath[MAX_PATH];	/* holds path for Interfaces */
static char interfile[MAX_PATH];	/* holds path for .i file */

/* indicates whether .i file existed for (top level) component
 * that is currently being compiled.
 * controls .i file renaming/removing.
 */
static Bool inter_old;


/* time that the .i file for current component was written.
 * for top level spec, global, or body that time is now.
 * when doing imports, the time is set to that of the component
 * being imported.
 * this lets us detect components that must be recompiled
 * and circularities (which are not allowed).
 */
static time_t inter_time;


/* the time is now.
 * note this is only used at top level;
 * so, if an error occurs later, don't need to reset inter_time.
 */
#define NOW(x) x = 0x7fffffff
/* OOPS_DOES_NOT_WORK_WITH_NFS
 * #define NOW(x) { struct timeb btp; ftime(&btp); x = btp.time; }
 */


/* Make sure the Interfaces directory exists; create it if necessary.
 * Avoid race conditions on making the directory (during concurrent
 * compililation) by checking to see if it exists (again) if there is
 * an error in creating it.
 */
void
init_inter_dir(directory)
char *directory;
{
    struct stat sbuf;

    if (!directory)
	directory = ".";
    strcpy(interpath,directory);
    if (stat(interpath,&sbuf) != 0)
	pexit(interpath);
    if ((sbuf.st_mode & S_IFMT) != S_IFDIR)  {
	strcat(interpath,": not a directory");
	mexit(interpath);
    }
    strcat(interpath,"/");
    strcat(interpath,INTER_DIR);
    if (stat(interpath,&sbuf) == 0 && (sbuf.st_mode & S_IFMT) == S_IFDIR)
	return;
    if (mkdir(interpath,0777) != 0)
	if (stat(interpath,&sbuf) != 0 || (sbuf.st_mode & S_IFMT) != S_IFDIR)
	    pexit(interpath);
}

/* called at start of spec or global to set things up to get .i file written.
 * (only called at top level; e.g., not for imports)
 * note that can't open .i file until the name of spec or global
 * is seen, so this routine also writes the first spec/global keyword
 * and then its name (as opposed to doing such through get_token).
 */
void
start_inter(mytok)
	Token mytok;
{
	char sourcepath[MAX_PATH];	/* holds path of source file. */
	char interold[MAX_PATH];
	struct stat status;

	get_sourcepath(sourcepath);

	/* open new .i file
	 * and write the source file name to it as a comment.
	 * we save the original .i file (if there is one) in .old
	 * and restore it in case of error.
	 * we write the new .i file to .new.
	 * thus, the .i file does not exist during compiling this
	 * component, which makes circularity easier to prevent.
	 */
	sprintf(interfile,"%s/%s.new",interpath, comp_sym->s_name);
	if ((wi_fd = fopen(interfile,"w")) == NULL) {
	    FATAL("can't open spec file when writing spec/global");
	    pexit(interfile);
	}
	sprintf(interfile,"%s/%s%s",interpath, comp_sym->s_name, INTER_SUF);
	if (! (inter_old = (stat(interfile,&status) == 0))) {
		if (errno != ENOENT) {
			FATAL("can't stat old .i file");
			end_inter_cleanup();
			return;
		}
	}
	else {
		sprintf(interold,"%s/%s.old",interpath,comp_sym->s_name);
		if (rename(interfile,interold) == -1) {
			FATAL("can't rename old .i file");
			end_inter_cleanup();
			return;
		}
	}


	fprintf(wi_fd,"#\t%s\n",sourcepath);
	NOW(inter_time);

	switch (mytok) {
		case TK_GLOBAL:
			fprintf(wi_fd,"global\n");
			break;
		case TK_RESOURCE:
			fprintf(wi_fd,"resource\n");
			break;
		default:
			boom("bad start_inter");
			/*NOTREACHED*/
	}
	fprintf(wi_fd,"%s\n",comp_sym->s_name);

	impcmpinit();
}



/* called at the end of a global or spec.
 * (only called at top level; e.g., not for imports)
 * for a combined resource or a separate spec writes out the keyword separate.
 * for global or abstract resource, the keyword end was already written out.
 * then just closes file and renames it.
 */
void
end_inter(mytok)
	Token mytok;
{
	impcmpfini();

	/* if already encountered an error. */
		if (wi_fd == NULL) return;

	switch (mytok) {
		case TK_END:
			break;
		case TK_RESOURCE:
			/* write separate, even if combined resource. */
			fprintf(wi_fd,"separate\n");
			break;
		default:
			boom("bad end_inter");
			/*NOTREACHED*/
	}

	end_inter_cleanup();
}


/* cleanup at end of writing interface file
 * or after an error occurred while writing interface file.
 */
static void
end_inter_cleanup()
{
	char intertemp[MAX_PATH];

	if (wi_fd) {
		fclose(wi_fd);
		wi_fd = NULL;
	sprintf(interfile,"%s/%s%s",interpath,comp_sym->s_name,INTER_SUF);
	sprintf(intertemp,"%s/%s.new",interpath,comp_sym->s_name);
	if (fatal_err_cnt==0) {
		if (rename(intertemp,interfile) == -1)
			FATAL("can't rename new .i file");
		if (inter_old) {
			sprintf(intertemp,"%s/%s.old",
					interpath,comp_sym->s_name);
			if (unlink(intertemp) == -1)
				FATAL("can't remove old .i file");
		}
	}
	else {
		if (unlink(intertemp) == -1)
			FATAL("can't remove new .i file");
		if (inter_old) {
			sprintf(intertemp,"%s/%s.old",
					interpath,comp_sym->s_name);
			if (rename(intertemp,interfile) == -1)
				FATAL("can't restore old .i file");
		}
	}
	}
}

/* writes yytext to the .i file, or escaped version for a string token.  */
void
write_to_inter(){
	if (tok == TK_STRLIT) {
		assert(yynode->e_op == TK_STRLIT);
		wescape(wi_fd,yynode->e_str);
		fprintf(wi_fd,"\n");
	}
	else
		fprintf(wi_fd,"%s\n",yytext);
}


/* called once for each resource body to establish 'st' table, if necessary.
 * mode is:
 *	TK_BODY for a separate body,
 *	TK_RESOURCE for a combined resource.
 * returns TRUE if all ok, FALSE otherwise.
 */
Bool
start_body_inter(mode)
Token mode;
{
    Symptr my_comp_sym;

    if (mode == TK_BODY) {
	NOW(inter_time);

	if ((my_comp_sym = read_inter(comp_name,TK_BODY)) == NULL){
	    return (FALSE);
	}
	if (my_comp_sym->s_type != T_SPEC) {
	    assert (my_comp_sym->s_type == T_GLOBAL);
	    FATAL("can't have body for a global component");
	    return (FALSE);
	}
	/* the just-read-in spec block got popped. put it back.
	 */
	push_block(my_comp_sym);
	comp_sym = my_comp_sym;
	impcmpinit();
	return (TRUE);
    } else if (mode == TK_RESOURCE) {
	/* even though we don't need to do anything here,
	 * it doesn't hurt much to call this routine.
	 * might want to do some initialization here sometime...
	 */
	impcmpinit();
	return (TRUE);
    } else {
	boom("invalid parameter to body_inter");
	/*NOTREACHED*/
    }
}


/* end_body_inter()
 * called at end of body (or combined resource) to update body file.
 */
void
end_body_inter()
{
    FILE *bfd;
    char bodyfile[MAX_PATH], mesg[30+MAX_PATH], sourcepath[MAX_PATH];
    int  arg_cnt;
    Symptr s;

    assert   (comp_sym->s_kind == K_BLOCK
	   && comp_sym->s_type == T_SPEC
	   && comp_sym->s_tdef != NULLSYM);
    assert (comp_name != NULL);

    impcmpfini();

/** alternatively, if fatal_err_cnt != 0 then remove body file for
 ** current resource.  the current way allows link with old body
 ** which might be good or bad ...
 **/
    if (fatal_err_cnt > 0) return;

    sprintf(bodyfile,"%s/%s%s",interpath,comp_name,BODY_SUF);

    if ((bfd = fopen(bodyfile,"w")) == NULL) {
	sprintf(mesg,"can't create body file (%s)",bodyfile);
	FATAL(mesg);
	pexit(bodyfile);
    }

    /* set arg_cnt to number of parameters in current resource. */
    for (arg_cnt = 0, s = comp_sym->s_tdef; s; s = s->s_next) {
	if (s->s_kind == K_PARAM)
 	    arg_cnt++;
    }

    get_sourcepath(sourcepath);

    fprintf(bfd,"%s %d %d\n",sourcepath,arg_cnt,final_done);

    fclose(bfd);
    return;
}


/* called once for each component in an import statement.
 * makes an K_IMPORT entry in the main st for cname and
 * an at table for the stuff in its spec.
 * returns pointer to that entry.
 * which tells whether import is explicit (TK_IMPORT) or implicit (TK_EXTEND)
 */
Symptr
import_inter(cname,which)
Token which;
char *cname;
{
	Symptr new;

	assert (which == TK_EXTEND || which == TK_IMPORT);

	if ((new = read_inter(cname,TK_IMPORT)) != NULL) {
		assert(new->s_kind == K_BLOCK);
		new->s_kind = K_IMPORT;
		new->s_how_imported = which;
		import_append(new);
	}
	return new;
}



/* called once for each component in an extend statement.
 * put the entries right in the main st as if they
 * had appeared there.
 */
void
extend_inter(cname)
char *cname;
{
	Symptr new;

	if ((new = read_inter(cname,TK_EXTEND)) != NULL) {
		extend_append(new->s_tdef);
		/* can free new here (it is not on lb list). */
	}
}



/*
 * install all objects in new's aux table into comp_sym's aux table.
 * need to check that no duplicates are inserted.
 * we'll just append the new table, after first checking for duplicates.
 * note: this unlinks the aux table from the old table.
 * this should be ok since that isn't used
 * (it does go on the lb list, but that shouldn't matter.)
 * also note that if do optimize imports so read given .i file only once,
 * that still must read .i so that create a new table for extend!
 * on error, we'll do the extend anyway since that is likely to
 * reduce number of subsequent error messages.
 */
static void
extend_append(table)
Symptr table;
{
	Symptr s, last;
	assert(table); /* points to empty header. */
	if (table->s_next == NULLSYM) return; /* empty spec. */
 	for (s = table->s_next; s; s = s->s_next) {
		if (st_dup(s->s_name)) {
			errmsg(E_FATAL,"duplicate identifier during extend: %s",
				s->s_name);
		}
	}
	assert(comp_sym->s_tdef);
	for (s = comp_sym->s_tdef; s; s = s->s_next)
		last = s;
	last->s_next = table->s_next;
	table->s_next->s_prev = last;
}



/* constructs pathname of source file.  fills in passed array.  */
static void
get_sourcepath(path)
	char path[];
{
	if (*source_file != '/') {
		if (getwd(path) == 0) {
			FATAL("can't get current directory (get_sourcepath)");
			return;
		}
		sprintf(path,"%s/%s",path,source_file);
	}
	else
		strcpy(path,source_file);

}

/* read in and build a symbol table for an .i file.
 * this is done by invoking parse recursively.
 * important input and parsing global variables are saved/restored.
 * note: we don't save all globals, since we know we're parsing just a spec;
 * e.g., we don't save final_done.
 * turn off writing .i file too.
 * note: saving the state of the scanner is very dependent on the code
 * lex generates. we cheat by saving only a single character in the buffer,
 * since that is all the look ahead that is required.
 * alternatively, we could copy the entire buffer, but we don't know
 * YYLMAX in here (of course we could hack it in...)
 * don't need to save:
 *	segment -- reset as needed.
 *	lb_end  -- want all blocks on list so can figure out sizes/offsets
 *		   (and don't tidy_sym unless top level)
 *	fatal_err_cnt	-- want these to propogate up to importer.
 *	warn_err_cnt	--
 * returns a pointer to the newly created symbol table or NULL on failure.
 */
static Symptr
read_inter(spec_name,which)
Token which; /* TK_BODY, TK_IMPORT, TK_EXTEND; controls read_inter
		and for better error messages. */
char *spec_name; /* spec or global name. */
{
	Scanptr old_scan_state;
	FILE *old_wi_fd;
	Bool old_parsing_body;
	Symptr old_comp_sym;
	char *old_comp_name;
	Symptr old_main_table;
	Classptr old_classes;

	FILE *myfd;
	Symptr old_block;
	Symptr my_comp_sym;
	time_t old_inter_time;

	sprintf(interfile,"%s/%s%s",interpath,spec_name,INTER_SUF);
	if ((myfd = fopen(interfile,"r")) == NULL) {
		switch (which) {
			case TK_BODY:
			  FATAL("can't open .i file for separate body;");
			  break;
			case TK_IMPORT:
			  FATAL("can't open .i file for imported object;");
			  break;
			case TK_EXTEND:
			  FATAL("can't open .i file for extended object;");
			  break;
			default:
			  boom("read_inter. bad which");
			  /*NOTREACHED*/
		}
		fprintf(stderr,"\tits spec probably hasn't been compiled.\n");
		if (which != TK_BODY)
			fprintf(stderr,"\t%simport/extend of %s into %s\n",
				inter_depth > 0?"nested ":"",
				spec_name, comp_name);
		pexit(interfile);
	}

	/* check if imported object is older than importer.
	 * note: let equality slip by since granularity of mtime
	 * is pretty coarse.
	 */
	if (do_time_check) {
	    struct stat status;
	    if (stat(interfile,&status) != 0) {
		boom("read_inter can't stat file");
		/*NOTREACHED*/
	    }
	    if (status.st_mtime > inter_time) {
		WARN("imported object newer than its importer.");
		fprintf(stderr,"\tnested import of %s into %s\n",
				spec_name, comp_name);
	    }
	    else {
		old_inter_time = inter_time;
		inter_time = status.st_mtime;
	    }
	}

	/* prevent circularities in imports. */
	    if ( impcmpcheck(spec_name) ) {
		FATAL("circularity in import or extend.");
		fprintf(stderr,"\tnested import or extend of %s into %s\n",
				spec_name, comp_name);
		fclose(myfd);
		return NULL;
	    }
	    impcmppush(spec_name);

	/* the object being imported/extended is brought in the environment
	 * in which it is was defined; i.e., at the top level with only
	 * predefs and any imports it does, not in the environment of its
	 * importer.
	 * (importing the spec for a body requires nothing special.)
	 * thus, the symbol table routine to do this should search only
	 * the predef level (and the object).
	 * to achieve this, we temporarily remove comp_sym's tables
	 * from the symbol table; they get restored later.
	 * this whole business and symbol table routines might
	 * benefit from massive reorganization.
	 */
		if (which != TK_BODY) {
			assert (comp_sym);
			assert (comp_sym->s_type == T_SPEC
				|| comp_sym->s_type == T_GLOBAL);
			assert (comp_sym->s_kind == K_BLOCK);
			assert (st_cb);
			old_block = pop_component();
		}

	/* save globals. */
	    old_scan_state = get_scan_state();
		old_wi_fd = wi_fd;
		old_parsing_body = parsing_body;
		old_comp_sym = comp_sym;
		old_comp_name = comp_name;
		old_main_table = main_table;
		old_classes = classes;
	/* re-initialize globals and parse the .i file. */
		yysptr = yysbuf;
		classes = NULLCLASS;
		inter_depth++;
		yyin = myfd;
		wi_fd = NULL;
		main_table = NULLSYM;
		parse();
		my_comp_sym = comp_sym;
		fclose(yyin);
		inter_depth--;
	/* restore globals. */
	    set_scan_state(old_scan_state);
		wi_fd = old_wi_fd;
		parsing_body = old_parsing_body;
		comp_sym = old_comp_sym;
		comp_name = old_comp_name;
		main_table = old_main_table;
		classes = old_classes;

	/* restore comp_sym's block. */
		if (which != TK_BODY) {
			push_component(old_block);
		}

	impcmppop();

	inter_time = old_inter_time;
	return (my_comp_sym);
}

/* imported components.
 * used for building a stack so can detect circularities in imports.
 * note: this stack could be threaded through the symbol table,
 * but we'll keep it separate for simplicity.
 */
typedef struct imp_comp {
	char *name;		/* comp name */
	struct imp_comp *next;	/* forward link */
	struct imp_comp *prev;	/* backward link */
} ImpComp;

typedef ImpComp *ImpCompPtr;

static ImpCompPtr icsp = NULL; /* the stack pointer. */

static void
impcmpinit()
{
	assert( icsp == NULL );
	impcmppush( comp_sym->s_name );
}

static void
impcmpfini()
{
	impcmppop();
	assert( icsp == NULL );
}

static void
impcmppush(name)
char *name;
{
	ImpCompPtr new;
	new = (ImpCompPtr) alloc( sizeof(ImpComp) );
	new->name = nt_lookup(name,TRUE);
	new->prev = icsp;
	new->next = NULL;
	icsp = new;
}

static void
impcmppop()
{
	ImpCompPtr top;
	assert( icsp );
	top = icsp;
	icsp = top->prev;
	free( (char *) top );
}

/* returns TRUE iff name exists on stack. */
static Bool
impcmpcheck(name)
char *name;
{
	ImpCompPtr x;
	for( x = icsp; x; x = x->prev ){
	    if( strcmp(x->name,name) == 0 ) return(TRUE);
	}
	return(FALSE);
}
