/* routines that deal with intermediate code for the input statement.  */

#include <stdio.h>
#include "../util.h"
#include "sr.h"
#include "funcs.h"
#include "globals.h"

static Icode input_what();
static void input_sched();
static void input_start(), input_get(), input_legs();
static void input_block_patch(), input_body_patch();
static void input_ic(), input_ic_all(), input_dup_op();
static void in_release(), leg_release();

static Inptr in_top = NULLIN;		/* input statement structures */
static Inptr in_free = NULLIN;		/* free list of structures */
static Legptr leg_free = NULLLEG;	/* free list of legs */


/* input_init
 * called at the beginning of each resource (spec or body).
 * initialize the input list and class list.
 * frees any input structures, legs, and classes left from last time.
 */
void
input_init()
{
    Inptr in, next_in;
    Legptr leg, next_leg;

    for (in = in_top; in != NULLIN; in = next_in) {
	next_in = in->in_next;
	for (leg = in->in_legs; leg != NULLLEG; leg = next_leg) {
	    next_leg = leg->leg_next;
	    leg_release(leg);
	}
		if (in->else_present) leg_release(in->else_leg);
	in_release(in);
    }
    in_top = NULLIN;

    class_init();
}


/* do_input
 * called at the end of a resource body to do everything for input statements:
 *
 *	get rid of proc ops and make sure all ops are implemented
 * 	make all the equivalence classes
 *	generate the intermediate code for input statements and patch it in
 *
 */
void
do_input()
{
    class_check();

    /* some things could blow up if we proceed after fatal errors,
     * especially because of unimplemented operations.
     * also, save time.
     */
    if (fatal_err_cnt > 0)
	return;

    equiv();
    class_print();
    input_ic_all();
}


/* input_ic_all()
 * produce the intermediate code for all input statements.
 */
static void
input_ic_all()
{
    Inptr in;

    emit(I_INP_ALL,NULLNODE,NOLAB);

    /* first, see if any operations are semaphore operations. */
    sem_analysis();

    for (in = in_top; in != NULLIN; in = in->in_next){
	/* set the class for this input statement.
	 * done here once instead of chasing pointers all the time.
	 */
	assert (in->in_legs != NULLLEG);
	assert (in->in_legs->leg_sym != NULLSYM);
	assert (in->in_legs->leg_sym->s_class != NULLCLASS);
	assert (in->in_legs->leg_sym->s_class->cl_members > 0);
	in->in_class = in->in_legs->leg_sym->s_class;

	input_ic(in);

    }
    emit(I_INP_ALL_END,NULLNODE,NOLAB);
}


/* input_ic(in)
 * produce the intermediate code for input statement in.
 * apply optimizations if possible.
 */
static void
input_ic(in)
Inptr in;
{
    int end_in;		/* label at end of input statement. */
    int top_loop;	/* label at top of loop. */
	int else_end; /* in exit for else leg. */
    Instptr old_ic_tail;
    Icode itype;

    end_in = NEWLAB;
    top_loop = NEWLAB;
	if (in->else_present)  else_end = NEWLAB;

    assert (in != NULLIN);

    /* remember where ic_tail is for repatch at bottom. */
		old_ic_tail = ic_mark();

    /* what kind of input statement and operation do we have? */
    itype = input_what(in);

    /* beginning of intermediate code. */
    switch (itype) {
	case I_INP_GET_INV:		/* get access to the class. */
	case I_INP_GET_NAMED_INV:	/* get access to the class. */
	    input_start(I_INP_ACCESS,in);
	    break;
	case I_INP_RECEIVE:		/* do the receive. */
	    input_start(I_INP_RECEIVE,in);
	    break;
	case I_INP_SEM_P:
	    /* do nothing. */
	    break;
	default:
	    boom("bad itype in input_ic beginning");
	    /*NOTREACHED*/
    }

    /* guts of intermediate code. */
    switch (itype) {
	case I_INP_GET_INV:
	case I_INP_GET_NAMED_INV:
	    /* loop over pending invocations in class. */
	    emit(I_LABEL,NULLNODE,top_loop);

	    /* get the next invocation. */
	    input_get(itype,in);

	    /* see if any of the legs wants it; generate code for legs. */
	    input_legs(itype,in,top_loop,end_in,else_end);

	    /* branch back for another invocation. */
	    emit(I_BRANCH, NULLNODE,top_loop);
	    break;

	case I_INP_SEM_P:
	    /* do the P operation. */
	    input_get(I_INP_SEM_P,in);
	    /* fall through to patch in block and body. */
				/* in op() -> leg1 [] op() -> leg2 ni
				 * such can also happen in other cases,
				 * though it is harder to detect,
				 * at least until Carole does dup_ops.
				 * such should be done in input_what?
				 */
					if (in->in_legs->leg_next)
					    WARN(
		"duplicate op with no synch expr -- nondeterministic choice");
	case I_INP_RECEIVE:
	    input_block_patch(in->in_legs);	/* patch in the block. */
	    input_body_patch(in->in_legs);	/* patch in the body. */
	    break;

	default:
	    boom("bad itype in input_ic guts");
	    /*NOTREACHED*/
    }

    /* generate pieces of code for exit and next.
     * each piece tells the RTS that we're done
     * and then branchs to the real exit or next label.
     * i.e., exit and next within in means to terminate
     * the in statement and then do the normal (e.g., for do)
     * exit or next.
     * if there are no exits or loops within the in,
     * we rely on the code generator to discard this unreachable code.
     *
     * note: the branch before each code piece code is important.
     * I_INP_SEM_P and I_INP_RECEIVE
     * would otherwise fall through into exit/next code.
     * the other itype's have a branch back right before this.
     * these branch arounds will be discarded by code generator
     * if unreachable.
     */
    if (in->in_old_exit_label != NOLAB) {
	emit(I_BRANCH,NULLNODE,end_in);
	emit(I_LABEL,NULLNODE,in->in_exit_label);
	if (itype != I_INP_SEM_P)
	    emit(I_INP_DONE,NULLNODE,NOLAB);
	emit(I_BRANCH,NULLNODE,in->in_old_exit_label);
    }
    if (in->in_old_next_label != NOLAB) {
	emit(I_BRANCH,NULLNODE,end_in);
	emit(I_LABEL,NULLNODE,in->in_next_label);
	    if (itype != I_INP_SEM_P)
		emit(I_INP_DONE,NULLNODE,NOLAB);
	emit(I_BRANCH,NULLNODE,in->in_old_next_label);
    }

    /* tell RTS that done with input statement and to do the reply. */
    emit(I_LABEL,NULLNODE,end_in);
    if (itype != I_INP_SEM_P){
	if (in->in_return_label != NOLAB)
	    emit(I_LABEL,NULLNODE,in->in_return_label);
	emit(I_INP_DONE,NULLNODE,NOLAB);
    }

    /* dump label for else exit if needed. */
    if (in->else_present)
	emit(I_LABEL,NULLNODE,else_end);

    /* tell code generator that we're done with input statement. */
    emit(I_INP_END,NULLNODE,NOLAB);

    /* patch the above generated intermediate code
     * into where it belongs (we hope).
     */
    assert(in->in_start->i_type == I_INP_START);
    ic_move(old_ic_tail,in->in_start);
}


/* input_get(itype,in)
 * produce intermediate code to get the invocation.
 * itype is the kind of intermediate code to generate.
 * note: RTS blocks us if there aren't any invocations.
 */
static void
input_get(itype,in)
Icode itype;
Inptr in;
{
    Legptr leg;

    assert(itype == I_INP_GET_INV
	|| itype == I_INP_GET_NAMED_INV || itype == I_INP_SEM_P);

    if (itype == I_INP_GET_INV)
	emit(I_INP_GET_INV,NULLNODE,NOLAB);
    else {
	/* tell the code generator which operation. */
	leg = in->in_legs;
	emit(itype,leg->leg_op,NOLAB);
    }
}


/* input_legs(itype,in,top_loop,end_in,else_end)
 * produce intermediate code for all the legs of input statement in.
 * itype is the kind of intermediate code to generate.
 */
static void
input_legs(itype,in,top_loop,end_in,else_end)
Icode itype;
Inptr in;
int top_loop, end_in, else_end;
{
    Legptr leg;
    /* this next flag is TRUE iff need to generate a label on this leg,
     * which is generally the case.
     * it is FALSE for the first leg, or if the previous leg had a quantifier,
     * in which case the code will fall through.
     * (note: the code for a quantifier branches to 'update loop vars'
     * if match fails.)
     */

    Bool need;
    int next_leg;
    Instptr ic_before;

    assert (itype == I_INP_GET_INV || itype == I_INP_GET_NAMED_INV);

    next_leg = NEWLAB;

    /* if else is present, gen code to test for NULL leg and execute else leg */
    if (in->else_present) {
	emit(I_INP_ELSEMATCH,NULLNODE,next_leg);
	emit(I_INP_REMOVE,NULLNODE,NOLAB);
	input_body_patch(in->else_leg);
	emit(I_BRANCH,NULLNODE,else_end);
	need = TRUE;
    } else
	need = FALSE;
    input_dup_op(in);
    for (leg = in->in_legs; leg != NULLLEG; leg = leg->leg_next) {
	if (need)
	    emit(I_LABEL,NULLNODE,next_leg);

	/* quantifier.
	 * if present, patch in the quantifier code.
	 * we'll patch the body into it later.
	 */
	ic_append(&leg->leg_quant);
	ic_before = ic_mark();
	next_leg = NEWLAB;
	assert (leg->leg_op != NULLNODE);

	/* if getting an invocation in a class with more
	 * than one op and we haven't specified its 'name',
	 * then see if it is what we want.
	 */
	if (itype != I_INP_GET_NAMED_INV && !input_singular(leg->leg_sym))
	    emit(I_INP_MATCH,leg->leg_op,next_leg);

	/* patch in the block code. */
	input_block_patch(leg);

	/* synchronization expression. */
	if (leg->leg_synch != NULLNODE) {
	    emit(I_BRANCH_FALSE,
		leg->leg_synch,leg->dup_op ? next_leg : top_loop);
	}

	/* scheduling expression.
	 * if present, select and remove the minimizing invocation.
	 * o.w., be happy with this invocation.
	 */
	if (leg->leg_sched != NULLNODE)
	    input_sched(leg);
	else
	    emit(I_INP_REMOVE,NULLNODE,NOLAB);

	/* finally, got a good invocation! patch in the body. */
	input_body_patch(leg);

		/* done with arm and therefore in statement. */
	emit(I_BRANCH,NULLNODE,end_in);

	/* if quantifier was present, patch all the above into it.
	 * update need.
	 */
	if (leg->leg_quant.il_head.i_next != NULLINST) {
	    /* be sure we generated something above. */
	    assert (ic_before != NULLINST);
	    /* generate the failure label. */
	    emit(I_LABEL,NULLNODE,next_leg);
	    need = FALSE;
	    ic_move(ic_before,leg->leg_quant_guts);
	} else
	    need = TRUE;
    }

    if (need)
	emit(I_LABEL,NULLNODE,next_leg);

}

/* input_sched(leg)
 * produce intermediate code for the scheduling expression on leg.
 * this code evaluates the scheduling expression for each invocation
 * of this operation and removes the minimum of those.
 */
static void
input_sched(leg)
Legptr leg;
{
    int top_loop, end_loop; /* labels on top and bottom of loop. */

    top_loop = NEWLAB;
    end_loop = NEWLAB;

    assert (leg != NULLLEG && leg->leg_sched != NULLNODE);

    /* evaluate scheduling expression and save its value as a minimum
     * and a pointer to this invocation structure.
     */
    emit(I_INP_SET_MIN,leg->leg_sched,NOLAB);

    /* loop over pending invocations in this class. */
    emit(I_LABEL,NULLNODE,top_loop);

    /* get the next invocation in this class. if it's null, exit loop. */
    emit(I_INP_GET_NAMED_INV_NB,NULLNODE,end_loop);

    /* see whether its synchronization expression is true. */
    if (leg->leg_synch != NULLNODE)
	emit(I_BRANCH_FALSE, leg->leg_synch,top_loop);

    /* evaluate its scheduling expression and
     * update current minimum if necessary.
     */
    emit(I_INP_UPDATE_MIN,leg->leg_sched,NOLAB);

    /* end of body of loop. go to top. */
    emit(I_BRANCH,NULLNODE,top_loop);

    /* bottom of loop: remove minimum. */
    emit(I_LABEL,NULLNODE,end_loop);
    emit(I_INP_REMOVE_MIN,NULLNODE,NOLAB);
}


/* input_start(itype,in)
 * called to produce code at start of input statement in.
 * itype is the kind of intermediate code to produce.
 */
static void
input_start(itype,in)
Icode itype;
Inptr in;
{
    union e_lu e;

    assert (itype == I_INP_ACCESS || itype == I_INP_RECEIVE);
    assert (in->in_class != NULLCLASS);

	if (in->else_present) itype = I_INP_ELSEACCESS;

    /* build a class node for the code generator. */
    e.e_class = in->in_class;
    emit(itype,make_node(TK_CLASS,e,NULLNODE),NOLAB);
}


/* input_block_patch(leg)
 * patch the block for leg into the intermediate code.
 */
static void
input_block_patch(leg)
Legptr leg;
{
    assert (leg != NULLLEG);
    ic_append(&leg->leg_block);
}


/* input_body_patch(leg)
 * patch the body for leg into the intermediate code.
 */
static void
input_body_patch(leg)
Legptr leg;
{
    assert (leg != NULLLEG);
    ic_append(&leg->leg_body);
}


/* input_what(in)
 * determine what kind of intermediate code to produce for input statement in;
 * it depends on both the operation and the input statement.
 * returns the principal Icode that distinguishes 
 * the code that will be produced as follows:
 *
 *	I_INP_RECEIVE
 *	I_INP_GET_INV	(general case)
 *	I_INP_GET_NAMED_INV
 *	I_INP_SEM_P
 */
static Icode
input_what(in)
Inptr in;
{
    Bool quant_present, synch_present, sched_present;
    Legptr leg;

    leg = in->in_legs;

    if (leg->leg_sym->s_impl == IM_SEMAPHORE)
	return (I_INP_SEM_P);

	quant_present = (Bool) (leg->leg_quant.il_head.i_next != NULLINST);

    if (!quant_present && leg->leg_next == NULLLEG) {
	/* note: the above test for the next leg
	 * tells if only one leg in the input statement;
	 * leg now points at that leg.
	 */
	synch_present = (Bool) (leg->leg_synch != NULLNODE);
	sched_present = (Bool) (leg->leg_sched != NULLNODE);
	if (	   !synch_present
		&& !sched_present 
		&& !in->else_present
		&& input_singular(leg->leg_sym))
	    /* simple -- just use the receive RTS primitive. */
	    return (I_INP_RECEIVE);
	else if (input_singular(leg->leg_sym))
	    /* if just one oper in class, then don't need to specify name. */
	    return (I_INP_GET_INV);
	else
	    /* use the named invocation RTS primitive. */
	    return (I_INP_GET_NAMED_INV);
    } else
	return (I_INP_GET_INV);
}


/* input_singular(sym) -- returns TRUE iff operation sym is singular.
 * singular means only one operation in the class
 * and if the operation is an array, then only one element in the array.
 * of course, arrays whose sizes are determined at run-time don't qualify.
 * (yeah, sort of silly to try to do this for arrays, but it is easy.)
 */
Bool
input_singular(sym)
Symptr sym;
{
    Symbol sig;

    assert (sym->s_class->cl_members > 0);
    if (sym->s_class->cl_members > 1) {
	return (FALSE);
    } else {
	range_size(sym,&sig,FALSE);
	return ((Bool) (sig.s_size == 1));
    }

}


/* input_dup_op()
 * called before intermediate code generated to determine
 * existence of multiple occurences of an op within in
 * no multiple ocurrences implies optimization of exit from failed
 * synchronization condition
 */
static void
input_dup_op(in)
Inptr in;
{
    Legptr leg, nleg;

    for (leg = in->in_legs; leg != NULLLEG; leg = leg->leg_next) {
	if (!leg->dup_op) {
	    for (nleg = leg->leg_next; nleg != NULLLEG; nleg = nleg->leg_next) {
		if (leg->leg_sym == nleg->leg_sym) {
		    leg->dup_op = TRUE ;
		    nleg->dup_op = TRUE ;
		}
	    }
	}
    }
}        


/* in_create()
 * called at the beginning of an input statement.
 * allocates an in structure and links it into the input statement list.
 * initialize its legs field to NULLLEGS.
 */
Inptr
in_create()
{
    register Inptr t;

    /* allocate. */
    if (in_free) {
	t = in_free;
	in_free = in_free->in_next;
    } else
	t = (Inptr)alloc(sizeof(In));

    /* link in. */
    t->in_next = in_top;
    in_top = t;

    t->in_return_label = NOLAB;
    t->in_legs = NULLLEG;
	t->else_present = FALSE;
    return (t);
}


/* free an in structure and link it into the free list.
 */
static void
in_release(in)
Inptr in;
{
    assert (in != NULLIN);

    in->in_next = in_free;
    in_free = in;
}


/* allocate a leg structure,
 * link it into in structure,
 * and initialize its link fields.
 */
Legptr
leg_create(in)
Inptr in;
{
    register Legptr t;

    assert (in != NULLIN);

    /* allocate. */
    if (leg_free) {
	t = leg_free;
	leg_free = leg_free->leg_next;
    } else
	t = (Legptr)alloc(sizeof(Leg));

    /* if not an else leg - link it in. */
    if (!in->else_present) {
	t->leg_next = in->in_legs;
	in->in_legs = t;
    }

    /* initialize; initializing Inst header i_next fields is crucial to
     * above input routines, e.g. to determine if quantifier was present.
     */
    t->leg_block.il_head.i_next = NULLINST;
    t->leg_body.il_head.i_next  = NULLINST;
    t->leg_quant.il_head.i_next = NULLINST;
    t->dup_op = FALSE;

    return (t);
}


/* free a leg structure and link it into the free list.
 */
static void
leg_release(leg)
Legptr leg;
{
    assert (leg != NULLLEG);

    leg->leg_next = leg_free;
    leg_free = leg;
}



/*  inlist() - return top of input list for others to use  */

Inptr
inlist()
{
    return in_top;
}
