/*  impl.c -- Implementation Statements.  */
/*  note: error recovery here is minimal.  */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"

static Symptr op_header();
static Nodeptr receive_formals(), receive_variables();
static void op_id();

static Token follow_in[] = {TK_SQUARE,TK_NI,TK_NOTATOKEN};


/* N.B. some of the code below should be reorganized, e.g.:
 *
 *	(1) doing the block_begin down in op_header and block_end in in_stmt
 *	makes for strange code.  especially makes error handling difficult.
 *
 *	(2) receive uses op_id. if there is an error, though, op_id uses the
 *	follow set for input.  op_id should probably be parameterized.
 */


/* the input statement.
 *
 * at this level, we just build up all the information about an input statement
 * into a structure.  later, we'll analyze this structure and generate the
 * appropriate intermediate code. the reason for this approach is that when
 * we are parsing an input statement we don't yet know how the operations are
 * divided into classes; without this information, optimizations are difficult.
 * also, these optimizations are simpler for us at this level than for the code
 * generator.
 */
void
in_stmt()
{
    int old_return_label;
    Inptr in;
    Instptr ic_block, ic_body;
    Symptr quant_block;		/* block for quantifier if present, else null.*/
    Legptr leg;

    old_return_label = return_label;
    return_label = NEWLAB;

    in = in_create();	

    /* remember both old & new labels in input structure. see input.c for why.*/
    in->in_return_label = return_label;
    in->in_old_exit_label = exit_label;
    in->in_old_next_label = next_label;

    /* make exit/next labels only if there already were valid ones;
     * e.g., we're in a do statement.
     */
    if (exit_label != NOLAB) {
	assert (next_label != NOLAB);
	exit_label = in->in_exit_label = NEWLAB;
	next_label = in->in_next_label = NEWLAB;
    } else {
	assert (exit_label == NOLAB);
	assert (next_label == NOLAB);
    }

    /* remember where the intermediate code for this input statement belongs. */
    emit(I_INP_START, NULLNODE, NOLAB);
    in->in_start = ic_mark();

    do {
	leg = leg_create(in);

	/* remember start of intermediate code for block. */
	ic_block = ic_mark();

	quant_block = op_header(leg);

	/* stow the block intermediate code, if any. */
	ic_stow(ic_block,&leg->leg_block);

	mustbe(TK_ARROW,"->");

	/* remember start of intermediate code for the body. */
	ic_body = ic_mark();

	command_block(follow_in);
	assert (tok == TK_SQUARE || tok == TK_NI);
	block_end();

	/* stow the body intermediate code, if any. */
	ic_stow(ic_body,&leg->leg_body);

	/* done with the quantifier block if there is one. */
	if (quant_block != NULLSYM)
	    block_end();

    } while (tok != TK_NI && !(maybe(TK_ELSE)));

    /* restore old labels here so else part works right for such. */
    return_label = old_return_label;
    exit_label = in->in_old_exit_label;
    next_label = in->in_old_next_label;

    if (tok == TK_ELSE) {
	in->else_present = TRUE;
	mustbe(TK_ARROW,"->");
	in->else_leg = leg_create(in);
	ic_body = ic_mark();
	new_command_block(follow_in);
	if (tok != TK_NI) 
	    FATAL("else must be last in in statement");
	ic_stow(ic_body,&in->else_leg->leg_body);
    }

    if (dbflags['T'])
	printf("end of input_statement\n");
}


/* parse the operation header and fill in leg.
 * returns pointer to quantifier block, if present, else null.
 */
static Symptr
op_header(leg)
Legptr leg;
{
    Symptr block_sym;
    Symptr quant_block;		/* return value. */
    Instptr ic_before;

/**** NEEDS WORK: fix up so if error, do something reasonable ****/

    /* quantifier.
     * store up its intermediate code and a pointer to where
     * its guts should be placed.
     */
    if (maybe(TK_LEFTPAREN)) {
	ic_before = ic_mark();
	quant_block = block_begin(T_CMD);
	leg->leg_quant_guts = quantifier(FALSE);
	ic_stow(ic_before,&leg->leg_quant);
	mustbe(TK_RIGHTPAREN,") around in's quantifier");
    } else {
	quant_block = NULLSYM;
    }

    op_id(leg);

    /* note: crucial to code generator that get I_BLOCK for input block
     * before synchronization, etc. expressions.
     */
    block_sym = block_begin(T_INPUT);
    block_sym->s_tdef = leg->leg_sym;

    formal_ids(block_sym);

    /* synchronization expression. */
		if (maybe(TK_AND) || maybe(TK_SUCHTHAT))
	leg->leg_synch = bool_expr();
    else
	leg->leg_synch = NULLNODE;

    /* scheduling expression. */
    if (maybe(TK_BY))
	leg->leg_sched = ot_expr();
    else
	leg->leg_sched = NULLNODE;

    return (quant_block);
}


/* op_id(leg)
 * parses operation identifier part, including subscripts, of operation_guard.
 * sets leg's symbol table entry of operation,
 * and leg's node.
 * if problem, set leg's symbol to NULLSYM.
 */
static void
op_id(leg)
Legptr leg;
{
    Symptr op_sym;
    Nodeptr n;

    assert(leg != NULLLEG);

    mustbe(TK_IDENTIFIER,"op_identifier");

    if ((op_sym = st_lookup(tk_str)) == NULLSYM) {
	ERROR(E_FATAL+1,tk_str);
	find_follow(follow_in);
	leg->leg_sym = NULLSYM;
	return;
    }
    if (op_sym->s_kind != K_OP
	    || (op_sym->s_type != T_FUNC && op_sym->s_type != T_VOID)) {
	FATAL("bad op identifier");
	leg->leg_sym = NULLSYM;
	return;
    }
    switch (op_sym->s_impl) {
	case IM_NOTYETIMPL:
	    op_sym->s_impl = IM_INPUT;
	    break;
	case IM_PROC:
	case IM_PROCESS:
	    errmsg(E_FATAL, "op already implemented by proc or process",
		op_sym->s_name);
	    break;
	case IM_INPUT:
	    break;
	case IM_EXTERNAL:
	    FATAL("attempt to implement an external");
	    return;
	default:
	    boom("bad impl in op_header");
	    /*NOTREACHED*/
    }

    /* make an id node for the operation. */
    n = idnode(op_sym);

    /* subscripts. */
    if (maybe(TK_LEFTBKET))
	n = bnode(TK_INDEX,n,indices());

    /* make the input node for the operation.
     * signature will check to see that subscripts if any are ok here.
     */
    leg->leg_op = bnode(TK_INPUT,n,NULLNODE);

    leg->leg_sym = op_sym;
}


/* receive or P statement (depending on tk).
 * treat just like an input statement with one leg.
 * builds a tree just like that for (multiple) assignment.
 * on the left-hand side are the denotations of the variables,
 * on the right-hand side are the denotations of the formals.
 * (no such tree gets built if there aren't any variables.)
 */
void
receive_stmt(tk)
Token tk;
{
    Inptr in;
    Legptr leg;
    Instptr ic_block, ic_body;
    Nodeptr lhs,rhs;
    Symptr block_sym;

    in = in_create();

    /* can't have exit or next within receive, but still need to set labels
     * so don't produce anything for next and exit.
     */
    in->in_old_exit_label =	in->in_old_next_label = NOLAB;

    /* remember where the intermediate code for this input statement belongs. */
    emit(I_INP_START, NULLNODE, NOLAB);
	in->in_start = ic_mark();

    leg = leg_create(in);

    if (tk == TK_P)
	mustbe(TK_LEFTPAREN, "( in P statement");
    op_id(leg);
    if (tk == TK_P)
	mustbe(TK_RIGHTPAREN, ") in P statement");
    if (leg->leg_sym == NULLSYM)
	return;

    /* initialize rest of fields in leg. */
    leg->leg_quant_guts  = NULLINST;
    leg->leg_synch  = NULLNODE;
    leg->leg_sched  = NULLNODE;

    if (leg->leg_sym->s_type != T_VOID)
	WARN("receive or P on a value-returning operation");

    /* remember start of intermediate code for block. */
	ic_block = ic_mark();

    /* make a block for the formals, even if none.
     * note: see note in receive_formals about names.
     */
    block_sym = block_begin(T_INPUT);
    block_sym->s_tdef = leg->leg_sym;

    /* save the block intermediate code, if any.
     * it is removed from the main intermediate code list.
     */
	ic_stow(ic_block,&leg->leg_block);

    /* remember where the intermediate code for the body begins. */
	ic_body = ic_mark();

    if (tk == TK_P)
		lhs = NULLNODE;
    else {	/* TK_RECEIVE */
	mustbe(TK_LEFTPAREN, "( in receive statement");
	lhs = receive_variables();
    }
	rhs = receive_formals(leg->leg_sym);

    /* if non-zero number of parameters or non-zero number of variables,
     * build a TK_ASSIGN node so that can get signature checked.
     * Note: can't give code generator an empty TK_ASSIGN node,
     * so we handle that case separately.
     * for a signature mismatch,
     * signature gives a fatal error, which inhibits code generation.
     */
    if (lhs == NULLNODE && rhs == NULLNODE){
	/* make some code so that patching works. */
	emit(I_LABEL,NULLNODE,NEWLAB);
    } else while (lhs != NULLNODE && rhs != NULLNODE) {
	/* make the assignment nodes. */
	emit(I_EXPR,bnode(TK_ASSIGN, lhs->e_l, rhs->e_l),NOLAB);
	lhs = lhs->e_r;
	rhs = rhs->e_r;
    }

    if (lhs != NULLNODE || rhs != NULLNODE)
	ERROR(E_FATAL+3,"identifier count");

    /* done with formals block. */
    block_end();

    /* save the body intermediate code, if any.
     * it is removed from the main intermediate code list.
     */
	ic_stow(ic_body,&leg->leg_body);

}


/* builds and returns the denotation list for the variables.  */
static Nodeptr
receive_variables()
{
    Nodeptr top, n, e;

    if (maybe(TK_RIGHTPAREN))
	return (NULLNODE);

    top = NULLNODE;
    do {
		if (!(n = denotation(TK_CALL)))
	    WARN("bad variable denotation in receive");
	else {
	    if (!top)
		top = e = bnode(TK_LIST,n,NULLNODE);
	    else
		e = e->e_r = bnode(TK_LIST,n,NULLNODE);
	    }
    } while (maybe(TK_COMMA));
    mustbe(TK_RIGHTPAREN,") in receive");
    return (top);
}


/* builds and returns the denotation list from the formals for op_sym.
 */
static Nodeptr
receive_formals(op_sym)
Symptr op_sym;
{
    Symptr formal_sym, s;
    Nodeptr top, last;
    Nodeptr n;

    assert(op_sym != NULLSYM);

    /* if no parameters, get out. */
    if (op_sym->s_tdef == NULLSYM)
	return (NULLNODE);

    /* note: skip over head of aux table. */
	formal_sym = op_sym->s_tdef->s_next;
	assert (formal_sym != NULLSYM);

    /* go through the parameter declaration and make unique identifiers.
     * this is necessary so that we have identifiers that appear to the code
     * generator just as parameters.  we use the parameter names themselves.
     * note there is no problem if one of the variables has the same name as
     * one of parameters because st_lookup on the variables is done before
     * parameters are put in this block.  we already caught receive of a
     * value returning op above, but to keep going, we'll just ignore
     * K_RESULT here.  (in this case top could be NULLNODE upon return.)
     */
    /* on parameter names:
     * the above seems silly.   why not just install with NULL name?
     */
    top = last = NULLNODE;
    while (formal_sym != NULLSYM)  {
	if (formal_sym->s_kind != K_RESULT)  {
	    assert (formal_sym->s_kind == K_PARAM);
	    s = st_install(formal_sym->s_name,K_PARAM);
	    assert (s != NULLSYM);
	    copy_from_decl(s,formal_sym);
	    n = bnode(TK_LIST,idnode(s),NULLNODE);
	    if (!top)
		top = n;
	    else
		last->e_r = n;
	    last = n;
	}
	formal_sym = formal_sym->s_next;
    }

    return (top);
}
