/* icode.c -- intermediate code collection and */

#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"
#include "../srsys.h"
#include "../util.h"

static void ic_unstow ();
static void cgen(), gen_expr_code(), free_all_blocks();
static Instptr icreate();



/* Maintain three lists of intermediate code:
 * ic -- the main list
 * rv -- icode that initializes resource variables
 *       collected and stuck after I_INIT
 * pr -- implicit sends to processes
 *       icode is generated for such; it includes icode for quantifiers,
 *       which can generate arbitrary icode.
 *       collected and stuck before I_INIT_END
 *
 * as code is emitted, emit decides which list to append based on tag field.
 * cl -- current list to append.
 */

/* to define an Inst. */
#define ichinit {0, I_NO_TYPE, NULLNODE, 0, NULLINST, NULLINST}

/* the lists: */
static Instlist ic = { ichinit, &ic.il_head };
static Instlist rv = { ichinit, &rv.il_head };
static Instlist pr = { ichinit, &pr.il_head };
static Instlistptr cl = &ic;               



static char func_name[NAMESIZE];	/* current C function name */

static int alloc_block_ind;
static int alloc_block[MAX_ALLOC];	/* list of fp-offsets of addresses of
					   allocated blocks to be freed when
					   expression evaluation is complete */




/* emit(tag,exp,lab)
 * store intermediate code in appropriate list
 * and generate code at the end of a resource (when see I_COMPONENT_END).
 * We assume that there is always an I_INIT and I_INIT_END.
 */
void
emit(tag, exp, lab)
Icode	tag;
int	lab;
Nodeptr exp;
{
    static Instptr i_init = NULLINST;		/* pointer to I_INIT */
    static Instptr i_init_end = NULLINST;	/* pointer to I_INIT_END */
    Instlistptr new_cl;				/* value of cl for next time */
    register Instptr t;

    /* make the ic node */
    t = icreate(tag,exp,lab);

    /* decide which list to append, and do it.  save INIT and END positions.
     * I_INP_ALL is needed because generated after I_COMPONENT_END.
     */
	switch (tag) {
		case I_INIT:
			assert (i_init == NULLINST && cl == &rv);
			i_init = t;
			/*NOBREAK*/
		case I_FINAL:
		case I_PROC:
		case I_INP_ALL:
			cl = &ic;
			new_cl = cl;
			break;
		case I_INIT_END:
			assert (i_init_end == NULLINST && cl == &ic);
			i_init_end = t;
			/*NOBREAK*/
		case I_COMPONENT:
		case I_COMPONENT_END:
		case I_FINAL_END:
		case I_PROC_END:
		case I_INP_ALL_END:
			cl = &ic;
			new_cl = &rv;
			break;
		case I_PROCESS_SEND:
 			cl = &pr;
 			new_cl = &pr;
 			break;
 		case I_PROCESS_SEND_END:
 			assert (cl == &pr);
 			new_cl = &rv;
 			break;
		case I_EXPR:
			if (dbflags['E'])
				pnode(exp);
			/*NOBREAK*/
		default:
			/* stay on current list. */
			new_cl = cl;
			break;
	}

    cl->il_tail->i_next = t;
    t->i_back = cl->il_tail;
    cl->il_tail = t;

    /* set list for next time. */
    cl = new_cl;

    if (tag != I_COMPONENT_END)
	return;

    /******* following code is for I_COMPONENT_END only ********/

    /* There was user initial or we made an initial.
     * this assumption is critical below.
     */
    assert (i_init != NULLINST && i_init_end != NULLINST);

    /* link in any rv initialization right after I_INIT.
     */
	    /* put the rv list right after I_INIT. */
		ic_unstow(i_init,&rv);
	    /* reinitialize for next time. */
		rv.il_head.i_next = NULLINST;
		rv.il_tail = &rv.il_head;

    /* link in any (implicit) sends, including quantifier implied loops,
     * to create processes right before the I_INIT_END.
     */
	    /* put the pr list right before I_INIT_END. */
		ic_unstow(i_init_end->i_back,&pr);
	    /* reinitialize for next time. */
		pr.il_head.i_next = NULLINST;
		pr.il_tail = &pr.il_head;

    /* don't bother generating code if there were fatal errors;
     * there might be internal inconsistencies.
     */
    if (fatal_err_cnt == 0) {
	cgen();
    }
    
    /* re-initialize. */
        i_init = i_init_end = NULLINST;

    /* free up the list of intermediate code and reinitialize */
	ic.il_head.i_next = NULLINST;
	ic.il_tail = &ic.il_head;

    /* compile and assemble the C code */
    /* (in the case of errors this will simply delete the .c and .h files */
    backend();
}



/* allocate and fill in an intermediate code instruction.
 * note: link fields are set to null.
 */
static Instptr 
icreate(tag,exp,lab)
Icode tag;
Nodeptr exp;
int lab;
{
    register Instptr t;
    t = (Instptr)alloc(sizeof(Inst));

    t->i_line = line_number;
    t->i_type = tag;
    t->i_exp = exp;
    t->i_lab = lab;
    t->i_next = t->i_back = NULLINST;
    return (t);
}



/* routines that deal with intermediate code lists.
 * they use current list as set in emit.
 * they aren't most flexible routines, but they are simpler
 * than the old ic_patch kludgery.
 */

/* ic_mark()
 * just return a mark to last Inst emitted.
 * mark is used later for stowing some code later;
 * one usage is to mark, generate some code, and then stow.
 * a second usage is to mark and then unstow,
 * which has the effect of unstowing to end of current list.
 * (yes, this routine is a candidate to be macroized.)
 */
Instptr
ic_mark()
{
    assert (cl->il_tail != &cl->il_head);
    return (cl->il_tail);
}

/* ic_stow(mark,stowed)
 * removes from mark->i_next to end of list from current list.
 * returns that list as stowed.
 * can stow an empty list.
 * (note: pass address of stowed.
 * Would it be better to have stowed return such as value?)
 */
void
ic_stow(mark, stowed)
Instptr mark;
Instlistptr stowed;
{
    assert (mark!=NULLINST && stowed!=NULLINSTLISTPTR);

    if (stowed->il_head.i_next = mark->i_next)
	stowed->il_tail = cl->il_tail;
    else
	stowed->il_tail = &stowed->il_head;

    cl->il_tail = mark;
}

/* ic_unstow(mark,stowed)
 * adds previously stowed list (which is not on current list!)
 * immediately after mark on current list,
 * updating current list's tail if necessary.
 * can unstow an empty list.
 */
static void
ic_unstow(mark, stowed)
Instptr mark;
Instlistptr stowed;
{
    Instptr yfirst, ylast;
    assert (mark!=NULLINST && stowed!=NULLINSTLISTPTR);

    if (yfirst = stowed->il_head.i_next) {
	ylast = stowed->il_tail;
	yfirst->i_back = mark;
	ylast->i_next = mark->i_next;
	if (mark->i_next)
	    mark->i_next->i_back = ylast;
	else
	    cl->il_tail = ylast;
	mark->i_next = yfirst;
    }
}

/* ic_append(stowed)
 * adds previously stowed list (which is not on current list!)
 * immediately after tail of current list.
 * just an abbreviation for a mark and a unstow.
 */
void
ic_append(stowed)
Instlistptr stowed;
{
    ic_unstow (ic_mark(), stowed);
}

/* ic_move(ic_from, ic_to)
 * moves list on current list from ic_from through tail, to ic_to.
 * just an abbreviation for a stow and a unstow.
 */
void
ic_move(ic_from,ic_to)
Instptr ic_from, ic_to;
{
    Instlist temp;
    ic_stow(ic_from, &temp);
    ic_unstow(ic_to, &temp);
}



/* generate C code from intermediate */
static void
cgen()
{
    register Instptr i;
    int arm_number = 0;		/* for co invocations. */
    Bool init_done = FALSE;
    int jt_sp = 0;
    final_done = FALSE;
    cur_proc = 0;
    assign_offset();
    cur_proc = 0;
    alloc_block_ind = 0;
    truncname(comp_name, NAMESIZE-2);
    
    for (i = ic.il_head.i_next; i != NULLINST; i = i->i_next) {
	line_number = i->i_line;
	switch (i->i_type) {
	case I_COMPONENT: 	/* ie, a resource */
	    cginit(comp_name);
	    break;
	case I_COMPONENT_END:
	    if (! init_done) {
		boom("no initial code");
		/*NOTREACHED*/
	    }
	    break;

	case I_PROC:
	    if (++ cur_proc == MAX_PROC)
		boom("too many procs");
	    gen_entry_code('P',i->i_exp->e_s->s_name);
	    break;
	case I_PROC_END:
	    gen_free_mem();
	    /* call finished_proc only if called from RTS; then return */
	    cprintf("if(!wc)sr_finished_proc();\n");
	    gen_exit_code(max_offset[cur_proc]);
	    break;
	case I_INIT:
	    if (++ cur_proc == MAX_PROC)
		boom("too many procs");
	    gen_entry_code('I',comp_name);
	    gen_init_prequel();
	    break;
	case I_INIT_END:
	    gen_free_mem();			/* free allocated memory */
	    cprintf("sr_finished_init();\n");	/* tell RTS we're done */
	    gen_exit_code(max_offset[cur_proc]);
	    init_done = TRUE;	/* so we won't generate default init */
	    break;

	case I_FINAL:
	    if (++ cur_proc == MAX_PROC)
		boom("too many procs");
	    gen_entry_code('F',comp_name);
	    break;
	case I_FINAL_END:
	    gen_free_mem();
	    cprintf("sr_finished_final();\n");
	    gen_exit_code(max_offset[cur_proc]);
	    final_done = TRUE;    /* so we won't generate default final */
	    break;
	case I_INP_ALL:
	case I_INP_ALL_END:
	case I_INP_START:
	case I_PROCESS_SEND:
	case I_PROCESS_SEND_END:
	    /* just markers. */
	    break;
	case I_EXPR:
	    if (dbflags['G'])
		pnode(i->i_exp);
	    gen_expr_code(i->i_exp, arm_number);
	    break;

	case I_BRANCH:
	    cprintf("goto %g;\n",i->i_lab);
	    break;
	case I_BRANCH_TRUE:
	    gen_logical(i->i_exp, i->i_lab, 0);
	    break;
	case I_BRANCH_FALSE:
	    gen_logical(i->i_exp, 0, i->i_lab);
	    break;
	    
	case I_LABEL:
	    wlab(i->i_lab);
	    break;
	case I_LOOPTOP:
	    cprintf("if(--sr_rem_loops<=0)sr_cswitch();\n");
	    break;
	    
	case I_REPLY:
	    gen_reply(i->i_exp->e_s);
	    break;
	
	case I_CO_START:
	    if (++jt_sp == MAX_NEST)
		boom("co statement nested too deeply");
	    gen_co_start();
	    break;
	case I_CO_END:
	    gen_co_end(i->i_lab);
	    -- jt_sp;
	    break;
	
	case I_CO_ARM:
	    arm_number = i->i_lab;
	    break;
	case I_CO_PPC_END:
	    gen_ppc_end(i->i_lab);
	    break;
	case I_JT:		/* number of arms */
	    gen_jt();
	    break;
	case I_JT_LAB:	/* a label */
	    add_jt(i->i_lab);
	    break;
	case I_JT_END:
	    end_jt();
	    break;
	case I_CO_WAIT:
	    gen_co_wait(i->i_lab);
	    break;
	
	case I_BLOCK:
	    assert(i->i_exp->e_op == TK_BLOCK);
	    if (i->i_exp->e_s->s_type == T_INPUT) {
		i->i_exp->e_s->s_offset = inoffset();
	    }
	    gen_block_prequel(i->i_exp->e_s);
	    break;
	
	case I_DO_DECLARE: {
	    Symptr sym = i->i_exp->e_s;
	    if (sym->s_kind == K_TYPE)
		build_type_desc(sym);
	    else
		build_desc(sym,
		   (sym->s_segment==S_RESOURCE) ? Use_mem_alloc : Use_stack,
		   OFF_UNK);
	    } break;
	
	case I_IMPORT:
	    init_const(i->i_exp->e_s);
	    break;
	case I_BLOCK_END:
	    gen_block_postquel();
	    break;
	
	case I_INP_ACCESS:
	    gen_input_begin();
	    gen_access(i->i_exp->e_left.e_class);
	    break;
        case I_INP_ELSEACCESS:
            gen_input_begin();
            gen_elseaccess(i->i_exp->e_left.e_class);
            break;

	case I_INP_GET_INV:
	    gen_get_inv();
	    break;
	case I_INP_GET_NAMED_INV:
	    gen_get_named_inv(i->i_exp);
	    break;
	case I_INP_GET_NAMED_INV_NB:
	    gen_get_named_inv_nb(i->i_lab);
	    break;
	case I_INP_RECEIVE:
	    gen_input_begin();
	    gen_receive(i->i_exp->e_left.e_class);
	    break;

	case I_INP_SEM_P:
	    gen_sem_p(i->i_exp);
	    break;
	
	case I_INP_MATCH:
	    gen_match(i->i_exp, i->i_lab);
	    break;

	case I_INP_ELSEMATCH:
	    gen_elsematch(i->i_lab);
	    break;
	case I_INP_REMOVE:
	    gen_remove_inv();
	    break;
	case I_INP_SET_MIN:
	    gen_set_min(i->i_exp);
	    break;
	case I_INP_UPDATE_MIN:
	    gen_update_min(i->i_exp);
	    break;
	case I_INP_REMOVE_MIN:
	    gen_remove_min();
	    break;
	case I_INP_DONE:
	    gen_input_done();
	    break;
	case I_INP_END:
	    gen_input_end();
	    break;
	case I_STOP:
	    cprintf("sr_stop(%e);\n",i->i_exp);
	    break;
	default:
	    boom("illegal I_ tag in cgen");
	    break;
	}
    }

    /* find all declared externals; generate code to for them */
    /* they should all be declared at the resource level */
    {
	Symptr sym;
	Local_blockptr spec_bl;

	/* first search the spec; i.e., exported externals */
	for (spec_bl = lb_top;
	     spec_bl && spec_bl->l_st->s_type != T_SPEC;
	     spec_bl = spec_bl->l_next)
	    ;
	assert(spec_bl);
	for (sym = spec_bl->l_st->s_tdef->s_next; sym; sym=sym->s_next) {
	    if (sym->s_kind == K_OP && sym->s_impl == IM_EXTERNAL)
		gen_ext(sym);
	}

	/* now non-exported (but still global). */
	for (sym = spec_bl->l_st->s_next; sym; sym=sym->s_next) {
	    if (sym->s_kind == K_OP && sym->s_impl == IM_EXTERNAL)
		gen_ext(sym);
	}
    }
}




/*  gen_entry_code(type,name) -- gen proc entry; type is 'I'|'P'|'F' */

void
gen_entry_code(type,name)
char type, *name;
{
    sprintf(func_name,"%c%s",type,name);
    cprintf("\n\n");
    if (type == 'P')
	cprintf("static ");
    cprintf("%s(rp,rv,pb,wc)\n",func_name);
    cprintf("register char *rp,*rv,*pb;\n");
    cprintf("int wc;\n");
    cprintf("{\n");
    cprintf("char lv[L%s];\n",func_name);
    cprintf("char *memlist=0;\n");
}



/*  gen_free_mem() - generate call to return talloc'd space, if any */

void
gen_free_mem()
{
    if (did_talloc) {			/* if we called talloc earlier */
	cprintf ("{char *t; ");
	cprintf ("while (t=memlist) {memlist=A(memlist); sr_free(t);}}\n");
	did_talloc = FALSE;
    }
}
 
 
 
/*  gen_exit_code(size) - generate end of procedure*/ 
 
void
gen_exit_code(size)   
{ 
    cprintf("}\n");
    cprintf("%h#define L%s %d\n", func_name, size ? size : INTSIZE);
}




static void
gen_expr_code(e, arm_number)
Nodeptr e;
{
    char c;

    switch (e->e_op) {
    case TK_CALL:
    case TK_SEND:
	assert(e->e_l->e_op == TK_INVOKE);
	gen_invoke(e, NO_RESULT, NOLAB);
	cprintf(";\n");
	break;
    case TK_CO_CALL:
    case TK_CO_SEND:
	assert(e->e_l->e_op == TK_INVOKE);
	gen_invoke(e, NO_RESULT, arm_number);
	cprintf(";\n");
	break;
    case TK_CO_CALL_COPY_BACK:
    case TK_CO_SEND_COPY_BACK:
	cprintf("0");
	gen_copy_back(e, NO_RESULT, TRUE);
	cprintf(";\n");
	break;
    case TK_CONST:
    case TK_ASSIGN:
	if (e->e_r->e_op == TK_CO_CALL)
	    gen_invoke(e->e_r, NO_RESULT, arm_number);
	else
	    gen_assign(e->e_l,e->e_r);
	cprintf(";\n");
	break;
    case TK_SWAP:
	gen_swap(e->e_l,e->e_r);
	cprintf(";\n");
	break;
    case TK_INCREMENT:
    case TK_DECREMENT:
	c = (e->e_op == TK_INCREMENT) ? '+' : '-';
	if (!is_lvalue(e->e_l))
	    FATAL("operator ++/-- only applies to lvalues");
	else if (is_a_const(e->e_l))
	    FATAL("can't increment/decrement a constant");
	else if (e->e_sig->s_type == T_PTR)
	    cprintf("%e%c=%*;\n",e->e_l,c,e->e_sig->s_tdef);
	else
	    cprintf("%c%c%e;\n",c,c,e->e_l);
        break;
    case TK_DESTROY:
	if (e->e_l->e_sig->s_tdef->s_kind == K_VM)
	    cprintf("sr_destvm(I(%a));\n",e->e_l);
	else
	    cprintf("sr_destroy(%a);\n",e->e_l);
	break;
    default:
	boom("unexpected node type at top of stmt");
	break;
    }
    free_all_blocks();
}


/* make note of block to be freed later */
void
free_later(n)
{
    if (alloc_block_ind == MAX_ALLOC)
	boom("too many invocations in an expression");
    alloc_block[alloc_block_ind++] = n;
}


/* free all currently-allocated invocation blocks */
static void
free_all_blocks()
{
    int j;
    for (j=0; j<alloc_block_ind; ++j)
	cprintf("sr_gen_free(LA(%d));\n",alloc_block[j]);
    alloc_block_ind = 0;
}

/* if name is longer than len, truncate it */
void
truncname(name, len)
char *name;
int len;
{
    char msg[80];
    if (strlen(name) > len) {
	 sprintf(msg, "name truncated to %d characters", len);
	WARN(msg);
	name[len] = '\0';
    }
}
