/* Note: error recovery here is minimal;
 * typically, just return NULLNODE.
 * Also, might leave some unclaimed nodes lying around if erroneous
 * user program.
 */
#include <stdio.h>
#include "sr.h"
#include "funcs.h"
#include "globals.h"


/* denotation.
 * returns pointer to constructed denotation, otherwise NULLNODE if bogus.
 * how_invoked tells how this denotation is being invoked
 * if it turns out to be an invocation, in which case
 * we build the appropriate kind of node
 * (e.g., TK_CALL, TK_CO_CALL, etc.) above the TK_INVOKE node.
 */

Nodeptr
denotation(how_invoked)
Token how_invoked;
{
    Nodeptr arglist, tmpnode, oldret;
    union e_lu sym,    /* sym.e_sym is the first identifier in the denotation;
			* this is used at the end of the routine.
			* so other identifier mustn't clobber this.  */
    ret;
    Bool first;	       /* TRUE iff doing first object name. */

    Bool cast;  /* TRUE iff doing a cast. */
    	/* allowed to cast into:
	 *    builtin type (e.g., int(red)),
	 *    user defined enumeration type (e.g., foo(3)),
	 *    user defined enumeration type in another component
	 *		(e.g., scuz.foo(3)).
	 * record constructor is treated as a cast.
	 *
	 * note: also set when type id appears by itself; e.g., low(color).
	 * such usage is the only possible denotation for which type id
	 * is legal.
	 */

    assert (how_invoked == TK_CALL || how_invoked == TK_SEND
	   || how_invoked == TK_CO_CALL || how_invoked == TK_CO_SEND);

    first = TRUE;
    cast = FALSE;
    ret.e_node = NULLNODE;
    do {
	get_token();

	switch (tok) {
	case TK_IDENTIFIER:
	    break;
	case TK_BOOL: case TK_CHAR: case TK_INT:
	    if (first) break;
	    /* fall through */
	default:
	   ERROR(E_WARN+6,"invalid denotation");
	   putback();
	   return (ret.e_node);
	}

	if (first) {
	    int i;

	    if (tok == TK_IDENTIFIER) {
		if (!(sym.e_sym = st_unqual_lookup(tk_str,1))) {
		    return (NULLNODE);
		}

		ret.e_node = NULLNODE;
		if (sym.e_sym->s_kind == K_CONST && sym.e_sym->s_value) {
		    if (is_constant(sym.e_sym->s_value, &i)) {
			if (sym.e_sym->s_type == T_BOOL) {
			    union e_lu temp;
			    temp.e_int = i;
			    sym.e_sym->s_used = TRUE;
			    return make_node(TK_BOOLEAN, temp, NULLNODE);
			} else
			    ret.e_node = numnode(i);
		    }
		    else if (sym.e_sym->s_value->e_sig->s_type == T_NULL)
			ret.e_node = bnode(TK_NULL, NULLNODE, NULLNODE);
		    else if (sym.e_sym->s_value->e_sig->s_type == T_NOOP)
			ret.e_node = bnode(TK_NOOP, NULLNODE, NULLNODE);
		} else
		    if((sym.e_sym->s_type == T_ENUM ||
			sym.e_sym->s_type == T_REC) &&
		       (sym.e_sym->s_kind==K_PREDEF ||
			sym.e_sym->s_kind==K_TYPE))    {
		    cast = TRUE;
		    ret = sym;
		    goto outcast;
		}

		if (ret.e_node == NULLNODE)
		    ret.e_node = idnode(sym.e_sym);
	    } else {
		assert(tok==TK_BOOL||tok==TK_CHAR||tok==TK_INT);
		cast = TRUE;
		/* make a symbol table like entry. */
		ret.e_sym = new_symbol(K_ANON);
		ret.e_sym->s_type = (tok == TK_BOOL ? T_BOOL :
				     (tok == TK_CHAR ? T_CHAR : T_INT));
		ret.e_sym->s_size = 1;
		ret.e_sym->s_tdef = NULLSYM;
		goto outcast;
	    }
	} else {
				/* field in a record,
				 * op in a resource cap,
				 * enum in import,
				 * constant in import.
				 */
	    Symptr type_def;
            oldret = ret.e_node;
            assert (oldret->e_sig != NULLSYM);
            switch (oldret->e_sig->s_type) {
	    case T_REC: /* field in record. */
		type_def = oldret->e_sig->s_tdef->s_tdef;
		assert(type_def!=NULLSYM);
		if (!(ret.e_sym = at_lookup(type_def,tk_str))) {
		    ERROR(E_WARN+1, "no such field");
		    return (oldret);
		}
		ret.e_node = idnode(ret.e_sym);
		break;

	    case T_CAP: /* op in resource capability. */
		type_def = oldret->e_sig->s_tdef;
		/* if there was a problem earlier; e.g., bad declaration. */
		if (type_def==NULLSYM) {
		    FATAL("trying to qualify a badly declared cap");
		    return (oldret);
		}
		if (type_def->s_type!=T_SPEC ||
		      (type_def->s_kind != K_BLOCK && 
		       type_def->s_kind!=K_IMPORT))   {
		    FATAL("not a qualifiable cap");
		    return (oldret);
		}
		if (!(ret.e_sym = at_lookup(type_def->s_tdef,tk_str))) {
		    ERROR(E_WARN+1, "no such operation");
		    return (oldret);
		}
		ret.e_node = idnode(ret.e_sym);
		break;

	    case T_SPEC: /* enum or constant in import. */
	    case T_GLOBAL:
		/* just being careful... */
		assert (oldret->e_op == TK_IDENTIFIER);

		type_def = oldret->e_s;
		assert(type_def!=NULLSYM);
		if (!(ret.e_sym = at_lookup(type_def->s_tdef,tk_str))) {
		    ERROR(E_WARN+1, "no such object in spec");
		    return (oldret);
		}
		if (ret.e_sym->s_kind == K_CONST) {
		    int i;

		    sym.e_sym = ret.e_sym;
		    ret.e_node = NULLNODE;
		    if (sym.e_sym->s_kind == K_CONST && sym.e_sym->s_value) {
			if (is_constant(sym.e_sym->s_value, &i))
			    ret.e_node = numnode(i);
			else if (sym.e_sym->s_value->e_sig->s_type == T_NULL)
			    ret.e_node = bnode(TK_NULL, NULLNODE, NULLNODE);
			else if (sym.e_sym->s_value->e_sig->s_type == T_NOOP)
			    ret.e_node = bnode(TK_NOOP, NULLNODE, NULLNODE);
		    }

		    if (ret.e_node == NULLNODE)
                        return (snode(TK_IMPORTED_CONST,sym.e_sym,NULLNODE));
		    else
			return ret.e_node;
		} else if ((ret.e_sym->s_kind == K_TYPE
			      || ret.e_sym->s_kind == K_PREDEF)
			  && (ret.e_sym->s_type == T_ENUM
			      || ret.e_sym->s_type == T_REC)) {
		    cast = TRUE;
		    goto outcast;
		} else if (ret.e_sym->s_kind == K_LITERAL &&
			   ret.e_sym->s_type == T_ENUM)      {
		    ret.e_node = idnode(ret.e_sym);
		    return ret.e_node;
		} else {
		    ERROR(E_FATAL+1,
			   "id after 'component_id .' is not a const or enum");
		    return (oldret);
		}

	    default:
		FATAL("not a qualifiable object");
		return (oldret);
	    }
	}

	if (first) {
	    first = FALSE;
	} else {
	    /* make the TK_PERIOD node before TK_INDEX or TK_HAT
	     * so that we build trees correctly for denotations like
	     * a.b[2] where var a: rec(b[1:3]:int).
	     */
	    ret.e_node = bnode(TK_PERIOD,oldret,ret.e_node);
	}

	if (maybe(TK_LEFTBKET))
	    ret.e_node = make_node(TK_INDEX,ret,indices());

	while (maybe(TK_HAT))
	    ret.e_node = make_node(TK_HAT,ret,NULLNODE);

    } while (maybe(TK_PERIOD));

    /* branch here when have seen cast above.
     * avoids doing nonsense like int[3:4] above.
     */
outcast:

    if (maybe(TK_LEFTPAREN)) {
	arglist = farg();
	if (cast)  {
	    tmpnode = make_node(TK_CAST, ret, arglist);
	    if (ret.e_sym->s_type != T_REC)  {
		assert(IS_ORDERED(ret.e_sym->s_type));
		if (arglist==NULLNODE || arglist->e_r != NULLNODE) {
		    FATAL("bad argument list for type conversion");
		    return (numnode (1));	/* error recovery */
		}
	    }
	    return (tmpnode);
	}
	ret.e_node = make_node(TK_INVOKE, ret, arglist);
	/* set used|invoked for the first identifier in the denotation.
	 */
	if (sym.e_sym->s_kind == K_OP) {
            /* operation was invoked; promote s_invoked.
	     * at this point, don't care exactly how invoked.
	     */
	    Restrict r_how_invoked;
	    if (how_invoked == TK_SEND || how_invoked == TK_CO_SEND)
		r_how_invoked = R_SEND;
	    else
		r_how_invoked = R_CALL;
	    promote_invoked(sym.e_sym,r_how_invoked);
	} else {
            /* invocation through capability; direct or part of record. */
	    sym.e_sym->s_used = TRUE;
	}
	/* build the appropriate node above the TK_INVOKE node. */
	assert (ret.e_node->e_op == TK_INVOKE);
	return (bnode(how_invoked,ret.e_node,NULLNODE));
    } else {
	if (cast) {
	    /* this allows, e.g., low(color), or low(int) but not low(rec) */
	    if (IS_ORDERED(ret.e_sym->s_type)) {
		return (idnode(ret.e_sym));
	    } else {
		FATAL("bad record constructor; missing '('");
		return (NULLNODE);
	    }
	}
	sym.e_sym->s_used = TRUE;
    }
    
    return (ret.e_node);   
}

/* returns a TK_LIST list node for the ranges.  */
Nodeptr
indices()
{
   Nodeptr n1,n2;

   n1 = subscr();

   if (maybe(TK_COMMA)) {
      n2 = subscr();
      if (n1->e_r != NULLNODE && n2->e_r != NULLNODE)
	      FATAL("multiple slices not allowed");
      n2 = bnode(TK_LIST,n2,NULLNODE);
   }
   else {
      n2 = NULLNODE;
   }
   mustbe(TK_RIGHTBKET, "right bracket in slice");
   return (bnode(TK_LIST,n1,n2));
}

/* promote_invoked(sym,r_how_invoked)
 * promote how the sym operation is invoked.
 */
void
promote_invoked(sym,r_how_invoked)
Symptr sym;
Restrict r_how_invoked;
{

	assert (sym->s_kind == K_OP);
	assert (r_how_invoked == R_CALL || r_how_invoked == R_SEND
		|| r_how_invoked == R_CALLSEND);

	switch (sym->s_invoked) {
	    case R_NOTARESTRICT:
		sym->s_invoked = r_how_invoked;
		break;
	    case R_CALL:
	    case R_SEND:
		if (sym->s_invoked != r_how_invoked)
		    sym->s_invoked = R_CALLSEND;
		break;
	    case R_CALLSEND:
		/* nothing. */
		break;
	    default:
		boom("bad R_ for s_invoked in promote_invoked");
	}
}
