/*  decl.c -- Types and Declarations.  */

#include <stdio.h>

#include "sr.h"
#include "funcs.h"
#include "globals.h"

static Symptr variab_name();
static Nodeptr sub_expr();
static Type type_def();
static Restrict op_restrictions();

static void enum_def(), rec_def(), cap_def(), ptr_def(), string_def();
static void op_spec(), comp_cap(), const_val(), cap_qual(), op_restrict();
static void result_spec(), subscripts(), subs_bounds(), check_2();


/* macro for generating I_DO_DECLARE if parsing_body. */
#define GEN_DECL(sym) {if(parsing_body) emit(I_DO_DECLARE, idnode(sym), NOLAB);}


/* auxiliary table for current namespace, if any */
static Symptr aux_table = NULLSYM;


/* follow sets for parsing */
static Token fl_enum_lit[] = {TK_COMMA, TK_RIGHTPAREN, TK_NOTATOKEN};
static Token fl_variab_name[] = {TK_COMMA, TK_COLON, TK_NOTATOKEN};
extern Token fl_declr_stmt[];	/* defined in globals.h */
extern Token fl_def_stmt[];



				/* 2. Types and Declarations */
/* type - fills in s_type and optionally s_tdef fields for num_syms
	symbols starting at start_sym
*/
static void
filltype(start_sym, num_syms, any_ok)
Symptr start_sym;
int num_syms;
Bool any_ok;			/* true if type `any' is allowed */
{	
    int i;
    Symptr p, anon_sym;
    
    get_token();
    switch (tok) {
    case TK_BOOL:
	start_sym->s_type = T_BOOL;
	break;
    case TK_CHAR:
	start_sym->s_type = T_CHAR;
	break;
    case TK_FILE:
	start_sym->s_type = T_FILE;
	break;
    case TK_INT:
	start_sym->s_type = T_INT;
	break;
    case TK_ANY:
	start_sym->s_type = T_ANY;
	if (! any_ok)
	    FATAL("type `any' not allowed here");
	break;
    case TK_PTR:
    case TK_ENUM:
    case TK_REC:
    case TK_UNION:
    case TK_CAP:
    case TK_STRING:
	putback();
	anon_sym = new_symbol(K_ANON);
	start_sym->s_type = type_def(anon_sym);
	start_sym->s_tdef = anon_sym;
	break;
    default:
	putback();
	mustbe(TK_IDENTIFIER,"identifier");
	if ((p = st_unqual_lookup(tk_str,2)) == NULLSYM) {
	    return;
	}
	if (p->s_kind == K_BLOCK || p->s_kind == K_IMPORT) {
	    mustbe(TK_PERIOD,".");
	    mustbe(TK_IDENTIFIER,"type identifier");
	    if ((p = at_lookup(p->s_tdef, tk_str)) == NULLSYM) {
		ERROR(E_FATAL+2,NULLSTR);
		return;
	    }
	}
	if (p->s_kind == K_TYPE) {
	    if (start_sym->s_kind == K_VAR || start_sym->s_kind == K_CONST
		|| start_sym->s_kind == K_PARAM 
		|| start_sym->s_kind == K_FIELD
		|| start_sym->s_kind == K_RESULT
		|| start_sym->s_kind == K_ANON) {
	        start_sym->s_type = p->s_type;
		start_sym->s_tdef = p;
	    }
	    else {
	        start_sym->s_type = p->s_type;
	        start_sym->s_tdef = p->s_tdef;
	    }

	    start_sym->s_type = p->s_type;
	} else
	    ERROR(E_FATAL+2,NULLSTR);
    }

    for (p = start_sym, i = num_syms-1; i-- > 0;) {
	p = p->s_next;
	p->s_type = start_sym->s_type;
	p->s_tdef = start_sym->s_tdef;
    }
}


				/* 2.2 Type Declarations */
void
type_declr()
{
	Symptr sym;
	Restrict type_restrict;

	mustbe(TK_IDENTIFIER,"type_identifier");
	if  (!(sym = main_install(tk_str,K_TYPE))) {
		find_follow(fl_declr_stmt);
		return;
	}
	mustbe(TK_EQ,"=");
	if ((sym->s_type = type_def(sym)) == T_NOTATYPE) {
		ERROR(E_FATAL+4, "type_declr");
		return;
	}

	type_restrict = R_PUBLIC;
	if (maybe(TK_LEFTBCE)) {
		mustbe(TK_IDENTIFIER,"type restriction");
		if (strcmp(tk_str,"private") == 0)
			type_restrict = R_PRIVATE;
		else if (strcmp(tk_str,"public") == 0)
			/* skip */;
		else {
			ERROR(E_FATAL+6,"type_restriction");
			find_follow(fl_declr_stmt);
		}
		mustbe(TK_RIGHTBCE,"type restriction");
	}
	sym->s_restrict = type_restrict;
	GEN_DECL(sym);
}

/* type_def -- given that sym is name (possibly anonymous) of user-defined 
	type, return symbolic type, update sym->s_tdef and sym->s_type 
	to reflect parsed type
*/
static Type
type_def(sym)
Symptr sym;
{
    get_token();
    switch (tok) {
    case TK_ENUM:
	sym->s_type = T_ENUM;
	enum_def(sym);
	return (T_ENUM);
    case TK_REC:
	sym->s_type = T_REC;
	rec_def(sym);
	return (T_REC);
    case TK_UNION:
	sym->s_type = T_UNION;
	rec_def(sym);
	return (T_UNION);
    case TK_CAP:
	sym->s_type = T_CAP;
	cap_def(sym);
	return (T_CAP);
    case TK_PTR:
	sym->s_type = T_PTR;
	ptr_def(sym);
	return (T_PTR);
    case TK_STRING:
	sym->s_type = T_STRING;
	string_def(sym);
	return (T_STRING);
    default:
	putback();
	ERROR(E_FATAL+6,"type_definition");
	find_follow(fl_def_stmt);
	return T_NOTATYPE;
    }
}

static void
enum_def(sym)
Symptr sym;
{
	int num_values;
	Symptr lit_sym;

	mustbe(TK_LEFTPAREN,"(");
	/* enumeration type points to itself to make signature compares
	 * easy when denotation is a cast into enum type.
	 */
	sym->s_tdef = sym;
	num_values = 0;
	do {
		mustbe(TK_IDENTIFIER,"enumeration_identifier");

				/* need to check for name collisions */
/* unique throughout entire program, including imports? */
/*  unique here refers to the fact that groups of enumerated literals should not
	be partially hidden from sight.  i.e.  after the declaration
		type color: enum (red, green, blue)
	we wouldn't want to permit the declaration of a variable named green.
		right?
	so the checking for name conflicts should actually be done elsewhere,
		if we decide to do it at all.
*/
		if (!(lit_sym = main_install(tk_str,K_LITERAL))) {
			find_follow(fl_enum_lit);
			return;
		}
		lit_sym->s_type = T_ENUM;
		lit_sym->s_tdef = sym;
		lit_sym->s_offset = num_values;
		num_values++;
	} while (maybe(TK_COMMA));
	sym->s_size = num_values;
	mustbe(TK_RIGHTPAREN,")");
}

  
static void
rec_def(sym)
Symptr sym;
{
	Symptr old_table;

	mustbe(TK_LEFTPAREN,"(");

	/* need to create a new namespace for field names within the record */
	old_table = aux_table;
	aux_table = sym->s_tdef = new_symbol(K_TYPE);
	do {
		var_decl(K_FIELD);
	} while (semi_sep());
	putback();
	aux_table = old_table;
	mustbe(TK_RIGHTPAREN,")");
}


/* a string type mimics the symbol table structure of a rec:  string(n)
   looks like  rec(len:int; str[n]:char; pad:char)  except that
   `T_REC' is replaced by `T_STRING'.  This allows a lot of the
   size and offset functions to treat them the same; obviously
   a lot of other functions must treat them differently.   
*/
static void
string_def(sym)
Symptr sym;
{
    Symptr old_table, s;
    Nodeptr maxlen, subs;
    
    mustbe(TK_LEFTPAREN, "(");
    old_table = aux_table;
    aux_table = sym->s_tdef = new_symbol(K_TYPE);
    s = at_install(aux_table, "len", K_FIELD);
    s->s_type = T_INT;
    s = at_install(aux_table, "str", K_FIELD);
    s->s_type = T_CHAR;
    if (maybe(TK_STAR))
	maxlen = bnode(TK_ARB, NULLNODE, NULLNODE);
    else 
	maxlen = int_expr();

    subs = bnode(TK_RANGE, numnode(1), maxlen);
    subs_bounds(subs, sym->s_kind);
    add_range(s, subs, NULLNODE);
    s = at_install(aux_table, "pad", K_FIELD);
    s->s_type = T_CHAR;
    mustbe(TK_RIGHTPAREN, ")");
    
    aux_table = old_table;
}


/* handle capability definitions, setting up sym's tdef accordingly.
 * in case of error, pretend that cap is for comp_sym so
 * that later routines (e.g., make_sig, get_ob_size) don't fail.
 */
static void
cap_def(sym)
Symptr sym;
{
	Symptr s;
	
	get_token();
	if (tok == TK_LEFTPAREN) {
		putback();
		s = new_symbol(K_OP);
		/* null name indicates that this is an anonymous op spec
		 * and size and offset stuff must be done.
		 * And, of course, K_OP is important.
		 */
		op_spec(s);
		sym->s_tdef = s;
		sym->s_restrict = s->s_restrict;
	}
	else if (tok == TK_IDENTIFIER) {
		/* set s to component, operation id, optype id. */
		if ((s = st_unqual_lookup(tk_str,1)) == NULLSYM) {
			sym->s_tdef = comp_sym;
			return;
		}
		if (s->s_kind == K_OP) {
			sym->s_tdef = s;
			sym->s_restrict = s->s_restrict;
		}
		else if (s->s_kind == K_OPTYPE) {
			sym->s_tdef = s->s_tdef;
			sym->s_restrict = s->s_restrict;
		}
		else if ((s->s_kind == K_BLOCK || s->s_kind == K_IMPORT)
		       && (s->s_type == T_SPEC || s->s_type == T_GLOBAL)) {
				comp_cap(s,sym);     /* component identifier. */
		}
		else if (s->s_kind == K_VM)  {
			sym->s_tdef = s;
		}
		else {
			FATAL("wrong kind of identifier after cap");
			sym->s_tdef = comp_sym;
		}
	}
	else {
		FATAL("invalid token after cap");
		sym->s_tdef = comp_sym;
	}
}



/* comp_cap(s,sym)
 * component capabilities.
 * accepts: resource cap, resource.op, or component.optype cap?
 * s is a component; sym is where result is stored.
 */
static void
comp_cap(s,sym)
Symptr s,sym;
{
	if (maybe(TK_PERIOD))
		cap_qual(sym,s);
	else if (s->s_type == T_GLOBAL)
		FATAL("can't have resource cap for a global component");
	else {
		sym->s_tdef = s;
		sym->s_restrict = s->s_restrict;
	}
}



/* handles capability declarations of the form:
 * component_id.op_id or component_id.optype_id.
 * have already seen component_id (s points at it) and ".".
 * sym is where result is stored.
 */
static void
cap_qual(sym,s)
Symptr sym,s;
{
	Symptr t;

	get_token();
	if (tok != TK_IDENTIFIER) {
		FATAL("missing id after 'component_id.'");
		sym->s_tdef = s;
		return;
	}

	/* look up name; it should be an operation id or optype id. */
		assert (s->s_tdef != NULLSYM);
		if ((t = at_lookup(s->s_tdef,tk_str)) == NULLSYM) {
			FATAL("id after . is not defined");
			sym->s_tdef = s;
			return;
		}

		if (t->s_kind == K_OP) {
			sym->s_tdef = t;
			sym->s_restrict = t->s_restrict;
		}
		else if (t->s_kind == K_OPTYPE) {
			sym->s_tdef = t->s_tdef;
			sym->s_restrict = t->s_restrict;
		}
		else {
			FATAL("id after . is not an operation or optype id");
			sym->s_tdef = s;
			return;
		}
}

static void
ptr_def(sym)
Symptr sym;
{
	Symptr sp;

	sp = new_symbol(K_ANON);
	filltype(sp, 1, TRUE);
	sym->s_tdef = sp;
}
    

					/* 2.3 Variable Declarations */

/* parses variable and record field declarations;
 * kind tells which: K_CONST, K_VAR, or K_FIELD.
 * note: do initialization on variable definitions here for simplicity:
 * stuff used in parsing fields is used later in initialization.
 */
void
var_decl(kind)
Kind kind;
{
    int cnt;
    Symptr first, s;
    Token t;

    assert (kind == K_VAR || kind == K_CONST || kind == K_FIELD);
    do {
	cnt = 0;
	do {				/* build name list */
	    if (s = variab_name(kind))
		if (cnt++ == 0)
		    first = s;
	} while (maybe(TK_COMMA));

	if (cnt == 0) {
	    FATAL("no identifier in var/const/field definition");
	    return;
	} else if (cnt > 1 && kind == K_CONST) {
	    FATAL("multiple identifiers illegal with const");
	    return;
	}

	if (maybe(TK_COLON)) {
	    filltype(first, cnt, FALSE);	/* set types if specified */
	} else if (cnt > 1 || kind == K_FIELD) {
	    FATAL("missing type spec for id list");
	    return;
	}

	if (kind != K_FIELD) {
	    for (s = first; s; s = s->s_next)
		GEN_DECL(s);		/* declare variables & constants */

	    t = get_token();
	    if (t == TK_EQ)  {
		WARN("this isn't FORTRAN; use :=");
		t = TK_ASSIGN;
	    }
	    if (t == TK_ASSIGN) {	/* process initialization */
		if (cnt > 1)  { 
		    FATAL("initialization illegal with multiple variables");
		    return;
		}
		if (kind == K_CONST)
		    const_val(first,expr());
		else
		    assign_rhs(TK_ASSIGN,idnode(first));
	    } else {
		putback();
		if (kind == K_CONST)
		    FATAL("missing constant value");
		else if (first->s_type == T_NOTYETATYPE) {
		    FATAL("missing type specification");
		    return;
		}
	    }
	}
    } while (maybe(TK_COMMA));
}



/* set value of a constant */

static void
const_val(s,e)
Symptr s;
Nodeptr e;
{
    Nodeptr n;
    int i;

    if (!e)
	return;
    n = idnode(s);
    if (!check_assign(n,e)) {
	FATAL("incompatible constant type");
	return;
    }
    s->s_value = e;

    /* generate code only in body.  code generator handles constants in specs
       and elsewhere. */
    if (parsing_body && !is_constant(e,&i)
	    && e->e_sig->s_type != T_NULL && e->e_sig->s_type != T_NOOP)
	emit(I_EXPR,bnode(TK_CONST,n,e),NOLAB);

}




/* installs var/const/param names, including return name and record fields */

static Symptr
variab_name(kind)
Kind kind;
{
	Symptr sym;

	in_variab_name = TRUE;
	mustbe(TK_IDENTIFIER,"identifier");
	in_variab_name = FALSE;

	/* for a record or operation, install fields/params) in aux table */
	if (aux_table)
		sym = at_install(aux_table, tk_str, kind);
	else
		sym = main_install(tk_str, kind);
	if (!sym)  {
		find_follow(fl_variab_name);
		return NULLSYM;
	}

	sym->s_type = T_NOTYETATYPE;	/* make unusable until type is set */

	if (maybe(TK_LEFTBKET)) {
		subscripts(kind,sym);
		mustbe(TK_RIGHTBKET,"subscripts");
	}
	return sym;
}


				/* 2.4 Operation Declarations */
void
op_declr(optok)
Token optok;	/* TK_OP, TK_EXTERNAL, or TK_SEM */
{
    Symptr first, sym;
    int cnt, n;

    do {
	assert(!aux_table);  /* otherwise variab_name installs in wrong place */
	cnt = 0;
	do {
	    if (sym = variab_name(K_OP)) {
		if (cnt++ == 0)
		    first = sym;
	    } else {
		find_follow(fl_declr_stmt);
		return;
	    }
	} while (maybe(TK_COMMA));

	if (optok != TK_SEM)
	    maybe(TK_COLON);	/* allow optional colon here */

	if (optok == TK_EXTERNAL)
	    for (sym = first, n = cnt; n--; sym = sym->s_next) {
		sym->s_impl = IM_EXTERNAL;
		sym->s_reply = FALSE;
		}

	if (optok == TK_SEM) 
	    for (sym = first, n = cnt; n--; sym = sym->s_next) {
		sym->s_tdef = NULLSYM;
		sym->s_type = T_VOID;
		sym->s_segment = S_NOTASEGMENT;
		sym->s_restrict = R_SEND;
		if (parsing_body)
		    new_class(sym);
	    }
	else if (maybe(TK_IDENTIFIER)) {
	    Symptr id_sym;
	    if ((id_sym=st_unqual_lookup(tk_str,1)) == NULLSYM) {
		ERROR(E_WARN+1,"in op declaration");
		return;
	    }
	    if (id_sym->s_type == T_SPEC || id_sym->s_type == T_GLOBAL) {
		if (get_token() != TK_PERIOD) {
		    WARN("missing . in spec/global optype");
		    return;
		}
		if (get_token() != TK_IDENTIFIER){
		    WARN("missing id in resource optype");
		    return;
		}
		if ((id_sym=at_lookup(id_sym->s_tdef,tk_str)) == NULLSYM) {
		   ERROR(E_WARN+1,"in op declaration");
		   return;
	       }
	    }
	    if (id_sym->s_kind != K_OPTYPE) {
		WARN("id not a optype in op declaration");
		return;
	    }
	    /* Just copy info from the optype entry. */
	    assert (id_sym->s_tdef != NULLSYM);
	    /* chase that tdef! */
	    id_sym = id_sym->s_tdef;

	    assert (id_sym->s_type == T_FUNC || id_sym->s_type == T_VOID);
	    for (sym = first, n = cnt; n--; sym = sym->s_next) {
		sym->s_type = id_sym->s_type;
		sym->s_tdef = id_sym->s_tdef;
		sym->s_restrict = id_sym->s_restrict;
		if (parsing_body)
		    new_class(sym);
	    }
	}
	else if (maybe(TK_LEFTPAREN)) {
	    putback();
	    if (cnt > 1) {
		FATAL("multiple names in op declaration");
		cnt = 1;
	    }
	    op_spec(first);
	    if (parsing_body)
		new_class(first);
	} else
	    FATAL("bad op declaration");
	for (sym = first, n = cnt; n--; sym = sym->s_next)
	    GEN_DECL(sym);
    } while (maybe(TK_COMMA));
}

void
optype_declr()
{
	Symptr sym;

	if (get_token() != TK_IDENTIFIER) {
		WARN("missing id after optype");
		return;
	}
	if  (!(sym = main_install(tk_str,K_OPTYPE))) {
		find_follow(fl_declr_stmt);
		return;
	}
	maybe(TK_EQ);
	sym->s_tdef = new_symbol(K_OP);
	op_spec(sym->s_tdef);
	sym->s_restrict = sym->s_tdef->s_restrict;
	GEN_DECL(sym);
}

static void
op_spec(sym)
Symptr sym;
{
	Symptr old_table;

	sym->s_tdef = NULLSYM;
	old_table = aux_table;
	mustbe(TK_LEFTPAREN,"(");
	if (!maybe(TK_RIGHTPAREN)) {
		aux_table = sym->s_tdef = new_symbol(K_NOTAKIND);
				/* create a new name space */
		do {
			param_spec(sym->s_impl==IM_EXTERNAL?TK_EXTERNAL:TK_OP);
		} while (semi_sep());
		putback();
		mustbe(TK_RIGHTPAREN,")");
	}
	if (maybe(TK_RETURNS)) {
		/* create a new table if didn't do it above. */
		if (sym->s_tdef == NULLSYM)
			aux_table = sym->s_tdef = new_symbol(K_NOTAKIND);
		result_spec();
		sym->s_type = T_FUNC;
	}
	else {
		sym->s_type = T_VOID;
	}

	aux_table = old_table;
	sym->s_segment = S_NOTASEGMENT;
	op_restrict(sym);
}

void
param_spec(mytok)
Token mytok; /* tells whether doing a resource heading, op, or external. */
{
	Symptr p, first;
	Restrict restriction;
	int cnt;

	get_token();
	switch (tok)  {
	case TK_VAL:
		restriction = R_VAL;
		break;
	case TK_VAR:
		restriction = R_VALRES;
		break;
	case TK_RES:
		restriction = R_RES;
		break;
	case TK_REF:
		FATAL("`ref' is not yet implemented");
		restriction = R_VALRES;
		break;
	default:
		putback();
		restriction = R_VAL;	/*default param_kind*/
		break;
	}
	if (mytok == TK_RESOURCE && restriction != R_VAL) {
		WARN("non-val restriction on resource parameter(s) ignored");
		restriction = R_VAL;
	}

	cnt = 0;
	first = NULLSYM;
	do {
		p = variab_name(K_PARAM);
		if (p)
		{	cnt++;
			if (first == NULLSYM)
				first = p;
			p->s_restrict = restriction;
		}
	} while (maybe(TK_COMMA));
	if (first) {
		mustbe(TK_COLON, ":");
		filltype(first, cnt, FALSE);
		/* later hope to allow T_ANY with externals */
	}
}

/* parses returns part of op spec.
 * note: degenerate case of param_spec code.
 */
static void
result_spec()
{
	Symptr p;

	p = variab_name(K_RESULT);
	if (p){
		p->s_restrict = R_RES;
		mustbe(TK_COLON, ":");
		filltype(p, 1, FALSE);
	}
}


/* This code is written so that it can be shared with denotation.
 * So, the tests to ensure that * is used only in parameter declarations
 * and that the declaration has an upper and a lower bound
 * is done above the other routines.
 * In either case, a TK_RANGE node is built.
 * A missing bound is represented by a NULLNODE.
 * A star is represented by a TK_ARB node.
 * We attempt to do some error recovery here, so that routines
 * that expect valid TK_RANGE's don't blow up;
 * for example, we sometimes make up bounds,
 * or assign a bogus upper bound the value of the lower bound.
 */
static void
subscripts(kind,sym)
Kind kind;
Symptr sym;
{
	Nodeptr ndim1, ndim2;

	assert (kind == K_VAR || kind == K_CONST || kind == K_FIELD
		|| kind == K_PARAM || kind == K_RESULT || kind == K_OP);
	ndim1 = subscr();
	subs_bounds(ndim1,kind);

	if (maybe(TK_COMMA)){
		ndim2 = subscr();
		subs_bounds(ndim2,kind);
		if (kind==K_PARAM || kind==K_VAR || kind==K_CONST || kind==K_OP)
			check_2(ndim1,ndim2);
	} else {
		ndim2 = NULLNODE;
	}
	add_range(sym,ndim1,ndim2);
}


/* validate subscripts used in context of array bounds declaration.
 * add lower bound of 1 if absent.
 * make sure not [*:*]
 * make sure that no *'s at all if variable declaration or proc return.
 */
static void
subs_bounds(n,kind)
Nodeptr n;
Kind kind;
{
	Nodeptr left, right;

	left = n->e_l;
	right = n->e_r;
	assert(left != NULLNODE);
	if (right == NULLNODE) {
		right = n->e_r = left;
		left = n->e_l = numnode(1);
		make_sig(n);
	} else if (kind != K_PARAM && kind != K_ANON && 
		   (left->e_op == TK_ARB || right->e_op == TK_ARB)) {
		FATAL("* in var or op declaration or proc return");
		/* error recovery */
		if (left->e_op == TK_ARB)  n->e_l = numnode(1);
		if (right->e_op == TK_ARB) n->e_r = numnode(10);
		make_sig(n);
	} else if (left->e_op == TK_ARB && right->e_op == TK_ARB) {
		FATAL("*:* not allowed in declaration");
		release_node(left);
		left = n->e_l = numnode(1);	/* error recovery */
		make_sig(n);
	}
}

/* for declarations of parameters,
 * make sure that we don't get, e.g., a[1:*,3:*]
 * for error recovery, if we do, we get rid of the second star.
 * note: this can't happen for variables because of what we did below.
 */
static void
check_2(n1,n2)
Nodeptr n1, n2;
{
	Nodeptr left1, left2, right1, right2;
	int l1,r1,l2,r2;

	left1 = n1->e_l;
	right1 = n1->e_r;
	left2 = n2->e_l;
	right2 = n2->e_r;
	assert(left1 != NULLNODE && right1 != NULLNODE);
	assert(left2 != NULLNODE && right2 != NULLNODE);

	l1 = (left1->e_op == TK_ARB);
	r1 = (right1->e_op == TK_ARB);
	l2 = (left2->e_op == TK_ARB);
	r2 = (right2->e_op == TK_ARB);

	assert (l1 + r1 + l2 + r2 <= 2);
	if (l1 + r1 + l2 + r2 == 2) {
		WARN("* in both bounds not allowed");
		/* error recovery: change the second bound. */
			if (l2) {
				release_node(left2);
				n2->e_l = copy_nodes(n2->e_r);
			}
			else if (r2) {
				release_node(right2);
				n2->e_r = copy_nodes(n2->e_l);
			}
			else {
				boom("check_2");
				/*NOTREACHED*/
			}
	}
}


/* handles one dimension; i.e., a:b or just a */
Nodeptr
subscr()
{
	Nodeptr lbound, ubound;

	lbound = sub_expr();
	if (maybe(TK_COLON))
		ubound = sub_expr();
	else
		ubound = NULLNODE;
	return (bnode(TK_RANGE,lbound,ubound));
}

/* handles * or expression.
 */
static Nodeptr
sub_expr()
{
	Nodeptr n;
	if (maybe(TK_STAR))
		return (bnode(TK_ARB,NULLNODE,NULLNODE));

	if ((n = ot_expr()) == NULLNODE)
		n = numnode(1);		/* error recovery */
	return (n);
}

static void
op_restrict(sym)
Symptr sym;
{		
	Restrict restrict;
		
	if (maybe(TK_LEFTBCE))
		restrict = op_restrictions();
	else
		restrict = R_CALLSEND;
	sym->s_restrict = restrict;
}

/* op_restrictions
 * parses operation restrictions plus closing }.
 * returns restriction.
 * separate routine only because wouldn't fit in above indenting scheme. yawn.
 */
static Restrict
op_restrictions()
{
	Restrict restrict;

	restrict = R_NOTARESTRICT;
	do {
		switch (get_token()) {
			case TK_CALL:
				switch (restrict) {
					case R_NOTARESTRICT:
					    restrict = R_CALL;
					    break;
					case R_CALL:
					case R_CALLSEND:
					    WARN("duplicate call restrictor");
					    break;
					case R_SEND:
					    restrict = R_CALLSEND;
					    break;
					default:
					    boom("bad op restriction");
					    /*NOTREACHED*/
				}
				break;
			case TK_SEND:
				switch (restrict) {
					case R_NOTARESTRICT:
					    restrict = R_SEND;
					    break;
					case R_SEND:
					case R_CALLSEND:
					    WARN("duplicate send restrictor");
					    break;
					case R_CALL:
					    restrict = R_CALLSEND;
					    break;
					default:
					    boom("bad op restriction");
					    /*NOTREACHED*/
				}
				break;
			default:
				ERROR(E_FATAL+6,"op_restriction");
				break;
			}
		} while (maybe(TK_COMMA));
		mustbe(TK_RIGHTBCE, "}");

		/* error recovery:
		 * return default (R_CALL) if no valid restrictor seen above. */
		return ((restrict==R_NOTARESTRICT)? R_CALLSEND:restrict);
}

/* semi_sep()
 * called when parsing declarations --
 * such as fields of records, or parameters of operations or resources --
 * that can have semicolon optionally terminating constituent parts.
 * returns TRUE iff didn't find the right parenthesis that ends declaration.
 * n.b. the semicolon is a terminator, so that can write, e.g.,
 * 	op f(x:int; y:char;)
 * we correctly handle such a case.
 */
Bool
semi_sep()
{
	Token mytok;

	mytok = get_token();
	if (mytok == TK_SEMICOLON) {
		mytok = get_token();
	}
	else if (mytok == TK_COMMA) {
		/* common mistake; skip it. */
		WARN("expected semicolon (or space) instead of comma");
		mytok = get_token();
	}

	putback();
	return ((Bool) (mytok != TK_RIGHTPAREN));
}
