/*  cprintf.c -- generate C code */

#include <ctype.h>
#include <stdio.h>
#include <varargs.h>
#include "../srsys.h"
#include "sr.h"
#include "funcs.h"
#include "globals.h"

static int inoff();
static void evalue(), deref(), eaddr(), iaddr(), saddr();
static void descaddr(), descfield(), esize(), elength(), emaxl();
static void sbsize(), nelem(), classaddr(), putconst();
static void strarg(), catargs(), dbound(), offadd();


struct state {			/* the state of an output file */
    FILE *fp;			/* file pointer */
    char lc;			/* last char output */
    int pc;			/* parentheses count */
};

static struct state cstate;	/* state of the .c file */
static struct state hstate;	/* state of the .h file */

static struct state *out;	/* pointer to current output state */
static char prevch;		/* previous char in cprintf -- see putconst */

static int strnum;		/* string constant number */


/* use these macros for quick output without overhead of cprintf.
   remember, however, they don't count {}() or handle special cases. */
#define PUTS(s) fputs(s,out->fp)
#define PUTC(c) putc(c,out->fp)
#define PRINT1(t,v) fprintf(out->fp,t,v)
#define PRINT2(t,v,w) fprintf(out->fp,t,v,w)

/* definitions for use in offadd() */
#define RP_OFFSET -1
#define RV_OFFSET -2
#define PB_OFFSET -3




/*  cginit(res) - initialize for resource generation  */

void
cginit(res)
char *res;
{
    chopen(res);		/* open files */
    cstate.fp = cfile;		/* set up pointers */
    hstate.fp = hfile;
    out = &cstate;		/* set .c file for default output */
    out->pc = -1;		/* ignore first { for indenting */
    strnum = 0;			/* init count of string constants */
}



/*  cprintf(template,p1,p2,...) - write formatted string to the C file
 *
 *  The template is similar in flavor to printf(3S).
 *  "argtype" gives the expected type of the corresponding "pn" argument.
 *  "result" gives the type of the generated C expression, if applicable.
 *
 *   field   argtype	result	meaning
 *   -----   -------	------	---------------------------------------------
 *	%a   Nodeptr	char*	address of addressable expression
 *	%A   Symptr	char*	address based on symptr
 *	%c   int		char argument
 *	%C   Classptr	char*	class address
 *	%d   int	int	integer argument
 *	%D   Symptr	char*	base address of a descriptor
 *	%e   Nodeptr	int	value of expression 
 *	%g   int		goto label
 *	%h			switches to .h file instead of .c file
 *	%l   Nodeptr	int	length of string expression
 *	%L   Nodeptr	int	maximum length of string variable
 *	%n   Nodeptr	int	number of elements in an expression
 *	%r   Symptr		'A' or 'I' or '*' to dereference following addr
 *	%s   char *		string argument
 *	%S   Nodeptr	char*,int  string expression as two RTS params
 *	%z   Nodeptr	int	size of addressable expression
 *	%Z   Symptr	int	base size of an array, given the signature
 *	%[   Symptr	int	lower bound 1 of array
 *	%]   Symptr	int	upper bound 1 of array
 *	%{   Symptr	int	lower bound 2 of array
 *	%}   Symptr	int	upper bound 2 of array
 *	%*   Symptr	int	size (from descriptor if necessary)
 *	%&   Symptr	char*	address (from descriptor if necessary)
 *	%#   Symptr	int	maximum length of element of string
 *	%%			percent sign
 *	%other			error
 *
 *   Some special cases simplify code generation:
 *   (1)  An initial ',' following '(' is suppressed for easy list generation.
 *   (2)  When "%d" follows '=', and the value is negative, a space is inserted.
 *   Both cases apply only to characters that are part of a template.
 */


/* VARARGS */
void
cprintf(va_alist)
va_dcl
{
    va_list ap;
    register char *s;
    char c;
    int n;
    struct state *oldout;

    if (fatal_err_cnt > 0)	/* exit immediately if fatal errors */
	return;
    oldout = out;	/* save output file in case changed by %h */
    va_start(ap);
    s = va_arg(ap, char *);
    if (*s == ',' && out->lc == '(')
	s++;
    while (c = *s++)  {
	prevch = out->lc;
	out->lc = c;
	switch (c) {
	    case '(': 
	    case '{':
		out->pc++;
		PUTC(c);
		break;
	    case ')': 
	    case '}':
		out->pc--;
		PUTC(c);
		break;
	    case '\n':
		PUTC('\n');
		for (n = out->pc; n > 0; n--)
		    PUTC(' ');
		break;
	    case '%':
		switch (c = *s++)  {
		case 'a':  eaddr(va_arg(ap,Nodeptr));			break;
		case 'A':  saddr(va_arg(ap,Symptr));			break;
		case 'c':  PUTC(va_arg(ap,int));			break;
		case 'C':  classaddr(va_arg(ap,Classptr));		break;
		case 'd':  putconst(va_arg(ap,int));			break;
		case 'D':  descaddr(va_arg(ap,Symptr));			break;
		case 'e':  evalue(va_arg(ap,Nodeptr));			break;
		case 'g':  PRINT1("z%d",va_arg(ap,int));		break;
		case 'h':  out = &hstate;				break;
		case 'l':  elength(va_arg(ap,Nodeptr));			break;
		case 'L':  emaxl(va_arg(ap,Nodeptr));			break;
		case 'n':  nelem(va_arg(ap,Nodeptr));			break;
		case 'r':  deref(va_arg(ap,Symptr));			break;
		case 's':  PUTS(va_arg(ap,char *));			break;
		case 'S':  strarg(va_arg(ap,Nodeptr));			break;
		case 'z':  esize(va_arg(ap,Nodeptr));			break;
		case 'Z':  sbsize(va_arg(ap,Symptr));			break;
		case '[':  descfield(va_arg(ap,Symptr),AD_LB1);		break;
		case ']':  descfield(va_arg(ap,Symptr),AD_UB1);		break;
		case '{':  descfield(va_arg(ap,Symptr),AD_LB2);		break;
		case '}':  descfield(va_arg(ap,Symptr),AD_UB2);		break;
		case '*':  descfield(va_arg(ap,Symptr),AD_SIZE);	break;
		case '&':  descfield(va_arg(ap,Symptr),AD_ADDR);	break;
		case '#':  descfield(va_arg(ap,Symptr),AD_MAXL);	break;
		case '%':  PUTC('%');					break;
		default:   boom(s-2);
		}
		break;
	    default:
		PUTC(c);
		break;
	    }
	}
    va_end(ap);
    out = oldout;	/* restore previous output file */
}



/*  wlab(n) - write a label  */

void
wlab(n)
int n;
{
    cprintf("%g:",n);
}



/*  wescape(f,s) - write string s to file f with unprintables escaped */

void
wescape(f,s)
FILE *f;
Strptr s;
{
    char c, *p;
    int n;
    
    putc('"',f);
    p = s->str_data;
    for (n = s->str_len;  n > 0;  n--)  {
	c = *p++;
	if (isprint(c) && (c != '"') && (c != '\\'))
	    putc(c,f);
	else {
	    putc('\\',f);
	    switch (c) {
		case '\b':  putc('b',f);  break;
		case '\f':  putc('f',f);  break;
		case '\n':  putc('n',f);  break;
		case '\r':  putc('r',f);  break;
		case '\t':  putc('t',f);  break;
		case '"':   putc('"',f);  break;
		case '\\':  putc('\\',f); break;
		default:    fprintf(f, "%03o", c & 0377);  break;
	    }
	}
    }
    putc('"',f);
}




/*********************  internal functions  *************************/



/*  evalue(e) - generate C code for computing the value of e
 *
 *  note that this may be used as an lvalue in some circumstances
 */

static void
evalue(e)
Nodeptr e;
{
    unsigned char c;
    Symptr base, s;

    assert(e);
    switch (e->e_op)  {
	case TK_IDENTIFIER:
	    s = e->e_s;
	    if (s->s_kind == K_LITERAL) {
		putconst(s->s_offset);
	    } else if (s->s_size == INTSIZE &&
			s->s_kind == K_VAR &&
			s->s_offset != OFF_UNK)  {
		    /* generate terser code for resource & proc vars */
		    switch (s->s_segment)  {
			case S_RESOURCE:
			    if (s->s_type == T_PTR)
				PRINT1("RA(%d)",s->s_offset);
			    else
				PRINT1("RE(%d)",s->s_offset);
			    break;
			case S_PROC:
			case S_INPUT:
			    if (s->s_type == T_PTR)
				PRINT1("LA(%d)",s->s_offset);
			    else
				PRINT1("LO(%d)",s->s_offset);
			    break;
			default:
			    boom("bad segment for int in evalue");
		    }
	    } else {
		deref(e->e_sig);
		saddr(s);
	    }
	    break;
	case TK_PERIOD:
	case TK_INDEX:
	case TK_ALOCAL:
	    deref(e->e_sig);
	    PUTC('(');
	    eaddr(e);
	    PUTC(')');
	    break;
	case TK_CO_CALL_COPY_BACK:
	case TK_CO_SEND_COPY_BACK:
	    PUTC(out->lc='(');
	    gen_copy_back(e, VAL_RESULT, TRUE);
	    PUTC(')');
	    break;
	case TK_FORMAL:
	    deref(e->e_sig);
	    offadd(e->e_r->e_i,e->e_s);
	    break;
	case TK_FILE_NULL:
	    putconst(NULL_FILE);
	    break;
	case TK_FILE_NOOP:
	    putconst(NOOP_FILE);
	    break;
	case TK_PTR_NOOP:
	    FATAL("noop cannot be assigned to a pointer");
	    break;
	case TK_PTR_NULL:
	    PUTC('0');
	    break;
	case TK_FILE_CONST:
	    switch (e->e_left.e_file) {
		case F_STDIN:	PUTS("(char*)stdin");	break;
		case F_STDOUT:	PUTS("(char*)stdout");	break;
		case F_STDERR:	PUTS("(char*)stderr");	break;
		default:	boom("bad file const");
	    }
	    out->lc = 0;
	    break;
	case TK_NEW:
	    if (e->e_r) {
	        assert(e->e_s->s_type == T_STRING);
		cprintf("sr_new(R(%e+%d))",e->e_r,STR_OVH);
	    } else
		cprintf("sr_new(%*)",e->e_s);
	    break;
	case TK_HAT:
	    deref(e->e_sig);
	    cprintf("(%e)",e->e_l);
	    break;
	case TK_ADDR:
	    if (!is_lvalue(e->e_l))
		ERROR(E_FATAL+4,"@ can only be applied to lvalues");
	    else
		cprintf("%a",e->e_l);
	    break;
	case TK_NUMBER:
	case TK_BOOLEAN:
	    putconst(e->e_i);
	    break;
	case TK_CHRLIT:
	    c = e->e_i;
	    if (!isprint(c))
		putconst(c);
	    else if (c == '\\' || c == '\'')
		PRINT1("'\\%c'",c);
	    else
		PRINT1("'%c'",c);
	    break;
	case TK_SUBSTR:
	    assert(e->e_sig->s_size == 1);
	    cprintf(" *%a",e);
	    break;
	case TK_CAST:
	    assert(e->e_r->e_op == TK_LIST);
	    switch (e->e_s->s_type) {  /* type to cast into */
		case T_INT:
		case T_ENUM:
		    if (e->e_r->e_l->e_sig->s_type == T_CHAR)
			cprintf("(unsigned char)");
		    cprintf("%e",e->e_r->e_l);
		    break;
		case T_CHAR:
		    cprintf("(%e&0xFF)",e->e_r->e_l);
		    break;
		case T_BOOL:
		    cprintf("(%e!=0)",e->e_r->e_l);
		    break;
		default:
		    FATAL("illegal cast");
		    break;
	    }
	    break;

	case TK_UMINUS:
	    cprintf(" -%e",e->e_l);	/* extra space avoids "=-" */
	    break;

	case TK_PLUS:
	    if (e->e_sig->s_type == T_PTR)
		if (e->e_l->e_sig->s_type == T_PTR)
		    cprintf("(%e+(%**%e))",e->e_l,e->e_sig->s_tdef,e->e_r);
		else
		    cprintf("(%e+(%**%e))",e->e_r,e->e_sig->s_tdef,e->e_l);
	    else
		cprintf("(%e+%e)",e->e_l,e->e_r);
	    break;

	case TK_MINUS:
	    if (e->e_sig->s_type == T_PTR)
		cprintf("(%e-%**%e)",e->e_l,e->e_sig->s_tdef,e->e_r);
	    else if (e->e_l->e_sig->s_type == T_PTR)
		if (e->e_l->e_sig->s_tdef->s_type == T_ANY)
		  cprintf("((%e-%e)/(%*))",e->e_l,e->e_r,e->e_r->e_sig->s_tdef);
		else
		  cprintf("((%e-%e)/(%*))",e->e_l,e->e_r,e->e_l->e_sig->s_tdef);
	    else
		cprintf("(%e-%e)",e->e_l,e->e_r);
	    break;

	/*  binary operators; the extra paren for division avoids generating
	 *  '/' followed by '*', which is taken as a comment  */
	case TK_STAR:	cprintf("(%e*%e)",e->e_l,e->e_r);   break;
	case TK_DIV:	cprintf("(%e/(%e))",e->e_l,e->e_r); break;
	case TK_MOD:	cprintf("(%e%%%e)",e->e_l,e->e_r);  break;
	case TK_LSHIFT:	cprintf("(%e<<%e)",e->e_l,e->e_r);  break;
	case TK_RSHIFT:	cprintf("(%e>>%e)",e->e_l,e->e_r);  break;

	case TK_EQ:
	case TK_NE:
	case TK_GT:
	case TK_GE:
	case TK_LT:
	case TK_LE:
	case TK_AND:
	case TK_OR:
	case TK_XOR:
	case TK_NOT:
	    gen_logical(e, 0, 0);
	    break;

	case TK_LB1:    descfield(e->e_s, AD_LB1);  break;
	case TK_UB1:    descfield(e->e_s, AD_UB1);  break;
	case TK_LB2:    descfield(e->e_s, AD_LB2);  break;
	case TK_UB2:    descfield(e->e_s, AD_UB2);  break;
	case TK_LENGTH: cprintf("I(%a)", e->e_l);   break;

	case TK_RUNTIMESIZE:
	    cprintf("((%]-%[+1",e->e_s,e->e_s);
	    if (e->e_s->s_ranges->r_dim2)
		cprintf(")*(%}-%{+1",e->e_s,e->e_s);
	    cprintf("))");
	    break;

	case TK_QUESTION:
	    switch (e->e_l->e_op) {
		case TK_IDENTIFIER:  base = e->e_l->e_s;       break;
		case TK_INDEX:       base = e->e_l->e_l->e_s;  break;
		default:             boom("bad TK_QUESTION op");
		}
	    if (base->s_impl == IM_INPUT)
		cprintf("sr_query_iop((opcap *)&%e)",e->e_l);
	    else if (base->s_impl == IM_SEMAPHORE)
		cprintf("sr_query_sem(%e)",e->e_l);
	    else {
		WARN("procs never have pending invocations");
		PUTC('0');
	    }
	    break;

	case TK_CALL:
	    PUTC('(');
	    gen_invoke(e, VAL_RESULT, NOLAB);
	    PUTC(')');
	    break;

	case TK_VECTORIZE:
	case TK_CLONE:
	    FATAL("vector or clone in illegal context");
	    return;
	default:
	    boom("bad token in evalue");
	    break;
    }
}



/*  deref(s) - generate type-dependent deref of address that will follow  */

static void
deref(s)
Symptr s;
{
    switch (s->s_type)  {
	case T_INT:
	case T_ENUM:
	case T_VOID:
	case T_FUNC:
	    PUTC('I');
	    break;
	case T_CHAR:
	case T_BOOL:
	    PUTS(" *");
	    break;
	case T_PTR:
	case T_CAP:
	case T_FILE:
	    PUTC('A');
	    break;
	default:
	    boom("bad type in deref");
    }
}



/*  eaddr(e) - generate C code for computing the address of e  */

static void
eaddr(e)
Nodeptr e;
{
    Symptr base;

    switch (e->e_op)  {
	case TK_STRLIT:
	    fprintf(hfile,"string(s%d,",strnum);
	    wescape(hfile,e->e_str);
	    fprintf(hfile,");\n");
	    PRINT1("(char*)&s%d",strnum++);
	    break;
	case TK_IDENTIFIER:
	case TK_IMPORTED_CONST:
	    saddr(e->e_s);
	    break;
	case TK_HAT:
	    cprintf("%e",e->e_l);
	    break;
	case TK_NULL:
	case TK_OPCAP_NULL:
	    PUTS("&sr_nu_ocap");
	    break;
	case TK_NOOP:
	case TK_OPCAP_NOOP:
	    PUTS("&sr_no_ocap");
	    break;
	case TK_RESCAP_NULL:
	    PUTS("&sr_nu_rcap");
	    break;
	case TK_RESCAP_NOOP:
	    PUTS("&sr_no_rcap");
	    break;
	case TK_VMCAP_NULL:
	    PUTS("&sr_nu_vmcap");
	    break;
	case TK_VMCAP_NOOP:
	    PUTS("&sr_no_vmcap");
	    break;
	case TK_FILE_CONST:
	    switch (e->e_left.e_file) {
		case F_STDIN:	PUTS("&sr_stdin");   break;
		case F_STDOUT:	PUTS("&sr_stdout");  break;
		case F_STDERR:	PUTS("&sr_stderr");  break;
		default:	boom("bad file const");
	    }
	    out->lc = 0;
	    break;
	case TK_INDEX:
	    iaddr(e);
	    break;
	case TK_CALL:
	    assert(e->e_l->e_op == TK_INVOKE);
	    PUTC('(');
	    gen_invoke(e, ADDR_RESULT, NOLAB);
	    PUTC(')');
	    break;
	case TK_CO_CALL_COPY_BACK:
	case TK_CO_SEND_COPY_BACK:
	    PUTC(out->lc='(');
	    gen_copy_back(e, ADDR_RESULT, TRUE);
	    PUTC(')');
	    break;
	case TK_FORMAL:
	    offadd(e->e_r->e_i,e->e_s);
	    break;
	case TK_ALOCAL:
	    PRINT1("LA(%d)",e->e_i);
	    break;
	case TK_SUBSTR:
	    assert(e->e_l->e_op == TK_IDENTIFIER || e->e_l->e_op == TK_PERIOD);
	    cprintf("(%a",e->e_l);
	    if (e->e_l->e_s->s_type == T_STRING)
		PRINT1("+%d",INTSIZE);
	    cprintf("+%e-1)",e->e_r->e_l->e_l);
	    break;
	case TK_PERIOD:		/* record and capability references */
	    assert(e->e_r->e_op = TK_IDENTIFIER);
	    base = e->e_r->e_s;
	    if (base->s_offset != OFF_UNK)
		cprintf("(%a+%d)",e->e_l,base->s_offset);
	    else
		/* sure looks bogus, but %A actually gets an offset here */
		cprintf("(%a+(int)%A)",e->e_l,base);
	    break;
	case TK_CONCAT:
	    PUTS("sr_cat(");
	    catargs(e);
	    PUTS("0,0)");
	    break;
	default:
	    boom("bad op in eaddr");
	    break;
    }
}



/*  iaddr(e) - generate address of an index espression  */

static void
iaddr(e)
Nodeptr e;
{
    Symptr base;

    if (e->e_l->e_op == TK_IDENTIFIER)
	base = e->e_l->e_s;
    else {
	assert(e->e_l->e_op == TK_PERIOD);
	base = e->e_l->e_r->e_s;
    }

    PUTC('(');				/* separate call to balance parens */
    cprintf("%a+",e->e_l);		/* base address */
    if (get_ob_size(base) != 1)
	cprintf("%Z*",base);		/* element size */

    if (e->e_r->e_r) 		  	/* if 2 dimensional */
	PUTC('(');

    if (e->e_r->e_l->e_l->e_op == TK_ARB)
	PUTC('0');
    else
	cprintf("(%e-%[)",e->e_r->e_l->e_l,base);  /* dim1 offset */

    if (e->e_r->e_r) {			/* if 2 dimensional */
	cprintf("*(%}-%{+1)",base,base);
	if (e->e_r->e_r->e_l->e_l->e_op != TK_ARB)
	    cprintf("+%e-%{",e->e_r->e_r->e_l->e_l,base);
	PUTC(')');
    }

    PUTC(')');
}




/*  saddr(sym) - generate parenthesized addr of item based on symbol table  */

static void
saddr(s)
Symptr s;
{
    switch (s->s_segment)  {
	case S_RESOURCE:
	    switch (s->s_kind) {
		case K_VAR:
		case K_FIELD:
		case K_CONST:
		case K_OP:
		case K_SEMAPHORE:
		    offadd(RV_OFFSET,s);
		    break;
		case K_PARAM:
		case K_RESULT:
		    offadd(RP_OFFSET,s);
		    break;
		default:
		    boom("bad resource kind in saddr");
		    break;
	    }
	    break;
	case S_PROC:
	    switch (s->s_kind) {
		case K_PARAM:
		case K_RESULT:
		    offadd(PB_OFFSET,s);
		    break;
		case K_VAR:
		case K_FIELD:
		case K_CONST:
		case K_OP:
		    if (s->s_offset != OFF_UNK)
			PRINT1("(lv+%d)",s->s_offset);
		    else
			PRINT1("(LA(%d))",s->s_desoff);
		    break;
		default:
		    boom("bad proc kind in saddr");
		    break;
	    }
	    break;
	case S_INPUT:
	    switch (s->s_kind) {
		case K_PARAM:
		case K_RESULT:
		    if (s->s_offset != OFF_UNK)
			PRINT2("(LA(%d)+%d)",inoff(s),s->s_offset);
		    else
			cprintf("(LA(%d)+I(LA(%d)+%d))",
			    inoff(s),inoff(s),s->s_desoff);
		    break;
		case K_VAR:
		case K_FIELD:
		case K_CONST:
		case K_OP:
		    if (s->s_offset != OFF_UNK)
			PRINT1("(lv+%d)",s->s_offset);
		     else
			PRINT1("(LA(%d))",s->s_desoff);
		    break;
		default:
		    boom("bad input kind in saddr");
	    }
	    break;
	case S_NOTASEGMENT:
	    FATAL("type name used where expression required");
	    break;
	default:
	    boom("bad segment in saddr");
	    break;
    }
}



/*  generate address of a descriptor  */

static void
descaddr(s)
Symptr s;
{
    switch (s->s_segment) {
	case S_RESOURCE:
	    switch (s->s_kind) {
		case K_ANON:
		case K_TYPE:
		case K_VAR:
		case K_FIELD:
		case K_CONST:
		case K_OP:
		    PRINT1("rv+%d",s->s_desoff);
		    break;
		case K_PARAM:
		    PRINT1("rp+%d",s->s_desoff);
		    break;
		default:
		    boom("bad resource kind in descaddr");
	    }
	    break;
	case S_PROC:
	    switch (s->s_kind) {
		case K_PARAM:
		case K_RESULT:
		    PRINT1("pb+%d",s->s_desoff);
		    break;
		case K_ANON:
		case K_TYPE:
		case K_VAR:
		case K_FIELD:
		case K_CONST:
		case K_OP:
		    PRINT1("lv+%d",s->s_desoff);
		    break;
		default:
		    boom("bad proc kind in descaddr");
		    break;
	    }
	    break;
	case S_INPUT:
	    switch (s->s_kind) {
		case K_PARAM:
		case K_RESULT:
		    PRINT2("LA(%d)+%d",inoff(s),s->s_desoff);
		    break;
		case K_ANON:
		case K_TYPE:
		case K_VAR:
		case K_FIELD:
		case K_CONST:
		case K_OP:
		    PRINT1("lv+%d",s->s_desoff);
		    break;
		default:
		    boom("bad input kind in descaddr");
	    }
	    break;
	default:
	    boom("bad segment in descaddr");
	    break;
    }
}



/*  descfield(sym,field) - gen field of descriptor  */

static void
descfield(s,d)
Symptr s;
int d;
{
    switch (d) {
	case AD_LB1:
	    assert(s->s_ranges);
	    dbound(s, s->s_ranges->r_dim1->e_l, AD_LB1);
	    break;
	case AD_UB1:
	    assert(s->s_ranges);
	    dbound(s, s->s_ranges->r_dim1->e_r, AD_UB1);
	    break;
	case AD_LB2:
	    assert(s->s_ranges);
	    dbound(s, s->s_ranges->r_dim2->e_l, AD_LB2);
	    break;
	case AD_UB2:
	    assert(s->s_ranges);
	    dbound(s, s->s_ranges->r_dim2->e_r, AD_UB2);
	    break;
	case AD_SIZE:
	    if (s->s_size == SIZE_UNK ||
		(s->s_kind == K_PARAM && s->s_size == SIZE_ARB)) {
		assert(s->s_desoff != OFF_UNK);
		cprintf("I(%D+%d)",s,AD_SIZE);
	    } else {
		assert(s->s_size != SIZE_ARB);
		putconst(s->s_size);
	    }
	    break;
	case AD_ADDR:
	    descaddr(s);
	    break;
	case AD_MAXL:
	    assert(s->s_type == T_STRING);
	    assert(s->s_kind == K_PARAM || s->s_kind == K_RESULT);
	    if (s->s_size != SIZE_ARB && s->s_size != SIZE_UNK) {
		putconst(s->s_tdef->s_tdef->s_next->s_next->s_size);
	    } else {
		assert(s->s_desoff != OFF_UNK);
		cprintf("I(%D+%d)",s,AD_MAXL);
	    }
	    break;
	default:
	    boom("bad field in descfield");
	    break;
    }
}



/*  esize(e) - generate C code for computing the size of e  */

static void
esize(e)
Nodeptr e;
{   
    Nodeptr left;
    Symptr s;
    int sz;
    
    switch (e->e_op) {
	case TK_IDENTIFIER:
	case TK_CAST:
	    descfield(e->e_s, AD_SIZE);
	    return;
	case TK_FORMAL:
	    if ((sz = get_size(e)) != SIZE_UNK) {
		putconst(sz);
	    } else {
		assert(e->e_s->s_desoff != OFF_UNK);
		PRINT2("I(LA(%d)+%d)", e->e_r->e_i, e->e_s->s_desoff+AD_SIZE);
	    }
	    return;
	case TK_ADDR:
	    putconst(PTRSIZE);
	    return;
	case TK_INDEX:
	    left = e->e_l;
	    if (left->e_op == TK_PERIOD)
		left = left->e_r;
	    assert (left->e_op == TK_IDENTIFIER);
	    if (e->e_sig->s_size != SIZE_UNK)
		cprintf("(%d*%Z)", e->e_sig->s_size, left->e_s);
	    else
		cprintf("(%e*%Z)", e->e_sig->s_value, left->e_s);
	    return;
	case TK_STRLIT:
	    /* may be needed, e.g., to allocate string(*) parameter space */
	    putconst(align(e->e_str->str_len+STR_OVH));
	    return;
	case TK_CONCAT:
	    cprintf("R(%l+%l+%d)",e->e_l,e->e_r,STR_OVH);
	    return;
	case TK_VECTORIZE:
	    if (e->e_l)
		cprintf("(%z+%z)",e->e_l,e->e_r);
	    else
		esize(e->e_r);
	    return;
	case TK_CLONE:
	    cprintf("(%e*%z)",e->e_l,e->e_r);
	    return;
	case TK_OPCAP_NULL:
	case TK_OPCAP_NOOP:
	    putconst(OP_CAP_SIZE);
	    return;
	case TK_CALL:
	case TK_CO_CALL:
	    e = e->e_l;
	    /* and continue below */
	    break;
    }

    /* for all other cases we get the size from the signature */

    s = e->e_sig;
    sz = get_ob_size(s);
    assert(sz > 0);
    if (s->s_size != SIZE_UNK)
	putconst(sz * s->s_size);
    else
	cprintf("(%d*%e)", sz, s->s_value);
}



/* elength(e) - generate (current) length of string or char array */

static void
elength(e)
Nodeptr e;
{
    if (e->e_op == TK_CONCAT)
	cprintf("(%l+%l)",e->e_l,e->e_r);
    else if (e->e_sig->s_type == T_CHAR)
	esize(e);
    else if (e->e_op == TK_STRLIT)
	putconst(e->e_str->str_len);
    else {
	assert(e->e_sig->s_type==T_STRING);
	cprintf("I(%a)",e);
    }
}



/* emaxl(e) - generate maximum length of a string */

static void
emaxl(e)
Nodeptr e;
{
    Symptr s;

    if (e->e_op == TK_PERIOD)
	e = e->e_r;
    if (e->e_op == TK_INDEX)
	e = e->e_l;
    if (e->e_sig->s_type==T_CHAR || e->e_op==TK_CONCAT || e->e_op==TK_STRLIT)
	elength(e);
    else {
	assert(e->e_sig->s_type == T_STRING);
	assert(e->e_op == TK_IDENTIFIER);
	if (e->e_s->s_kind == K_PARAM || e->e_s->s_kind == K_RESULT)
	    descfield(e->e_s, AD_MAXL);
	else {
	    s = e->e_s->s_tdef->s_tdef->s_next->s_next;
	    assert(strcmp(s->s_name,"str")==0);
	    descfield(s,AD_UB1);
	}
    }
}



/*  sbsize(s) - generate base size of an array  */

static void
sbsize(s)
Symptr s;
{
    int sz = get_ob_size(s);
    if (sz > 0)
	putconst(sz);
    else {
	assert(s->s_type == T_REC || s->s_type == T_STRING);
	assert(s->s_tdef);
	descfield(s->s_tdef,AD_SIZE);
    }
}



/*  nelem(e) - generate number of elements in e  */

static void
nelem(e)
Nodeptr e;
{
    Symptr s;

    switch (e->e_op) {
	case TK_IDENTIFIER:
	    s = e->e_s;;
	    if (s->s_ranges) {
		cprintf("((%]-%[+1",s,s);
		if (s->s_ranges->r_dim2)
		    cprintf(")*(%}-%{+1",s,s);
		cprintf("))");
	    } else
		PUTC('1');
	    break;
	case TK_PERIOD:
	    nelem(e->e_r);
	    break;
	case TK_CONCAT:
	    elength(e);
	    break;
	case TK_CALL:
	    e = e->e_l;
	    /* FALL THROUGH */
	case TK_INVOKE:
	case TK_INDEX:
	case TK_VECTORIZE:
	case TK_SUBSTR:
	    s = e->e_sig;
	    if (s->s_size != SIZE_UNK)
		putconst(s->s_size);
	    else
		cprintf("%e",s->s_value);
	    break;
	default:
	    PUTC('1');
	    break;
    }
}



/*  classaddr(clp) - generate address of class pointer */

static void
classaddr(clp)
Classptr clp;
{
    switch (clp->cl_segment)  {
	case S_RESOURCE:
	    PRINT1("(rv+%d)",clp->cl_offset);
	    break;
	case S_PROC:
	    PRINT1("(lv+%d)",clp->cl_offset);
	    break;
	case S_INPUT:
	    ERROR(E_WARN+4,"class addresses");
	    break;
	default:
	    boom("bad segment in classaddr");
    }
}



/*  putconst(n) - output a constant, special casing for <0 following '='  */

static void
putconst(n)
int n;
{
    if (n < 0 && (prevch == '=' || prevch == '-'))
	PUTC(' ');		/* avoid false "=-" or "--" */
    PRINT1("%d",n);		/* output value */
}




/*  strarg(e) - generate string argument to runtime function
 *
 *  generates "<addr>,<len>"
 */

static void
strarg(e)
Nodeptr e;
{
    int t;

    if (e->e_op == TK_STRLIT) {		/* string constant */
	wescape(out->fp,e->e_str);
	PRINT1(",%d",e->e_str->str_len);
    } else if (e->e_sig->s_type == T_STRING)	/* other string */
	cprintf("%a+%d,I(%a)",e,INTSIZE,e);
    else if (is_addressable(e))			/* char array */
	cprintf("%a,%z",e,e);
    else {					/* char expr */
	assert(e->e_sig->s_type == T_CHAR);
	assert(e->e_sig->s_size == 1);
	t = temp_alloc(INTSIZE);
	cprintf("(*(lv+%d)=%e,lv+%d),1",t,e,t);
    }
}




/********************* second level internal functions ********************/ 



/*  catargs(e) - generate the args of concatenation from left to right  */

static void
catargs(e)
Nodeptr e;
{
     if (e->e_op == TK_CONCAT)  {
	catargs(e->e_l);
	catargs(e->e_r);
     } else {
	strarg(e);
	PUTC(',');
     }
}



/*  dbound(s,e,d) - gen descr bound e if NUMBER, else field d of descr for s */

static void
dbound(s,e,d)
Symptr s;
Nodeptr e;
int d;
{
    if (e->e_op == TK_NUMBER)
	putconst(e->e_i);
    else {
	assert(s->s_desoff != OFF_UNK);
	cprintf("I(%D+%d)",s,d);
    }
}



/* inoff(s) - find stack offset of input invocation block for s.  */

static int
inoff(s)
Symptr s;
{
    while (s && s->s_kind != K_BLOCK)
	 s = s->s_prev;
    assert(s && s->s_type == T_INPUT);
    return s->s_offset;
}




/*  offadd(base,sym) - generate parenthesized addr using offsets from symtab  */

static void
offadd(base,s)
int base;
Symptr s;
{
    if (s->s_offset != OFF_UNK)
	switch (base)  {
	    case RP_OFFSET:
		PRINT1("(rp+%d)",s->s_offset);
		break;
	    case RV_OFFSET:
		PRINT1("(rv+%d)",s->s_offset);
		break;
	    case PB_OFFSET:
		PRINT1("(pb+%d)",s->s_offset);
		break;
	    default:
		PRINT2("(LA(%d)+%d)",base,s->s_offset);
		break;
	}
    else /* offset unknown */
	switch (base)  {
	    case RP_OFFSET:
		PRINT1("(rp+I(rp+%d))",s->s_desoff);
		break;
	    case RV_OFFSET:
		PRINT1("(A(rv+%d))",s->s_desoff);
		break;
	    case PB_OFFSET:
		PRINT1("(pb+I(pb+%d))",s->s_desoff);
		break;
	    default:
		cprintf("(LA(%d)+I(LA(%d)+%d))",base,base,s->s_desoff+AD_ADDR);
		break;
	}
}
