/*
 *  Common Socket I/O Routines for SR Execution Manager & Run-time Support.
 *
 *  These routines call sr_net_abort(message) in case of trouble.
 */

#include "rts.h"
#include <errno.h>
#include <fcntl.h>

static void syserr();


extern int errno;

extern int sr_my_vm;


static int lfd;			/* listner fd */
static int mfd [1+MAX_VM];	/* machine to fd mapping */
static int fdm [FD_SETSIZE];	/* fd to machine mapping */

/*
 *  The set of input files.  We depend on one thread being able to change this
 *  up to the time select() is actually called, even if the other thread has
 *  already called sr_iowait().
 */

static fd_set waitset;			/* set of fd's to check for input*/
static int maxfd;			/* maximum fd to check */
static int currfd;			/* current fd being checked */



/*
 *  Initialize socket routines.  Create and bind a stream socket in the Internet
 *  domain.  Return socket address (as a static string).
 */
char *
sr_net_start ()
{
    char host [HOST_NAME_LEN];
    struct hostent *hp;
    struct sockaddr_in sin;
    static char myaddr[50];
    unsigned char *cp;

    /* get network address of our host */
    gethostname (host, sizeof (host));
    if ((hp = gethostbyname (host)) == NULL)
	syserr ("gethostbyname");
    if (hp->h_addrtype != AF_INET)
	syserr ("host addr type not INET");
	
    /* get socket and look for a port number */
    if ((lfd = socket (hp->h_addrtype, SOCK_STREAM, 0)) < 0)
	syserr ("socket creation");
    
    memset ((daddr) &sin, 0, sizeof (struct sockaddr_in));
    memcpy ((daddr) &sin.sin_addr, hp->h_addr, hp->h_length);
    sin.sin_family = hp->h_addrtype;
    sin.sin_port = IPPORT_RESERVED;
    
    while (bind (lfd,(struct sockaddr *) &sin,sizeof(struct sockaddr_in)) < 0) {
	if (errno != EADDRINUSE && errno != EACCES)
	    syserr ("bind");
	if (++sin.sin_port > 16383)
	    sr_net_abort ("no port available for open_socket");
    }

    if (fcntl(lfd,F_SETFD,1) == -1)	/* set close-on-exec */
	syserr ("close-on-exec");
    if (listen(lfd, MAX_VM) < 0)	/* prepare to accept conns */
	syserr ("listen");

    FD_SET(lfd,&waitset);
    currfd = maxfd = lfd;

    cp = (unsigned char *) &sin.sin_addr;
    sprintf (myaddr, "%d.%d.%d.%d.%d", cp[0], cp[1], cp[2], cp[3],
	sin.sin_port);
    DEBUG (0x40, "listen(%d)  %s", lfd, myaddr, 0);
    return (myaddr);
}



/*
 *  Is machine n known?
 */
bool
sr_net_known (n)
int n;
{
    return (mfd[n] != 0);
}



/*
 *  Connect to machine n at the given internet socket.
 */
void
sr_net_connect (n, address)
int n;
char *address;
{
    u_char *cp;
    int i, na[4], port, fd;
    struct sockaddr_in sin;

    if (sr_net_known(n))
	sr_net_abort("attempt to establish duplicate connection");

    /* construct network address in structure */
    sscanf (address, "%d.%d.%d.%d.%d", na+0, na+1, na+2, na+3, &port);
    memset ((daddr) &sin, 0, sizeof (sin));
    cp = (u_char *) &sin.sin_addr;
    for (i = 0 ; i < 4 ; i++)  
	*cp++ = (u_char) na[i];
    sin.sin_family = AF_INET;
    sin.sin_port = (u_short) port;
    
    /* create socket */
    if ((fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	syserr ("socket creation");
    
    /* connect to socket */
    DEBUG (0x40, "connect(%d) %s  (vm %d)", fd, address, n);
    if (connect (fd, (struct sockaddr *) &sin, sizeof (sin)) < 0)
	syserr ("connect");

    /* add to set of known machines and fds */
    mfd[n] = fd;
    fdm[fd] = n;
    if (fd > maxfd)
	maxfd = fd;
    if (fd > FD_SETSIZE)
	sr_net_abort("fd too big to select");
    FD_SET(fd,&waitset);
}



/*
 *  Send a packet to an already-known machine.
 *  Buffer begins with a standard packet header.
 *  Origin (from global "sr_my_vm"), type, dest, and size are added.
 */
void
sr_net_send (dest, type, ph, size)
int dest;
enum ms_type type;
pach ph;
int size;
{
    char *addr;
    int fd, n, rem;

    DEBUG (0x10, "to %d:   t%d, n=%d", dest, type, size);
    ph->origin = sr_my_vm;
    ph->dest = dest;
    ph->size = size;
    ph->type = type;
    fd = mfd[ph->dest];
    if (!fd)
	if (dest == sr_my_vm)
	    sr_net_abort("sr_net_send to self");
	else
	    sr_net_abort("sr_net_send to unknown destination");
    rem = ph->size;
    addr = (char *) ph;
    while (rem > 0) {
	n = write (fd, addr, rem);
	if (n < 0)
	    syserr("sr_net_send");
	rem -= n;
	addr += n;
    }
}



/*
 *  Read the next available packet (from anyone).
 *  Return the message type, or MSG_EOF if EOF is read.
 *
 *  The different machines are polled in round-robin fashion.
 *  New connections are accepted transparently as part of the loop.
 */
enum ms_type
sr_net_recv (ph)
pach ph;
{
    int n;
    static fd_set readyset;		/* fd's with input available */

    for (;;)	{
	/*
	 *  Continue looping from previous call; currfd is the last
	 *  fd tried.  When currfd reaches the listener socket:
	 *	-- refresh the select data
	 *	-- accept a new connection if offered
	 */
	if (++currfd > maxfd)
	    currfd = 0;

	if (currfd == lfd)  {
	    sr_iowait (&waitset, &readyset, INPUT);  /* wait for some input */

	    /* accept a new connection if one is available */
	    if (FD_ISSET(lfd,&readyset)) {
		n = accept(currfd, (struct sockaddr *) 0,
		    (int *) 0);
		DEBUG (0x40, "accept(%d) => %d", currfd, n, 0);
		if (n < 0)
		    syserr("accept");
		if (n > maxfd)
		    maxfd = n;
		if (n > FD_SETSIZE)
		    sr_net_abort("fd too big to select");
		FD_SET(n,&waitset);
	    }

	} else if (FD_ISSET(currfd,&readyset)) {
	    /* read packet */
	    n = read (currfd, (daddr) ph, PACH_SZ);
	    if (n != PACH_SZ)
		if (n == 0) {
		    /* got EOF -- fake an EOF packet */
		    DEBUG (0x20, "from %d: EOF", fdm[currfd], 0, 0);
		    close (currfd);
		    FD_CLR (currfd, &waitset);
		    ph->size = PACH_SZ;
		    ph->origin = fdm[currfd];
		    ph->dest = sr_my_vm;
		    fdm[currfd] = 0;
		    mfd[ph->origin] = 0;
		    return (ph->type = MSG_EOF);
		}
		else if (n < 0) {
		    DEBUG (0x40, "read(%d) [from %d], errno=%d",
			currfd, fdm[currfd], errno);
		    if (errno == ECONNRESET) {
			/* one of our cohorts died. ignore here --
			 * could be a shutdown in progress */
			close (currfd);
			FD_CLR (currfd, &waitset);
			continue;
		    }
		    syserr ("packet read");
		}
		else {
		    sr_net_abort ("packet truncated");
		}
	    DEBUG (0x20, "from %d: t%d, n=%d", ph->origin, ph->type, ph->size);
	    if (fdm[currfd] != ph->origin)
		if (fdm[currfd] == 0)  {
		    fdm[currfd] = ph->origin;
		    if (mfd[ph->origin] != 0)
			sr_net_abort("duplicate connection detected");
		    mfd[ph->origin] = currfd;
		} else {
		    DEBUG (0x20, "  fdm[%d]=%d; mfd[origin]=%d",
			currfd, fdm[currfd], mfd[ph->origin]);
		    sr_net_abort("misdelivered mail");
		}
	    return (ph->type);
	}
    }
}



/*
 *  Read the rest of a message for which we have only the header.
 */
void
sr_net_more (ph)
pach ph;
{
    int fd, n, rem;
    char *addr;

    rem = ph->size - PACH_SZ;
    if (rem == 0)
	return;
    else if (rem < 0)
	sr_net_abort ("sr_net_more: bad size");
    fd = mfd[ph->origin];
    if (!fd)
	sr_net_abort ("sr_net_more: unknown origin");
    addr = (char *) ph + PACH_SZ;
    while (rem > 0) {
	n = read (fd, addr, rem);
	if (n < 0)
	    syserr("sr_net_more");
	if (n == 0) 
	    sr_net_abort("EOF in mid-message");
	rem -= n;
	addr += n;
    }
}



/*
 *  Diagnose an error from a system call.
 *  Format a message and call sr_net_abort.
 */
static void
syserr (message)
char *message;
{
    char s1[100], s2[100];
    extern int errno, sys_nerr;
    extern char *sys_errlist[];

    if (errno > 0 && errno < sys_nerr)
	strcpy (s1, sys_errlist[errno]);
    else
	sprintf (s1, "error %d", errno);
    sprintf (s2, "%s: %s", message ? message : "network I/O", s1);
    sr_net_abort (s2);
}
