/*
 *  SR Run-Time Support.  Process Management Routines.
 */

#include "rts.h"

static proc proc_free;		/* process descriptor free list */

 
/*
 *  Return a process descriptor to the free list.
 */
#define free_proc(pr)	{ \
    (pr)->status = FREE; \
    (pr)->next = proc_free; \
    proc_free = (pr); \
}

static void rem_proc(), terminate();



/*
 *  Create a new process:
 *  	Allocate a new process descriptor.
 *  	Add process to list for specified resource.
 *  	Allocate stack space.
 *  	Set up stack for process initiation.
 *  	Return process descriptor.
 *  	(Do not put process on the ready list.)
 */

proc
sr_spawn (pc, res, arg1, arg2, arg3, arg4)
void (*pc)();
int arg1, arg2, arg3, arg4;
rint res;
{
    proc pr;

    if ((pr = proc_free) == NULL)
	sr_abort ("too many processes");

    proc_free = pr->next;
    pr->next = NULL;
    if ((pr->res = res) != NULL) {
	pr->procs = res->procs;
	res->procs = pr;
    }

    pr->status = INFANT;
    if (!pr->stack)
	pr->stack = sr_alloc (sr_stack_size);

    sr_build_context (pc, pr->stack, sr_stack_size, arg1, arg2, arg3, arg4);
    DEBUG (0x400, "r%06X p%06X spawn    p%06X", sr_cur_res, sr_cur_proc, pr);
    return (pr);
}



/*
 *  Place a process created by sr_spawn() the ready list.
 */
void
sr_activate (pr)
proc pr;
{

    pr->status = READY;
    sr_enqueue (& sr_ready_list, pr);
    DEBUG (0x100, "r%06X p%06X activate p%06X", sr_cur_res, sr_cur_proc, pr);
}



/*
 *  Kill an SR process:
 *  	Remove process from list for its resource.
 *  	Remove process from appropriate scheduler list.
 *  	Free process descriptor.
 *	Don't free the stack; we can probably reuse it, and if we're
 *	     killing ourself then we're still using it.
 */
void
sr_kill (pr, do_rem_proc)
proc pr;
bool do_rem_proc;
{
    if (do_rem_proc)
	rem_proc (pr);

    DEBUG (0x400, "r%06X p%06X kill     p%06X", sr_cur_res, sr_cur_proc, pr);
    switch (pr->status) {
	case ACTIVE:
	    pr->status = FREE;		/* will be freed by sr_cswitch() */
	    sr_cswitch();
	    break;

	case READY:
	    sr_dequeue (& sr_ready_list, pr);
	    free_proc (pr);
	    break;

	case BLOCKED:
	    sr_dequeue (pr->blocked_on, pr);
	    free_proc (pr);
	    sr_num_blocked--;
	    break;

	case INFANT:
	    free_proc (pr);
	    break;

	case FREE:
	    sr_abort ("can't kill a free process");
	default:
	    sr_abort ("unknown process status in kill");
    }
}




/*
 *  A context switch is requested.  Allow the next process on the ready list to
 *  execute (if there is one).
 */
void
sr_cswitch ()
{
    int n;
    proc pr;

    /* first, requeue the current process if it's still viable */
    if (sr_cur_proc->status < BLOCKED) {
	sr_cur_proc->status = READY;
	sr_enqueue (& sr_ready_list, sr_cur_proc);

	/* reevaluate blocked processes if switching due to loop timeout */
	if (sr_rem_loops <= 0)
	    sr_evcheck (FALSE);
    }

    /* otherwise, try to find something to run */
    else while (sr_ready_list.head == NULL) {

	/* nothing to run now, so wait for an event */
	n = sr_evcheck (TRUE);

	/* if no events were waiting, terminate */
	/* (but may return if there's final code) */
	if (n == 0)
	    terminate ();
    }

    /* free the current process if it's no longer needed */
    /* [deferred until now to prevent reuse of its stack] */
    if (sr_cur_proc->status == FREE)
	free_proc (sr_cur_proc);

    /* pop the ready list */
    pr = sr_ready_list.head;
    sr_ready_list.head = pr->next;
    if (pr->next == NULL) {
	sr_ready_list.tail = NULL;
    }
    pr->status = ACTIVE;
    sr_rem_loops = sr_max_loops;	/* reset the loop counter */

    /* start the process; if it's the current one that's really easy */
    DEBUG (0x800, "r%06X p%06X start    p%06X", sr_cur_res, sr_cur_proc, pr);
    if (pr == sr_cur_proc)
	return;
    sr_cur_proc = pr;
    sr_cur_res = pr->res;
    sr_chg_context (pr->stack);
}



/*
 *  Place a process descriptor at the end of the specified queue.
 */
void
sr_enqueue (pl, pr)
proc_queue *pl;
proc  pr;
{

    if ((*pl).head == NULL) {
	(*pl).head = pr;
	(*pl).tail = pr;
    }
    else {
	(*pl).tail->next = pr;        
	(*pl).tail = pr;
    }

    pr->next = NULL;
}



/*
 *  Remove a process descriptor from the specified queue;  assume it's there.
 */
void
sr_dequeue (pl, pr)
proc_queue *pl;
proc pr;
{
    proc tpp;

    if ((tpp = (*pl).head) == pr)
	(*pl).head = pr->next;
    else {
	while (tpp->next != pr)  tpp = tpp->next;
	tpp->next = pr->next;
	if (tpp->next == (*pl).tail)
	    (*pl).tail = tpp;
    }
}



/*
 *  Remove a process from the list of processes associated with the specified
 *  resource.
 */
static void
rem_proc (pr)
proc pr;
{
    proc this, last = NULL;

    P (pr->res->mutex);
    for (this = pr->res->procs ; this ; this = this->procs) {
	if (this == pr) {
	    if (last == NULL)
		pr->res->procs = this->procs;
	    else
		last->procs = this->procs;
	    
	    V (pr->res->mutex);
	    return;
	}

	last = this;
    }

    sr_abort ("process not found on resource proc list");
}



/*
 *  Initialize the process management system:
 *  	Set up the free list of process descriptors.
 *	Execute the startup code in an SR process context.
 *	DOES NOT RETURN.
 */
void
sr_init_proc (start_code)
void (*start_code)();
{
    int i;
    proc pr;

    /* make sure max_loops is strictly positive */
    if (sr_max_loops <= 0)
	sr_max_loops = 0x7FFFFFFF;	/* treat 0 as effectively infinite */

    /* set up the free list */
    pr = (proc) sr_alloc (sr_max_processes * sizeof (struct proc_st));
    for (i = 1 ; i < sr_max_processes ; i++) {
	pr[i].next = &pr[i+1];
	pr[i].status = FREE;
	pr[i].stack = NULL;
    }
    pr[sr_max_processes-1].next = NULL;
    proc_free = &pr[1];

    /* make an SR process and execute the startup code */
    pr[0].stack = sr_alloc (sr_stack_size);
    pr[0].status = ACTIVE;
    pr[0].next = NULL;
    pr[0].res = NULL;
    sr_cur_proc = &pr[0];
    sr_cur_res = NULL;

    sr_build_context (start_code, pr[0].stack, sr_stack_size, 0, 0, 0, 0);
    sr_chg_context (pr[0].stack);
}



/*
 *  An SR program has reached the terminated condition.  Make sure there are no
 *  blocked processes (and hence that the program has not deadlocked).
 */
static void
terminate ()
{
    proc pr;
    int code = 0;
    static bool final_done = FALSE;

    /* Execute the main routine's finalization code, if any.  */

    if (sr_rpat[0].final != NULL && sr_main_res.res != NULL && !final_done
    && sr_main_res.seqn == sr_main_res.res->seqn++) {
	final_done = TRUE;		/* only do this once */
	DEBUG (8, "executing final code for main resource", 0, 0, 0);
	pr = sr_spawn (sr_rpat[0].final, sr_main_res.res,
	    sr_main_res.res->crb_addr, sr_main_res.res->rv_base, 0, RTS_OWN);
	pr->wait = sr_make_sem (0);
	pr->ptype = FINAL;
	sr_activate (pr);
	return;
    }

    if (sr_num_blocked != 0) {
	/* check for over-consumption of user objects */

	if (sr_rid_count->value == 0)
	    rts_error ("too many active resources");
	else if (sr_class_count->value == 0)
	    rts_error ("too many input operation classes");
	else if (sr_cob_avail->value == 0)
	    rts_error ("too many active co statements");
	else if (sr_rem_count->value == 0)
	    rts_error ("too many outstanding remote requests");
	else if (sr_oper_free == NULL)
	    rts_error ("too many active operations");
	else {
	    char mesg [100];
	    sprintf (mesg, "%d blocked process%s during termination",
			sr_num_blocked, (sr_num_blocked > 1) ? "es" : "");
	    rts_warn (mesg);
	    }
	code = 1;
    }
    sr_stop (code);
}
