/*
 *  SR Run-time Support.
 *  Miscellaneous routines supporting the generated code.
 */

#include <varargs.h>
#include "rts.h"



/*
 *  sr_cat (addr1, len1, addr2, len2, ... , NULL, 0)
 *  Allocate a buffer and concatenate one or more char[] strings.
 *  Build an SR string (int length followed by character data).
 *  Return the address of the string.
 */
daddr
sr_cat (va_alist)				/* RTS Primitive */
va_dcl	{
    va_list ap;
    daddr buf;
    int bufsiz, n;
    char *p, *bp;

    va_start (ap);
    n = 0;					/* init total length counter */
    while (va_arg (ap, char*))			/* for each string, */
	n += va_arg (ap, int);			/* add its length */
    va_end (ap);
    bufsiz = sizeof (int) + n + 1;		/* length field + data + NUL */
    buf = sr_own_alloc (bufsiz, sr_cur_res);	/* allocate buffer */

    va_start (ap);
    * ((int *) buf) = n;			/* set string length */
    bp = (char *) ((int *) buf + 1);		/* data follows length field */
    while (p = va_arg (ap, char*)) {		/* for each input string: */
	n = va_arg (ap, int);			    /* get length */
	memcpy (bp, p, n);			    /* copy into buffer */
	bp += n;				    /* advance buffer pointer*/
    }

    va_end (ap);
    return (buf);				/* return buffer address */
}



/*
 *  Compare two strings and return <0, 0, or >0.
 */
int
sr_strcmp (laddr, llen, raddr, rlen)		/* RTS Primitive */
    char *laddr, *raddr;
    int llen, rlen;
{
    int d;
    while (rlen--)  {
	if (!llen--)
	    return -1;
	if ((d = *laddr++ - *raddr++) != 0)
	    return d;
    }
    return llen;
}



/*
 *  sr_str_result (p, pstr, len) -- store string function result.
 */
void
sr_str_result (p, pstr, len)			/* RTS Primitive */
    register char *p;
    struct	{ int slen; char data [1]; } *pstr;
    register int len;
{
    register char *d;

    d = pstr->data;
    if (p)
	while (len-- && *p)
	    *d++ = *p++;
    pstr->slen = d - pstr->data;
}



/*
 *  sr_max (nargs, v,... ) -- return maximum of n values.
 */
int
sr_max (va_alist)				/* RTS Primitive */
    va_dcl	{
    va_list ap;
    int n, r, v;

    va_start (ap);
    n = va_arg (ap, int);
    r = 0x80000000;	/* smallest signed number */
    while (n--)  {
	v = va_arg (ap, int);
	if (v > r)
	    r = v;
    }
    va_end (ap);
    return (r);
}



/*
 *  sr_min (nargs, v,... ) -- return minimum of n values.
 */
int
sr_min (va_alist)				/* RTS Primitive */
    va_dcl	{
    va_list ap;
    int n, r, v;

    va_start (ap);
    n = va_arg (ap, int);
    r = 0x7FFFFFFF;	/* largest signed number */
    while (n--)  {
	v = va_arg (ap, int);
	if (v < r)
	    r = v;
    }
    va_end (ap);
    return (r);
}



/*
 *  Implement a clone operation by making duplicates beyond
 *  the first and incrementing the store address.
 */
daddr
sr_clone (addr, len, n)				/* RTS Primitive */
    daddr addr;
    int len, n;
{
    daddr new = addr;
    while (--n > 0)
	memcpy (new += len, addr, len);
    return (new + len);
}



/*
 *  Swap two items in memory.  If len is 0, they're strings,
 *  and the maximum of the current lengths is to be used.
 */
void
sr_swap (laddr, raddr, len)			/* RTS Primitive */
    char *laddr, *raddr;
    int len;
{
    register t;

    if (len == 0)  {
	len = * (int *) laddr;
	if (* (int *) raddr > len)
	    len = * (int *) raddr;
	len += sizeof (int);
    }
    while (len-- > 0)  {
	t = *laddr;
	*laddr++ = *raddr;
	*raddr++ = t;
    }
}



/*
 *  Allocate memory for an SR new(type) call.
 */

#define ALLOC_MAGIC 314159265
static unsigned long low_alloc = ~0;
static unsigned long high_alloc = 0;

daddr
sr_new (len)					/* RTS Primitive */
    int len;
{
    daddr addr;
    unsigned long uaddr;

    if (len < 0)
	sr_abort ("negative argument to sr_new()");
    addr = (daddr) malloc ((unsigned) (len + sizeof(long)));
    DEBUG(0x10000,"sr_new(%d) -> %06X",len,addr,0);
    if (!addr)
	sr_abort ("out of memory");
    * (unsigned long *) addr = ALLOC_MAGIC;
    addr += sizeof(long);
    uaddr = (unsigned long) addr;
    if (uaddr < low_alloc)
	low_alloc = uaddr;
    if (uaddr > high_alloc)
	high_alloc = uaddr;
    return (addr);
}



/*
 *  Deallocate a block allocated by sr_new; noop if null address.
 */

void
sr_newfree (addr)				/* RTS Primitive */
    daddr addr;
{
    unsigned long uaddr;

    if (addr)  {
	uaddr = (unsigned long) addr;
	addr -= sizeof(long);
	DEBUG(0x10000,"sr_newfree(%06X)",addr,0,0);
	if (uaddr < low_alloc || uaddr > high_alloc
	|| (* (unsigned long *) addr != ALLOC_MAGIC))
	    sr_abort
	    ("bad address for free(), or block overwritten");
	* (unsigned long *) addr = 0;	/* prevent double free */
	free (addr);
    }
}



/*
 *  Return the number of command line arguments.
 */
int
sr_numargs ()					/* RTS Primitive */
{
    sr_check_stk();
    return ((sr_argc > 0) ? sr_argc - 1 : 0);
}



/*
 *  Interpret command line argument "n" as a Boolean literal.
 */
int
sr_arg_bool (n, pbool)				/* RTS Primitive */
int n;
char *pbool;
{
    char *ap;

    sr_check_stk();
    if (n < 0 || n >= sr_argc)
	return (EOF);
    ap = sr_argv [n];

    if (ap[0] == 't' && ap[1] == 'r' && ap[2] == 'u' &&
        ap[3] == 'e' && ap[4] == '\0') {
        	*pbool = TRUE;
	return (1);
    }

    if (ap[0] == 'f' && ap[1] == 'a' && ap[2] == 'l' &&
        ap[3] == 's' && ap[4] == 'e' && ap[5] == '\0') {
        	*pbool = FALSE;
	return (1);
    }

    return (0);
}



/*
 *  Interpret command line argument "n" as an integer.
 */
int
sr_arg_int (n, pint)				/* RTS Primitive */
int n;
int *pint;
{

    sr_check_stk();
    if (n < 0 || n >= sr_argc)
	return (EOF);
    return (sscanf (sr_argv[n], "%d", pint) == 1 ? 1 : 0);
}



/*
 *  Copy the "n"th command line argument to SR char array.
 */
int
sr_arg_chars (n, pstr, len)			/* RTS Primitive */
int n;
char *pstr;
int len;
{
    char *ap;
    int count = 0;

    sr_check_stk();
    if (n < 0 || n >= sr_argc)
	return (EOF);

    for (ap = sr_argv [n] ; *ap != '\0' && len > 0 ; len--, count++)
	*pstr++ = *ap++;

    return (count);
}



/*
 *  Copy the "n"th command line argument to SR string.
 */
int
sr_arg_string (n, pstr, len)			/* RTS Primitive */
int n;
struct	{ int slen; char data [1]; } *pstr;
int len;
{
    char *ap, *s;
    int count = 0;

    sr_check_stk();
    if (n < 0 || n >= sr_argc)
	return (EOF);

    s = pstr->data;
    for (ap = sr_argv [n] ; *ap != '\0' && len > 0 ; len--, count++)
	*s++ = *ap++;
    pstr->slen = count;
    return (count);
}
