/*  mips.s -- assembly code for the MIPS RISC architecture
 *
 *  A MIPS context array is laid out like this:
 *
 *	saved sp register ------------------------------|
 *	magic word for checking integrity		|
 *	unused stack space				| 
 *	saved registers, including $31 (return addr) <--- saved sp points here
 *	older stack data
 *
 *  We don't save the floating point registers; they shouldn't be needed.
 */

#define MAGIC 61407		/* an unlikely integer */

#define RSIZE 40		/* space for 10 saved registers */
	
	.sdata
	.align	2
curr_ctx:			/* saves addr of current context array */
	.word	dummy_ctx	/* initialized with dummy context */

dummy_ctx:			/* dummy initial context */
	.word	dummy_stack	/* pointer to stack */
	.word	MAGIC		/* magic word to pass the sanity check */
dummy_stack:
	.space	RSIZE		/* room to save the registers */

	.text
	.align	2



/*  sr_build_context(code,context,stksize,arg1,arg2,arg3,arg4)
 *
 *  args passed in:   $4    $5      $6     $7  16($sp) .. 24($sp)
 *
 *  code	entry point of the code to be executed in the context
 *  context	buffer for holding the context array
 *  stksize	size of this buffer
 *  arg1..arg4	four int-sized arguments to be passed to the code
 *
 *  All we need to do is set up a context that will execute the startup
 *  code, below, when activated for the first time.
 */


	.globl	sr_build_context
	.ent	sr_build_context
sr_build_context:
	.frame	$sp, 0, $31
	li	$3, MAGIC
	sw	$3, 4($5)	/* save magic word */
	addu	$2, $5, $6	/* end of buffer */
	subu	$2, RSIZE	/* $2 =  new stack pointer */
	sw	$2, 0($5)	/* save in context array */
	sw	$7, 0($2)	/* save first arg in $16 slot */
	lw	$3, 16($sp)
	sw	$3, 4($2)	/* save second arg in $17 slot */
	lw	$3, 20($sp)
	sw	$3, 8($2)	/* save third arg in $18 slot */
	lw	$3, 24($sp)
	sw	$3, 12($2)	/* save fourth arg in $19 slot */
	sw	$4, 16($2)	/* save entry point in $20 slot */
	la	$3, startup
	sw	$3, 36($2)	/* save startup address in $31 slot */
	j	$31		/* return */

startup:	
	move	$4,$16		/* load arg registers for first-time call */
	move	$5,$17
	move	$6,$18
	move	$7,$19
	la	$31,under	/* detect underflow if called code returns */
	j	$20		/* jump to entry point */
	.end	sr_build_context



/*  sr_chg_context(newctx) -- switch to the specified context  */

	.globl	sr_chg_context
	.ent	sr_chg_context
sr_chg_context:
	subu	$sp, RSIZE	/* make room on stack for saving registers */
	.frame	$sp, RSIZE, $31
	.mask	0xC0FF0000, -4
	lw	$14, curr_ctx	/* get address of context array */
	ble	$sp, $14, over	/* if stack is currently overflowing */
	li	$13, MAGIC
	lw	$12, 4($14)
	bne	$12, $13, over	/* if it overflowed before & clobbered magic */

	lw	$12, 4($4)
	bne	$12, $13, bad	/* if new context isn't intact */

	sw	$sp, 0($14)	/* save stack pointer, then registers */
	sd	$30, 32($sp)	/* includes $31, our caller's return address */
	sd	$22, 24($sp)
	sd	$20, 16($sp)
	sd	$18, 8($sp)
	sd	$16, 0($sp)

	lw	$sp, 0($4)	/* load new stack pointer, then registers */
	ld	$30, 32($sp)	/* includes $31, the resumption address */
	ld	$22, 24($sp)
	ld	$20, 16($sp)
	ld	$18, 8($sp)
	ld	$16, 0($sp)

	addu	$sp, RSIZE	/* adjust stack pointer to remove reg area */
	sw	$4,curr_ctx	/* save new context address */
	j	$31		/* return into new context */

	.end	sr_chg_context



/*  sr_check_stk() -- check that the stack is not overflowing  */

	.globl	sr_check_stk
	.ent	sr_check_stk
sr_check_stk:
	.frame	$sp, 0, $31
	lw	$14, curr_ctx	/* address of current context array */
	ble	$sp, $14, over	/* check that we haven't grown beyond it */
	j	$31		/* return */



/*  stack problem handlers  (these calls do not return)  */

over:	jal	sr_stk_overflow
under:	jal	sr_stk_underflow
bad:	jal	sr_stk_corrupted

	.end	sr_check_stk
